# Plot results of clustering and find relationship between d18Oc, d18Ow and temperature
# Project "Saiwan_data_processing"
# Author: Niels J. de Winter
# Study: "Living on the edge: Response of rudist bivalves (Hippuritida) to hot and highly seasonal climate in the low-latitude Saiwan site, Oman"

require(tidyverse)
require(ggpubr) 
source("calculate_temp_and_d18Osw_NJW.R") # load d18Oc-d18Ow conversion functions

# Load stable stable and clumped isotope data
Clumped_data <- read.csv("7_Results_clumped_clustering/Clumped_TS_intcorr_cluster_review.csv") # Clumped data
Clumped_clusters <- read.csv("7_Results_clumped_clustering/Clumped_TS_clusters_location.csv") # Import cluster results by location

# ------------------------------------------------------------------------------

# Regress temperature and d18Ow against d18Oc
# Using location-based clusters
loess_T47_d18Oc_loc <- loess(
    formula = T47 ~ d18O,
    data = Clumped_data,
    span = 1,
    control = loess.control(surface = "direct")
)

fitted_loess_T47_d18Oc_loc <- data.frame(
    d18Oc = unname(loess_T47_d18Oc_loc$x),
    T47 = loess_T47_d18Oc_loc$fitted
)

loess_d18Ow_d18Oc_loc <- loess(
    formula = d18Ow ~ d18O,
    data = Clumped_data,
    span = 1,
    control = loess.control(surface = "direct")
)
fitted_loess_d18Ow_d18Oc_loc <- data.frame(
    d18Oc = unname(loess_d18Ow_d18Oc_loc$x),
    d18Ow = loess_d18Ow_d18Oc_loc$fitted
)

# Assign seasons to the four clusters
Clumped_clusters$season <- c("Winter", " Spring/\nAutumn", "Summer", "Spring/\nAutumn")

# Plot temperature and d18Ow against d18Oc
Cluster_stats_T47_d18Oc <- ggplot(Clumped_data) +
    geom_smooth(
        aes(
            x = d18O,
            y = T47
        ),
        method = "loess",
        span = 1,
        color = "black",
        linetype = "dashed",
        linewidth = 2,
        fullrange = TRUE
    ) +
    geom_pointrange(data = Clumped_clusters,
        aes(
            x = d18Omean,
            y = Tmean,
            ymin = Tmin,
            ymax = Tmax,
            color = Tmean,
            shape = location_cluster
        ),
        linewidth = 2,
        size = 1.5
    ) +
    geom_errorbarh(data = Clumped_clusters,
        aes(
            y = Tmean,
            xmin = d18Omean - qt(0.95, N - 1) * d18Osd / sqrt(N),
            xmax = d18Omean + qt(0.95, N - 1) * d18Osd / sqrt(N),
            color = Tmean
        ),
        height = 0,
        linewidth = 2
    ) +
    scale_y_continuous("Temperature based on D47\n(OGLS23 calibration)",
        breaks = seq(15, 50, 5)
    ) +
    scale_x_continuous(expression(paste(delta^18 * "Oc ", "(\u2030 VPDB)")),
        breaks = seq(-7, -4, 0.25)
    ) +
    scale_color_distiller(palette = "RdYlBu") +
    coord_cartesian(
        xlim = c(-6.5, -4.5),
        ylim = c(15, 50)
    ) +
    theme_bw()

Cluster_stats_d18Ow_d18Oc <- ggplot(Clumped_data) +
    geom_hline(
        yintercept = -1,
        size = 2,
        linetype = "dashed",
        alpha = 0.5
    ) +
    geom_smooth(
        aes(
            x = d18O,
            y = d18Ow
        ),
        method = "loess",
        span = 1,
        color = "black",
        linetype = "dashed",
        linewidth = 2,
        fullrange = TRUE
    ) +
    geom_pointrange(data = Clumped_clusters,
        aes(
            x = d18Omean,
            y = dwmean,
            ymin = dwmin,
            ymax = dwmax,
            color = Tmean,
            shape = as.factor(location_cluster)
        ),
        linewidth = 2,
        size = 1.5
    ) +
    geom_errorbarh(data = Clumped_clusters,
        aes(
            y = dwmean,
            xmin = d18Omean - qt(0.95, N - 1) * d18Osd / sqrt(N),
            xmax = d18Omean + qt(0.95, N - 1) * d18Osd / sqrt(N),
            color = Tmean
        ),
        height = 0,
        linewidth = 2
    ) +
    scale_y_continuous(expression(paste("Reconstructed ", delta^18 * "Ow ", "(\u2030 VSMOW; Kim & O'Neil, 1997)")),
        breaks = seq(-5.5, 2, 1)
    ) +
    scale_x_continuous(expression(paste(delta^18 * "Oc ", "(\u2030 VPDB)")),
        breaks = seq(-7, -4, 0.5)
    ) +
    scale_color_distiller(palette = "RdYlBu") +
    coord_cartesian(
        xlim = c(-6.5, -4.5),
        ylim = c(-5.5, 2)
    ) +
    theme_bw()

# Plot cluster stats by assigned season
Cluster_stats_T47_season <- ggplot(Clumped_clusters) +
    geom_pointrange(data = Clumped_clusters,
        aes(
            x = season,
            y = Tmean,
            ymin = Tmin,
            ymax = Tmax,
            color = Tmean,
            shape = location_cluster
        ),
        linewidth = 2,
        size = 1.5
    ) +
    scale_y_continuous("Temperature based on D47\n(OGLS23 calibration)",
        breaks = seq(15, 50, 5)
    ) +
    scale_x_discrete("Season") +
    scale_color_distiller(palette = "RdYlBu") +
    coord_cartesian(
        ylim = c(15, 50)
    ) +
    theme_bw()

Cluster_stats_d18Ow_season <- ggplot(Clumped_clusters) +
    geom_pointrange(data = Clumped_clusters,
        aes(
            x = season,
            y = dwmean,
            ymin = dwmin,
            ymax = dwmax,
            color = Tmean,
            shape = location_cluster
        ),
        linewidth = 2,
        size = 1.5
    ) +
    scale_y_continuous(expression(paste("Reconstructed ", delta^18 * "Ow ", "(\u2030 VSMOW; Kim & O'Neil, 1997)")),
        breaks = seq(-5.5, 2, 1)
    ) +
    scale_x_discrete("Season") +
    scale_color_distiller(palette = "RdYlBu") +
    coord_cartesian(
        ylim = c(-5.5, 2)
    ) +
    theme_bw()

Cluster_T47_d18Ow_plot_combined <- ggarrange(
    Cluster_stats_T47_d18Oc,
    Cluster_stats_T47_season,
    Cluster_stats_d18Ow_d18Oc,
    Cluster_stats_d18Ow_season,
    common.legend = TRUE,
    ncol = 2,
    nrow = 2,
    widths = c(1, 0.5, 1, 0.5)
)