# Plot trace elements based on XRF
# Project "Saiwan_data_processing"
# Author: Niels J. de Winter
# Study: "Living on the edge: Response of rudist bivalves (Hippuritida) to hot and highly seasonal climate in the low-latitude Saiwan site, Oman"

require(tidyverse)
require(cowplot)

# Load data and add species names
TS <- read.csv("0_raw_data/TS_XRF.csv") %>%
mutate(
    species = "Torreites sanchezi"
)
OF <- read.csv("0_raw_data/OF_XRF.csv") %>%
mutate(
    species = "Oscillopha figari"
)
VV <- read.csv("0_raw_data/VV_XRF.csv") %>%
mutate(
    species = "Vaccinites vesiculosus"
)

# Combine data
XRF_combined <- full_join(
    full_join(
        TS,
        OF),
    VV    
)

# Select and lengthen data
XRFlong <- pivot_longer(
    select(XRF_combined, Mn, Fe, Mg, Sr, species),
    cols = c("Mn", "Fe", "Mg", "Sr"),
    values_to = "concentration",
    names_to = "element"
)

# Calculate stats and pivot to wide format
XRFstats <- XRFlong %>%
group_by(species, element) %>%
summarize(
    N = n(),
    mean = mean(concentration, na.rm = TRUE),
    median = median(concentration, na.rm = TRUE),
    SD = sd(concentration, na.rm = TRUE),
    SE = SD / sqrt(N - 1),
    CL95 = SE * qt(0.95, N - 1)
) %>%
pivot_wider(
    id_cols = "species",
    names_from = "element",
    values_from = N:CL95
)

# ------------------------------------------------------------------------------------------
# Plot Mn vs Fe
MnFe_plot <- ggplot(XRF_combined) +
# Add individual points
geom_point(
    aes(
        x = Mn,
        y = Fe,
        color = species
    ),
    alpha = 0.15,
    stroke = 0.2
) +
# Add medians and ranges
geom_errorbarh(data = XRFstats,
    aes(
        y = median_Fe,
        xmin = ifelse(median_Mn - 2 * SD_Mn < 0, 0, median_Mn - 2 * SD_Mn),
        xmax = median_Mn + 2 * SD_Mn
    ),
    height = 0,
    linewidth = 1.5
) +
geom_pointrange(data = XRFstats,
    aes(
        x = median_Mn,
        y = median_Fe,
        ymin = ifelse((mean_Fe - SD_Fe) < 0, 0, mean_Fe - SD_Fe),
        ymax = mean_Fe + SD_Fe,
        fill = species
    ),
    linewidth = 1.5,
    size = 2,
    pch = 21
) +
# modify plot aesthetics
scale_x_continuous(
    "Mn (ug/g)",
    breaks = seq(0, 600, 100)
) +
scale_y_continuous(
    "Fe (ug/g)",
    breaks = seq(0, 800, 200)
) +
theme_minimal() +
coord_cartesian(
    xlim = c(0, 400),
    ylim = c(0, 800)
)

# frequency diagram for Mn
hist_mn <- ggplot(XRF_combined,
    aes(
        x = Mn,
        y = ..ncount..,
        fill = species
    ),
) +
geom_histogram(binwidth = 5, alpha = 0.5, position = "identity") +
labs(x = NULL, y = "Relative\nfrequency", fill = "species") +
theme_minimal() +
scale_x_continuous(
    "",
    breaks = seq(0, 400, 100)
) +
scale_y_continuous(
    "Relative\nfrequency",
    breaks = seq(0, 1)
) +
coord_cartesian(
    xlim = c(0, 400)
)

# frequency diagram for Fe
hist_fe <- ggplot(XRF_combined,
    aes(
        x = Fe,
        y = ..ncount..,
        fill = species
    ),
) +
geom_histogram(binwidth = 5, alpha = 0.5, position = "identity") +
labs(x = NULL, y = "Relative\nfrequency", fill = "species") +
theme_minimal() +
scale_x_continuous(
    "",
    breaks = seq(0, 800, 200),
    limits = c(0, 800)
) +
scale_y_continuous(
    "Relative\nfrequency",
    breaks = seq(0, 1)
) +
coord_flip()

# ------------------------------------------------------------------------------------------
# Plot Mg vs Sr
MgSr_plot <- ggplot(XRF_combined) +
# Add individual points
geom_point(
    aes(
        x = Mg,
        y = Sr,
        color = species
    ),
    alpha = 0.15,
    stroke = 0.2
) +
# Add medians and ranges
geom_errorbarh(data = XRFstats,
    aes(
        y = median_Sr,
        xmin = ifelse(median_Mg - 2 * SD_Mg < 0, 0, median_Mg - 2 * SD_Mg),
        xmax = median_Mg + 2 * SD_Mg
    ),
    height = 0,
    linewidth = 1.5
) +
geom_pointrange(data = XRFstats,
    aes(
        x = median_Mg,
        y = median_Sr,
        ymin = ifelse((mean_Sr - SD_Sr) < 0, 0, mean_Sr - SD_Sr),
        ymax = mean_Sr + SD_Sr,
        fill = species
    ),
    linewidth = 1.5,
    size = 2,
    pch = 21
) +
# modify plot aesthetics
scale_x_continuous(
    "Mg (ug/g)",
    breaks = seq(0, 4000, 1000)
) +
scale_y_continuous(
    "Sr (ug/g)",
    breaks = seq(0, 2000, 500)
) +
theme_minimal() +
coord_cartesian(
    xlim = c(0, 4000),
    ylim = c(0, 2000)
)

# frequency diagram for Mg
hist_Mg <- ggplot(XRF_combined,
    aes(
        x = Mg,
        y = ..ncount..,
        fill = species
    ),
) +
geom_histogram(binwidth = 50, alpha = 0.5, position = "identity") +
labs(x = NULL, y = "Relative\nfrequency", fill = "species") +
theme_minimal() +
scale_x_continuous(
    "",
    breaks = seq(0, 4000, 1000)
) +
scale_y_continuous(
    "Relative\nfrequency",
    breaks = seq(0, 1)
) +
coord_cartesian(
    xlim = c(0, 4000)
)

# frequency diagram for Sr
hist_Sr <- ggplot(XRF_combined,
    aes(
        x = Sr,
        y = ..ncount..,
        fill = species
    ),
) +
geom_histogram(binwidth = 25, alpha = 0.5, position = "identity") +
labs(x = NULL, y = "Relative\nfrequency", fill = "species") +
theme_minimal() +
scale_x_continuous(
    "",
    breaks = seq(0, 2000, 500),
    limits = c(0, 2000)
) +
scale_y_continuous(
    "Relative\nfrequency",
    breaks = seq(0, 1)
) +
coord_flip()

# ------------------------------------------------------------------------------------------
# Align plots and merge
combined_TE_plots <- ggarrange(
    plot_grid(
        hist_mn + theme(axis.text.x = element_blank(), axis.ticks.x = element_blank(), plot.margin = margin(l = 0, r = 3), legend.position = "none"),
        NULL,
        MnFe_plot + theme(plot.margin = margin(b = 20, t = 5), legend.position = "none"),
        hist_fe + theme(axis.text.y = element_blank(), axis.ticks.y = element_blank(), legend.position = "none"),
        nrow = 2,
        rel_widths = c(1, 0.2),
        rel_heights = c(0.2, 1)
    ),
    plot_grid(
        hist_Mg + theme(axis.text.x = element_blank(), axis.ticks.x = element_blank(), plot.margin = margin(l = 3, r = 3), legend.position = "none"),
        NULL,
        MgSr_plot + theme(plot.margin = margin(b = 20, t = 5), legend.position = "none"),
        hist_Sr + theme(axis.text.y = element_blank(), axis.ticks.y = element_blank(), legend.position = "none"),
        nrow = 2,
        rel_widths = c(1, 0.2),
        rel_heights = c(0.2, 1)
    ),
    nrow = 1
)