# Combine and plot growth rate data from Daydacna
# Project "Saiwan_data_processing"
# Author: Niels J. de Winter
# Study: "Living on the edge: Response of rudist bivalves (Hippuritida) to hot and highly seasonal climate in the low-latitude Saiwan site, Oman"

require(tidyverse)
require(RColorBrewer)
require(ggpubr)

# Load chronology data
ShellChron_results <- read.csv("2_Results_dating/Age_model_results.csv") # ShellChron
B10_dated_DD_long <- read.csv("2_Results_dating/Age_model_results_DD.csv") # Daydacna

# Create common color scale for all plots
proxycolors = c(
    "d18O" = "blue",
    "d13C" = "red",
    "MgCa" = "green",
    "SrCa" = "darkblue",
    "MgLi" = "lightblue",
    "SrLi" = "lightgreen",
    "ShellChron" = "black"
)

# Reduce DD dataset to 100 micrometer resolution for easier handling without losing information on the relevant scale
B10_DD_daily <- data.frame(
    D_zero = seq(0, 131500, 100),
    MgCa = approx(x = B10_dated_DD_long$D_zero[which(B10_dated_DD_long$proxy == "MgCaBP")], y = B10_dated_DD_long$mean[which(B10_dated_DD_long$proxy == "MgCaBP")], xout = seq(0, 131500, 100))$y,
    MgLi = approx(x = B10_dated_DD_long$D_zero[which(B10_dated_DD_long$proxy == "MgLi")], y = B10_dated_DD_long$mean[which(B10_dated_DD_long$proxy == "MgLi")], xout = seq(0, 131500, 100))$y,
    SrCa = approx(x = B10_dated_DD_long$D_zero[which(B10_dated_DD_long$proxy == "SrCaBP")], y = B10_dated_DD_long$mean[which(B10_dated_DD_long$proxy == "SrCaBP")], xout = seq(0, 131500, 100))$y,
    SrLi = approx(x = B10_dated_DD_long$D_zero[which(B10_dated_DD_long$proxy == "SrLi")], y = B10_dated_DD_long$mean[which(B10_dated_DD_long$proxy == "SrLi")], xout = seq(0, 131500, 100))$y
) %>%
pivot_longer(
    cols = MgCa:SrLi,
    names_to = "proxy",
    values_to = "day"
)

# Plot growth curves comparing ShellChron with Daydacna for specimen B10
B10_Gcurves_SC_DD <- ggplot(filter(ShellChron_results, species == "Torreites sanchezi")) +
    geom_vline(xintercept = seq(-4, 11, 1) * 365) + # Add year lines
    geom_smooth( # Add smoothed growth curves per species for ShellChron
        aes(
            x = mean,
            y = D_zero / 1000,
            color = "ShellChron"
        ),
        method = "loess",
        span = 0.2,
        linewidth = 2
    ) +
    geom_point( # Add datapoints
        aes(
            x = mean,
            y = D_zero / 1000
        ),
        color = "black",
        alpha = 0.15,
        size = 0.5
    ) +
    geom_smooth(data = B10_DD_daily, # Add smoothed growth curves per species for Daydacna results
        aes(
            x = day,
            y = D_zero / 1000,
            color = proxy
        ),
        method = "loess",
        span = 0.2,
        se = FALSE,
        linewidth = 2,
        alpha = 0.5
    ) +
    scale_x_continuous("age (days)",
        limits = c(-3, 11) * 365) +
    scale_y_continuous("Height (mm)") +
    scale_color_manual(values = proxycolors) +
    ggtitle("Comparison between ShellChron and Daydacna\nresults for specimen B10 (T. sanchezi)") +
    coord_cartesian(xlim = c(0, 4000), ylim = c(0, 160)) +
    theme_bw()

# ------------------------------------------------------------------------------
# Plot growth curves for all specimens and proxies
Gcurves_TS <- ggplot() +
    geom_vline(xintercept = seq(-4, 11, 1) * 365) + # Add year lines
    geom_smooth(data = filter(ShellChron_results, specimen_name == "B10"), # Add smoothed growth curves per specimen and per proxy
        aes(
            x = mean,
            y = D_zero / 1000,
            color = proxy,
            fill = proxy
        ),
        method = "loess",
        span = 0.2,
        alpha = 0.1
    ) +
    geom_smooth(data = filter(ShellChron_results, specimen_name == "H576"), # Add smoothed growth curves per specimen and per proxy
        aes(
            x = mean,
            y = D_zero / 1000,
            color = proxy,
            fill = proxy
        ),
        method = "loess",
        span = 0.2,
        alpha = 0.1
    ) +
    geom_smooth(data = filter(ShellChron_results, specimen_name == "H585"), # Add smoothed growth curves per specimen and per proxy
        aes(
            x = mean,
            y = D_zero / 1000,
            color = proxy,
            fill = proxy
        ),
        method = "loess",
        span = 0.2,
        alpha = 0.1
    ) +
    geom_smooth(data = filter(ShellChron_results, specimen_name %in% c("H579A", "H579B", "H579C", "H579D", "H579E")), # Add smoothed growth curves per specimen and per proxy
        aes(
            x = mean,
            y = D_zero / 1000,
            color = proxy,
            fill = proxy
        ),
        method = "loess",
        span = 0.2,
        alpha = 0.1
    ) +
    geom_smooth(data = filter(ShellChron_results, species == "Torreites sanchezi"), # Combine all growth curves for this species
        aes(
            x = mean,
            y = D_zero / 1000,
        ),
        method = "loess",
        span = 0.2,
        color = "black",
        linewidth = 2
    ) +
    geom_point(data = filter(ShellChron_results, species == "Torreites sanchezi"), # Add datapoints
        aes(
            x = mean,
            y = D_zero / 1000,
            color = proxy
        ),
        alpha = 0.15,
        size = 0.5
    ) +
    scale_x_continuous("age (days)",
        limits = c(-3, 11) * 365) +
    scale_y_continuous("Height (mm)",
        limits = c(0, 180)) +
    scale_color_manual(values = proxycolors) +
    scale_fill_manual(values = proxycolors) +
    ggtitle("ShellChron results\nTorreites sanchezi") +
    coord_cartesian(xlim = c(0, 4000), ylim = c(0, 160)) +
    theme_bw()

Gcurves_VV <- ggplot(data = filter(ShellChron_results, species == "Vaccinites vesiculosus")) +
    geom_vline(xintercept = seq(-4, 11, 1) * 365) + # Add year lines
    geom_smooth( # Add smoothed growth curves per proxy
        aes(
            x = mean,
            y = D_zero / 1000,
            color = proxy,
            fill = proxy
        ),
        method = "loess",
        span = 0.2,
        alpha = 0.1
    ) +
    geom_point(data = filter(ShellChron_results, species == "Vaccinites vesiculosus"), # Add datapoints
        aes(
            x = mean,
            y = D_zero / 1000,
            color = proxy
        ),
        alpha = 0.15,
        size = 0.5
    ) +
    scale_x_continuous("age (days)",
        limits = c(-3, 11) * 365) +
    scale_y_continuous("Height (mm)",
        limits = c(0, 180)) +
    scale_color_manual(values = proxycolors) +
    scale_fill_manual(values = proxycolors) +
    ggtitle("ShellChron results\nVaccinites vesiculosus") +
    coord_cartesian(xlim = c(0, 4000), ylim = c(0, 80)) +
    theme_bw()

Gcurves_OF <- ggplot(data = filter(ShellChron_results, species == "Oscillopha figari")) +
    geom_vline(xintercept = seq(-4, 11, 1) * 365) + # Add year lines
    geom_smooth( # Add smoothed growth curves per proxy
        aes(
            x = mean,
            y = D_zero / 1000,
            color = proxy,
            fill = proxy
        ),
        method = "loess",
        span = 0.2,
        alpha = 0.1
    ) +
    geom_point(data = filter(ShellChron_results, species == "Oscillopha figari"), # Add datapoints
        aes(
            x = mean,
            y = D_zero / 1000,
            color = proxy
        ),
        alpha = 0.15,
        size = 0.5
    ) +
    scale_x_continuous("age (days)",
        limits = c(-3, 11) * 365) +
    scale_y_continuous("Height (mm)",
        limits = c(0, 180)) +
    scale_color_manual(values = proxycolors) +
    scale_fill_manual(values = proxycolors) +
    ggtitle("ShellChron results\nOscillopha figari") +
    coord_cartesian(xlim = c(0, 4000), ylim = c(0, 25)) +
    theme_bw()

# Combine growth plots with nested ggarrange calls
Chronologies_plot <- ggarrange(
    ggarrange(
        Gcurves_TS,
        ggarrange(
            Gcurves_VV,
            Gcurves_OF,
            ncol = 2,
            nrow = 1,
            legend = "none",
            labels = c("B", "C")
        ) + theme(legend.position = "none"),
        ncol = 1,
        nrow = 2,
        legend = "none",
        labels = c("A", "")
    ) + theme(legend.position = "none"),
    B10_Gcurves_SC_DD,
    ncol = 2,
    nrow = 1,
    common.legend = TRUE,
    labels = c("", "D")
)