# Figure 3: Plot all d18O and d13C profiles
# Project "Saiwan_data_processing"
# Author: Niels J. de Winter
# Study: "Living on the edge: Response of rudist bivalves (Hippuritida) to hot and highly seasonal climate in the low-latitude Saiwan site, Oman"

require(tidyverse)
require(ggpubr)

# Load data
Saiwan_SI <- read.csv("3_Combine_monthly_SI_data/3_Saiwan_SI_dated.csv")
Clumped_data <- read.csv("1_Results_Clumped_data_processing/Clumped_TS_intcorr_review2.csv") # Load clumped data
# Clumped_data <- Clumped_data[complete.cases(Clumped_data), ] # Remove rows with missing data

# Find year transitions in ShellChron outcomes
Saiwan_SI$YM_d18O <- Saiwan_SI$YM_d13C <- FALSE
Saiwan_SI$YM_d18O[which(diff(Saiwan_SI$DOY.d18O) < -150)] <- TRUE
Saiwan_SI$YM_d13C[which(diff(Saiwan_SI$DOY.d13C) < -150)] <- TRUE

# ------------------------------------------------------------------------------
# Plot isotope profiles

TS_plotlist <- list() # Create list to store plots of T. sanchezi samples (except for parrallel profiles of specimen H579)

# Loop through specimens
for(spec in c("B10", "H576", "H585", "HU_027")){
    # Isolate data for this specimen
    specdat <- Saiwan_SI %>%
        filter(
            specimen_name == spec,
            species == "Torreites sanchezi"
        )
    # Add column for end D values for colored background rectangles
    specdat$D_end <- c(specdat$D[-1], max(specdat$D) + specdat$D[2] - specdat$D[1])

    # Plot stable oxygen isotope results
    d18Oplot <- ggplot(specdat) +
    geom_pointrange(
        aes(
            x = D / 1000,
            y = d18O,
            ymin = d18O - 2 * d18O_SD,
            ymax = d18O + 2 * d18O_SD
        ),
        color = "blue",
        fatten = 0.8,
        alpha = 0.5
    ) +
    geom_line(data = specdat %>%
            group_by(D) %>%
            summarize(mean_d18O = mean(d18O)),
        aes(
            x = D / 1000,
            y = mean_d18O
        ),
        color = "blue"
    ) +
    scale_x_continuous("") +
    scale_y_continuous(expression(paste(delta^18 * "O ", "(\u2030 VPDB)")),
        breaks = seq(-7, -1, 1)
    ) +
    scale_fill_distiller(palette = "Blues") +
    ggtitle(paste(spec, "\nTorreites sanchezi")) +
    theme_classic() +
    coord_cartesian(
        ylim = c(-7, -4)
    )

    # Plot stable carbon isotope results
    d13Cplot <- ggplot(specdat) +
    geom_pointrange(
        aes(
            x = D / 1000,
            y = d13C,
            ymin = d13C - 2 * d13C_SD,
            ymax = d13C + 2 * d13C_SD
        ),
        color = "red",
        fatten = 0.8,
        alpha = 0.5
    ) +
    geom_line(data = specdat %>%
            group_by(D) %>%
            summarize(mean_d13C = mean(d13C)),
        aes(
            x = D / 1000,
            y = mean_d13C
        ),
        color = "red"
    ) +
    scale_x_continuous("Distance [mm]") +
    scale_y_continuous(expression(paste(delta^13 * "C ", "(\u2030 VPDB)")),
        breaks = seq(-7, 3, 1)
    ) +
    scale_fill_distiller(palette = "Reds") +
    ggtitle(paste(spec, "\nTorreites sanchezi")) +
    theme_classic() +
    coord_cartesian(
        ylim = c(-2, 3)
    )

    # Add age model bars for all specimens with age models
    if(spec != "HU_027"){
        d18Oplot <- d18Oplot + 
        geom_rect(
            aes(
                xmin = D / 1000,
                xmax = D_end / 1000,
                ymin = -Inf,
                ymax = Inf,
                fill = DOY.d18O
            ),
            alpha = 0.1
        )
        d13Cplot <- d13Cplot + 
        geom_rect(
            aes(
                xmin = D / 1000,
                xmax = D_end / 1000,
                ymin = -Inf,
                ymax = Inf,
                fill = DOY.d13C
            ),
            alpha = 0.2
        )
    } else if(spec == "HU_027"){ # Add clumped-associated d18O and d13C data for specimen HU_027
        Clumped_data_summary <- Clumped_data |> # Summarize clumped-associated d18O and d13C data
        group_by(sample) |>
        summarize(
            D = first(D) / 1000,
            N = n(),
            mean.d18O = mean(Final.d18O),
            sd.d18O = sd(Final.d18O),
            cl95.d18O = qt(0.95, N) * sd.d18O / sqrt(N),
            mean.d13C = mean(Final.d13C),
            sd.d13C = sd(Final.d13C),
            cl95.d13C = qt(0.95, N) * sd.d13C / sqrt(N)
        )
        # Add summarized SI data to plots
        d18Oplot <- d18Oplot + 
        geom_pointrange(data = Clumped_data_summary,
            aes(
                x = D,
                y = mean.d18O,
                ymin = mean.d18O - cl95.d18O,
                ymax = mean.d18O + cl95.d18O
            ),
        color = "black",
        size = 0.5,
        alpha = 0.5
        )
        d13Cplot <- d13Cplot + 
        geom_pointrange(data = Clumped_data_summary,
            aes(
                x = D,
                y = mean.d13C,
                ymin = mean.d13C - cl95.d13C,
                ymax = mean.d13C + cl95.d13C
            ),
        color = "black",
        size = 0.5,
        alpha = 0.5
        )
    }

    # Remove y-axis labels from all but the first plot to save space in panel
    if(spec != "B10"){
        d18Oplot <- d18Oplot + theme(axis.text.y = element_blank(), axis.title.y = element_blank())
        d13Cplot <- d13Cplot + theme(axis.text.y = element_blank(), axis.title.y = element_blank())
    }

    # Add plot to plotlist
    TS_plotlist[[spec]] <- ggarrange(
        d18Oplot + theme(legend.position = "none", plot.title = element_text(colour = "darkgreen", face = "bold")),
        d13Cplot + theme(legend.position = "none", plot.title = element_text(colour = "darkgreen", face = "bold")),
        ncol = 1,
        nrow = 2
    )
}

H579_plotlist <- list() # Create list to store plots of specimen H579

# Loop through specimens
for(spec in c("H579A", "H579B", "H579C", "H579D", "H579E")){
    # Isolate data for this specimen
    specdat <- Saiwan_SI %>%
        filter(
            specimen_name == spec,
            species == "Torreites sanchezi"
        )
    # Add column for end D values for colored background rectangles
    specdat$D_end <- c(specdat$D[-1], max(specdat$D) + specdat$D[2] - specdat$D[1])

    # Plot stable oxygen isotope results
    d18Oplot <- ggplot(specdat) +
    # Add background according to age model
    geom_rect(
        aes(
            xmin = D / 1000,
            xmax = D_end / 1000,
            ymin = -Inf,
            ymax = Inf,
            fill = DOY.d18O
        ),
        alpha = 0.1
    ) +
    geom_pointrange(
        aes(
            x = D / 1000,
            y = d18O,
            ymin = d18O - 2 * d18O_SD,
            ymax = d18O + 2 * d18O_SD
        ),
        color = "blue",
        fatten = 0.8,
        alpha = 0.5
    ) +
    geom_line(data = specdat %>%
            group_by(D) %>%
            summarize(mean_d18O = mean(d18O)),
        aes(
            x = D / 1000,
            y = mean_d18O
        ),
        color = "blue"
    ) +
    scale_x_continuous("") +
    scale_y_continuous(expression(paste(delta^18 * "O ", "(\u2030 VPDB)")),
        breaks = seq(-7, -1, 1)
    ) +
    scale_fill_distiller(palette = "Blues") +
    ggtitle(paste(spec, "\nT. sanchezi")) +
    theme_classic() +
    coord_cartesian(
        ylim = c(-7, -4)
    )

    # Plot stable carbon isotope results
    d13Cplot <- ggplot(specdat) +
    # Add background according to age model
    geom_rect(
        aes(
            xmin = D / 1000,
            xmax = D_end / 1000,
            ymin = -Inf,
            ymax = Inf,
            fill = DOY.d13C
        ),
        alpha = 0.2
    ) +
    geom_pointrange(
        aes(
            x = D / 1000,
            y = d13C,
            ymin = d13C - 2 * d13C_SD,
            ymax = d13C + 2 * d13C_SD
        ),
        color = "red",
        fatten = 0.8,
        alpha = 0.5
    ) +
    geom_line(data = specdat %>%
            group_by(D) %>%
            summarize(mean_d13C = mean(d13C)),
        aes(
            x = D / 1000,
            y = mean_d13C
        ),
        color = "red"
    ) +
    scale_x_continuous("Distance [mm]") +
    scale_y_continuous(expression(paste(delta^13 * "C ", "(\u2030 VPDB)")),
        breaks = seq(-7, 3, 1)
    ) +
    scale_fill_distiller(palette = "Reds") +
    ggtitle(paste(spec, "\nT. sanchezi")) +
    theme_classic() +
    coord_cartesian(
        ylim = c(-2, 3)
    )

    # Remove y-axis labels from all but the first plot to save space in panel
    if(spec != "H579A"){
        d18Oplot <- d18Oplot + theme(axis.text.y = element_blank(), axis.title.y = element_blank())
        d13Cplot <- d13Cplot + theme(axis.text.y = element_blank(), axis.title.y = element_blank())
    }

    # Add plot to plotlist
    H579_plotlist[[spec]] <- ggarrange(
        d18Oplot + theme(legend.position = "none", plot.title = element_text(colour = "darkgreen", face = "bold")),
        d13Cplot + theme(legend.position = "none", plot.title = element_text(colour = "darkgreen", face = "bold")),
        ncol = 1,
        nrow = 2
    )
}

# Plot V. vesiculosus data
# Isolate data for this specimen
specdat <- Saiwan_SI %>%
    filter(
        specimen_name == "B6"
    )
# Add column for end D values for colored background rectangles
specdat$D_end <- c(specdat$D[-1], max(specdat$D) + specdat$D[2] - specdat$D[1])

# Plot stable oxygen isotope results
d18Oplot <- ggplot(specdat) +
# Add background according to age model
geom_rect(
    aes(
        xmin = D / 1000,
        xmax = D_end / 1000,
        ymin = -Inf,
        ymax = Inf,
        fill = DOY.d18O
    ),
    alpha = 0.2
) +
geom_pointrange(
    aes(
        x = D / 1000,
        y = d18O,
        ymin = d18O - 2 * d18O_SD,
        ymax = d18O + 2 * d18O_SD
    ),
    color = "blue",
    fatten = 0.8,
    alpha = 0.5
) +
geom_line(data = specdat %>%
        group_by(D) %>%
        summarize(mean_d18O = mean(d18O)),
    aes(
        x = D / 1000,
        y = mean_d18O
    ),
    color = "blue"
) +
scale_x_continuous("") +
scale_y_continuous(expression(paste(delta^18 * "O ", "(\u2030 VPDB)")),
    breaks = seq(-7, -1, 1)
) +
scale_fill_distiller(palette = "Blues") +
ggtitle("B6\nVaccinites vesiculosus") +
theme_classic() +
coord_cartesian(
    ylim = c(-6, -2.5)
)

# Plot stable carbon isotope results
d13Cplot <- ggplot(specdat) +
# Add background according to age model
geom_rect(
    aes(
        xmin = D / 1000,
        xmax = D_end / 1000,
        ymin = -Inf,
        ymax = Inf,
        fill = DOY.d13C
    ),
    alpha = 0.2
) +
geom_pointrange(
    aes(
        x = D / 1000,
        y = d13C,
        ymin = d13C - 2 * d13C_SD,
        ymax = d13C + 2 * d13C_SD
    ),
    color = "red",
    fatten = 0.8,
    alpha = 0.5
) +
geom_line(data = specdat %>%
        group_by(D) %>%
        summarize(mean_d13C = mean(d13C)),
    aes(
        x = D / 1000,
        y = mean_d13C
    ),
    color = "red"
) +
scale_x_continuous("Distance [mm]") +
scale_y_continuous(expression(paste(delta^13 * "C ", "(\u2030 VPDB)")),
    breaks = seq(-7, 3, 1)
) +
scale_fill_distiller(palette = "Reds") +
ggtitle("B6\nVaccinites vesiculosus") +
theme_classic() +
coord_cartesian(
    ylim = c(-1, 2)
)

# Add plot to plotlist
VV_plot <- ggarrange(
    d18Oplot + theme(legend.position = "none", plot.title = element_text(colour = "blue", face = "bold")),
    d13Cplot + theme(legend.position = "none", plot.title = element_text(colour = "blue", face = "bold")),
    ncol = 1,
    nrow = 2
)

# Plot O. figari data
# Isolate data for this specimen
specdat <- Saiwan_SI %>%
    filter(
        specimen_name == "B11"
    )
# Add column for end D values for colored background rectangles
specdat$D_end <- c(specdat$D[-1], max(specdat$D) + specdat$D[2] - specdat$D[1])

# Plot stable oxygen isotope results
d18Oplot <- ggplot(specdat) +
# Add background according to age model
geom_rect(
    aes(
        xmin = D / 1000,
        xmax = D_end / 1000,
        ymin = -Inf,
        ymax = Inf,
        fill = DOY.d18O
    ),
    alpha = 0.2
) +
geom_pointrange(
    aes(
        x = D / 1000,
        y = d18O,
        ymin = d18O - 2 * d18O_SD,
        ymax = d18O + 2 * d18O_SD
    ),
    color = "blue",
    fatten = 0.8,
    alpha = 0.5
) +
geom_line(data = specdat %>%
        group_by(D) %>%
        summarize(mean_d18O = mean(d18O)),
    aes(
        x = D / 1000,
        y = mean_d18O
    ),
    color = "blue"
) +
scale_x_continuous("") +
scale_y_continuous(expression(paste(delta^18 * "O ", "(\u2030 VPDB)")),
    breaks = seq(-7, -1, 1)
) +
scale_fill_distiller(palette = "Blues") +
ggtitle("B11\nOscillopha figari") +
theme_classic() +
coord_cartesian(
    ylim = c(-5, -2.5)
)

# Plot stable carbon isotope results
d13Cplot <- ggplot(specdat) +
# Add background according to age model
geom_rect(
    aes(
        xmin = D / 1000,
        xmax = D_end / 1000,
        ymin = -Inf,
        ymax = Inf,
        fill = DOY.d13C
    ),
    alpha = 0.2
) +
geom_pointrange(
    aes(
        x = D / 1000,
        y = d13C,
        ymin = d13C - 2 * d13C_SD,
        ymax = d13C + 2 * d13C_SD
    ),
    color = "red",
    fatten = 0.8,
    alpha = 0.5
) +
geom_line(data = specdat %>%
        group_by(D) %>%
        summarize(mean_d13C = mean(d13C)),
    aes(
        x = D / 1000,
        y = mean_d13C
    ),
    color = "red"
) +
scale_x_continuous("Distance [mm]") +
scale_y_continuous(expression(paste(delta^13 * "C ", "(\u2030 VPDB)")),
    breaks = seq(-7, 3, 1)
) +
scale_fill_distiller(palette = "Reds") +
ggtitle("B11\nOscillopha figari") +
theme_classic() +
coord_cartesian(
    ylim = c(-1, 3)
)

# Add plot to plotlist
OF_plot <- ggarrange(
    d18Oplot + theme(legend.position = "none", plot.title = element_text(colour = "red", face = "bold")),
    d13Cplot + theme(legend.position = "none", plot.title = element_text(colour = "red", face = "bold")),
    ncol = 1,
    nrow = 2
)

# Plot isotope profiles together using nested ggarrange calls
SI_plots <- ggarrange(
    ggarrange(
        ggarrange(
            plotlist = TS_plotlist,
            ncol = 4,
            nrow = 1,
            common.legend = TRUE,
            widths = c(1, 0.8, 0.8, 0.8)
        ),
        ggarrange(
            plotlist = H579_plotlist,
            ncol = 5,
            nrow = 1,
            common.legend = TRUE,
            widths = c(1, 0.7, 0.7, 0.7, 0.7)
        ),
        ncol = 1,
        nrow = 2
    ),
    ggarrange(
        VV_plot,
        OF_plot,
        ncol = 1,
        nrow = 2
    ),
    ncol = 2,
    nrow = 1,
    widths = c(0.7, 0.3)
)