# Combine growth rate data with isotope data and find seasonality
# Project "Saiwan_data_processing"
# Author: Niels J. de Winter
# Study: "Living on the edge: Response of rudist bivalves (Hippuritida) to hot and highly seasonal climate in the low-latitude Saiwan site, Oman"

require(tidyverse)
require(zoo)
require(ggpubr)
require(RColorBrewer)

# Load SI data and change column names where needed
B10_SI <- read.csv("0_raw_data/B10_SI.csv") %>%
    mutate(
        d13C_SD = SD_d13C,
        d18O_SD = SD_d18O,
        SD_d13C = NULL,
        SD_d18O = NULL,
        species = "Torreites sanchezi",
        specimen_name = "B10"
    )
B6_SI <- read.csv("0_raw_data/B6_SI.csv") %>%
    mutate(
        species = "Vaccinites vesiculosus",
        specimen_name = "B6"
    )
B11_SI <- read.csv("0_raw_data/B11_SI.csv") %>%
    mutate(
        D = length / 1000,
        length = NULL,
        species = "Oscillopha figari",
        specimen_name = "B11"
    )
H576_SI <- read.csv("0_raw_data/H576_SI.csv") %>%
    mutate(
        species = "Torreites sanchezi",
        specimen_name = "H576"
    )
H579A_SI <- read.csv("0_raw_data/H579A_SI.csv") %>%
    mutate(
        species = "Torreites sanchezi",
        specimen_name = "H579A"
    )
H579B_SI <- read.csv("0_raw_data/H579B_SI.csv") %>%
    mutate(
        species = "Torreites sanchezi",
        specimen_name = "H579B"
    )
H579C_SI <- read.csv("0_raw_data/H579C_SI.csv") %>%
    mutate(
        species = "Torreites sanchezi",
        specimen_name = "H579C"
    )
H579D_SI <- read.csv("0_raw_data/H579D_SI.csv") %>%
    mutate(
        species = "Torreites sanchezi",
        specimen_name = "H579D"
    )
H579E_SI <- read.csv("0_raw_data/H579E_SI.csv") %>%
    mutate(
        species = "Torreites sanchezi",
        specimen_name = "H579E"
    )
H585_SI <- read.csv("0_raw_data/H585_SI.csv") %>%
    mutate(
        species = "Torreites sanchezi",
        specimen_name = "H585"
    )

HU_027_SI <- read.csv("0_raw_data/HU_027_SI.csv") %>%
    mutate(
        species = "Torreites sanchezi",
        specimen_name = "HU_027"
    )

# Load age model results and growth rate results and merge
Age_model_results <- read.csv("2_Results_dating/Age_model_results.csv") |>
    rename(
        mean.age = mean,
        CL95.age = CL95
    )

Growth_rate_results <- read.csv("2_Results_dating/Growth_rate_results.csv") |>
    rename(
        mean.GR = mean,
        CL95.GR = CL95
    )

# Merge isotope data
Saiwan_SI <- reduce(
    list(B10_SI %>%
            mutate(D = D / 1000),
        B11_SI,
        B6_SI,
        H576_SI,
        H579A_SI,
        H579B_SI,
        H579C_SI,
        H579D_SI,
        H579E_SI,
        H585_SI,
        HU_027_SI),
    function(x, y, ...) full_join(x, y)
) %>%
    mutate( # replace empty uncertainty values with 0.1
        d13C_SD = replace_na(d13C_SD, 0.1),
        d18O_SD = replace_na(d18O_SD, 0.1),
        D = D * 1000 # Convert to micrometers
    )

# Add age model and growth rate results based on d18O and d13C ShellChron
Saiwan_SI <- left_join(
    Saiwan_SI,
    select(
        filter(
            Age_model_results,
            proxy == "d18O"
        ),
        D,
        specimen_name,
        mean.age,
        CL95.age
    )
) %>%
    rename(
        mean.age.d18O = mean.age,
        CL95.age.d18O = CL95.age
    ) %>%
    left_join(
        .,
        select(
            filter(
                Age_model_results,
                proxy == "d13C"
            ),
            D,
            specimen_name,
            mean.age,
            CL95.age
        )
    ) %>%
    rename(
        mean.age.d13C = mean.age,
        CL95.age.d13C = CL95.age
    ) %>%
    left_join(
        .,
        select(
            filter(
                Growth_rate_results,
                proxy == "d13C"
            ),
            D,
            specimen_name,
            mean.GR,
            CL95.GR
        )
    ) %>%
    rename(
        mean.GR.d13C = mean.GR,
        CL95.GR.d13C = CL95.GR
    ) %>%
    left_join(
        .,
        select(
            filter(
                Growth_rate_results,
                proxy == "d18O"
            ),
            D,
            specimen_name,
            mean.GR,
            CL95.GR
        )
    ) %>%
    rename(
        mean.GR.d18O = mean.GR,
        CL95.GR.d18O = CL95.GR
    )

# ------------------------------------------------------------------------------
# Fill in NA gaps in age model and growth rate results
# First find and replace NAs at beginning of each specimen record
for(i in 1:nrow(Saiwan_SI)){ # loop through the rows in Saiwan_SI
    spec <- Saiwan_SI$specimen_name[i] # Find specimen
    if(Saiwan_SI$D[i] == min(Saiwan_SI$D[which(Saiwan_SI$specimen_name == spec)])){ # Find if sample is the first sample of the specimen
        if(is.na(Saiwan_SI$mean.age.d18O[i])){ # Find if age model result is missing
            Saiwan_SI$mean.age.d18O[i] <- 0 # If so, replace by 0
            Saiwan_SI$mean.GR.d18O[i] <- 0 # If so, replace by 0
        }
        if(is.na(Saiwan_SI$mean.age.d13C[i])){ # Find if age model result is missing
            Saiwan_SI$mean.age.d13C[i] <- 0 # If so, replace by 0
            Saiwan_SI$mean.GR.d13C[i] <- 0 # If so, replace by 0
        }
    }
}
# Now replace all remaining NA's with interpolated values
Saiwan_SI$mean.age.d18O <- na.approx(select(Saiwan_SI, D, mean.age.d18O))[, 2]
Saiwan_SI$mean.age.d13C <- na.approx(select(Saiwan_SI, D, mean.age.d13C))[, 2]
Saiwan_SI$mean.GR.d18O <- na.approx(select(Saiwan_SI, D, mean.GR.d18O))[, 2]
Saiwan_SI$mean.GR.d13C <- na.approx(select(Saiwan_SI, D, mean.GR.d13C))[, 2]

# ------------------------------------------------------------------------------
# Now find seasonal averages and uncertainties
Saiwan_SI <- Saiwan_SI %>% # Calculate day of the year according to d18O model
    mutate(
        DOY.d18O = mean.age.d18O %% 365,
        DOY.d13C = mean.age.d13C %% 365
    )

# ------------------------------------------------------------------------------
# Plot distribution of growth days using different ShellChron results

# # Create tiered color scale
# specimencolors <- c(
#     brewer.pal(8, "Greens")[1],
#     "red",
#     "blue",
#     brewer.pal(8, "Greens")[2:8]
# )
# names(specimencolors) <- unique(Saiwan_SI$specimen_name)

# DOY_hist_d18O <- ggplot(Saiwan_SI) +
#     geom_density(
#         aes(
#             DOY.d18O,
#             fill = specimen_name
#         ),
#         alpha = 0.3,
#         bw = 10
#     ) +
#     geom_density(
#         aes(
#             DOY.d18O
#         ),
#         bw = 10,
#         fill = NA,
#         color = "black",
#         size = 3,
#         alpha = 0.5
#     ) +
#     geom_point(
#         aes(
#             x = DOY.d18O
#         ),
#         y = 0,
#         shape = 3
#     ) +
#     coord_cartesian(
#         xlim = c(0, 365),
#         ylim = c(0, 0.01)
#     ) +
#     scale_x_continuous("Day of the year (d18O ShellChron)") +
#     scale_y_continuous("Frequency",
#         labels = scales::percent) +
#     scale_fill_manual(values = specimencolors) +
#     theme_bw()

# DOY_hist_d13C <- ggplot(Saiwan_SI) +
#     geom_density(
#         aes(
#             DOY.d13C,
#             fill = specimen_name
#         ),
#         alpha = 0.3,
#         bw = 10
#     ) +
#     geom_density(
#         aes(
#             DOY.d13C
#         ),
#         bw = 10,
#         fill = NA,
#         color = "black",
#         size = 3,
#         alpha = 0.5
#     ) +
#     geom_point(
#         aes(
#             x = DOY.d13C
#         ),
#         y = 0,
#         shape = 3
#     ) +
#     coord_cartesian(
#         xlim = c(0, 365),
#         ylim = c(0, 0.01)
#     ) +
#     scale_x_continuous("Day of the year (d13C ShellChron)") +
#     scale_y_continuous("Frequency",
#         labels = scales::percent) +
#     scale_fill_manual(values = specimencolors) +
#     theme_bw()

# DOY_hist_combined <- ggarrange(
#     DOY_hist_d18O,
#     DOY_hist_d13C,
#     common.legend = TRUE
# )

# # ------------------------------------------------------------------------------
# # Plot distribution of growth rates using different ShellChron results
# # Linear scale
# GR_hist_d18O_lin <- ggplot(Saiwan_SI) +
#     geom_density(
#         aes(
#             mean.GR.d18O,
#             fill = specimen_name
#         ),
#         alpha = 0.3,
#         bw = 5
#     ) +
#     geom_density(
#         aes(
#             mean.GR.d18O
#         ),
#         bw = 5,
#         fill = NA,
#         color = "black",
#         size = 3,
#         alpha = 0.5
#     ) +
#     geom_point(
#         aes(
#             x = mean.GR.d18O
#         ),
#         y = 0,
#         shape = 3
#     ) +
#     coord_cartesian(
#         xlim = c(0, 150),
#         ylim = c(0, 0.04)
#     ) +
#     scale_x_continuous("Growth rate (d18O ShellChron; um/d)") +
#     scale_y_continuous("Frequency",
#         labels = scales::percent) +
#     scale_fill_manual(values = specimencolors) +
#     theme_bw()

# GR_hist_d13C_lin <- ggplot(Saiwan_SI) +
#     geom_density(
#         aes(
#             mean.GR.d13C,
#             fill = specimen_name
#         ),
#         alpha = 0.3,
#         bw = 5
#     ) +
#     geom_density(
#         aes(
#             mean.GR.d13C
#         ),
#         bw = 5,
#         fill = NA,
#         color = "black",
#         size = 3,
#         alpha = 0.5
#     ) +
#     geom_point(
#         aes(
#             x = mean.GR.d13C
#         ),
#         y = 0,
#         shape = 3
#     ) +
#     coord_cartesian(
#         xlim = c(0, 150),
#         ylim = c(0, 0.04)
#     ) +
#     scale_x_continuous("Growth rate (d13C ShellChron; um/d)") +
#     scale_y_continuous("Frequency",
#         labels = scales::percent) +
#     scale_fill_manual(values = specimencolors) +
#     theme_bw()

# GR_hist_combined_lin <- ggarrange(
#     GR_hist_d18O_lin,
#     GR_hist_d13C_lin,
#     common.legend = TRUE
# )

# # Log scale
# GR_hist_d18O_log <- ggplot(Saiwan_SI) +
#     geom_density(
#         aes(
#             mean.GR.d18O,
#             fill = specimen_name
#         ),
#         alpha = 0.3,
#         bw = 0.2
#     ) +
#     geom_density(
#         aes(
#             mean.GR.d18O
#         ),
#         bw = 0.2,
#         fill = NA,
#         color = "black",
#         size = 3,
#         alpha = 0.5
#     ) +
#     geom_point(
#         aes(
#             x = mean.GR.d18O
#         ),
#         y = 0,
#         shape = 3
#     ) +
#     coord_cartesian(
#         xlim = c(1e-4, 1e3),
#         ylim = c(0, 1)
#     ) +
#     scale_x_log10("Growth rate (d18O ShellChron; um/d)") +
#     scale_y_continuous("Frequency",
#         labels = scales::percent) +
#     scale_fill_manual(values = specimencolors) +
#     theme_bw()

# GR_hist_d13C_log <- ggplot(Saiwan_SI) +
#     geom_density(
#         aes(
#             mean.GR.d13C,
#             fill = specimen_name
#         ),
#         alpha = 0.3,
#         bw = 0.2
#     ) +
#     geom_density(
#         aes(
#             mean.GR.d13C
#         ),
#         bw = 0.2,
#         fill = NA,
#         color = "black",
#         size = 3,
#         alpha = 0.5
#     ) +
#     geom_point(
#         aes(
#             x = mean.GR.d13C
#         ),
#         y = 0,
#         shape = 3
#     ) +
#     coord_cartesian(
#         xlim = c(1e-4, 1e3),
#         ylim = c(0, 1)
#     ) +
#     scale_x_log10("Growth rate (d13C ShellChron; um/d)") +
#     scale_y_continuous("Frequency",
#         labels = scales::percent) +
#     scale_fill_manual(values = specimencolors) +
#     theme_bw()

# GR_hist_combined_log <- ggarrange(
#     GR_hist_d18O_log,
#     GR_hist_d13C_log,
#     common.legend = TRUE
# )

# # ------------------------------------------------------------------------------
# # Plot seasonal d18O and d13C spread using different ShellChron results
# Seasonal_d18O_d18O <- ggplot(Saiwan_SI) +
#     geom_pointrange(
#         aes(
#             x = DOY.d18O,
#             y = d18O,
#             ymin = d18O - 2 * d18O_SD,
#             ymax = d18O + 2 * d18O_SD,
#             color = specimen_name
#         ),
#         fatten = 1,
#         alpha = 0.3
#     ) +
#     geom_errorbarh(
#         aes(
#             y = d18O,
#             xmin = DOY.d18O - CL95.age.d18O,
#             xmax = DOY.d18O + CL95.age.d18O,
#             color = specimen_name
#         ),
#         alpha = 0.3
#     ) +
#     geom_smooth(
#         aes(
#             x = DOY.d18O,
#             y = d18O,
#             fill = specimen_name,
#             color = specimen_name
#         ),
#         se = TRUE
#     ) +
#     geom_smooth(
#         aes(
#             x = DOY.d18O,
#             y = d18O
#         ),
#         se = TRUE,
#         fill = "black",
#         color = "black"
#     ) +
#     coord_cartesian(
#         xlim = c(0, 365),
#         ylim = c(-7, -1)
#     ) +
#     scale_x_continuous("Day of the year (d18O ShellChron)") +
#     scale_y_continuous("d18O (VSMOW)") +
#     scale_color_manual(values = specimencolors) +
#     scale_fill_manual(values = specimencolors) +
#     theme_bw()

# Seasonal_d18O_d13C <- ggplot(Saiwan_SI) +
#     geom_pointrange(
#         aes(
#             x = DOY.d13C,
#             y = d18O,
#             ymin = d18O - 2 * d18O_SD,
#             ymax = d18O + 2 * d18O_SD,
#             color = specimen_name
#         ),
#         fatten = 1,
#         alpha = 0.3
#     ) +
#     geom_errorbarh(
#         aes(
#             y = d18O,
#             xmin = DOY.d13C - CL95.age.d13C,
#             xmax = DOY.d13C + CL95.age.d13C,
#             color = specimen_name
#         ),
#         alpha = 0.3
#     ) +
#     geom_smooth(
#         aes(
#             x = DOY.d13C,
#             y = d18O,
#             fill = specimen_name,
#             color = specimen_name
#         ),
#         se = TRUE
#     ) +
#     geom_smooth(
#         aes(
#             x = DOY.d13C,
#             y = d18O
#         ),
#         se = TRUE,
#         fill = "black",
#         color = "black"
#     ) +
#     coord_cartesian(
#         xlim = c(0, 365),
#         ylim = c(-7, -1)
#     ) +
#     scale_x_continuous("Day of the year (d13C ShellChron)") +
#     scale_y_continuous("d18O (VSMOW)") +
#     scale_color_manual(values = specimencolors) +
#     scale_fill_manual(values = specimencolors) +
#     theme_bw()

# Seasonal_d13C_d18O <- ggplot(Saiwan_SI) +
#     geom_pointrange(
#         aes(
#             x = DOY.d18O,
#             y = d13C,
#             ymin = d13C - 2 * d13C_SD,
#             ymax = d13C + 2 * d13C_SD,
#             color = specimen_name
#         ),
#         fatten = 1,
#         alpha = 0.3
#     ) +
#     geom_errorbarh(
#         aes(
#             y = d13C,
#             xmin = DOY.d18O - CL95.age.d18O,
#             xmax = DOY.d18O + CL95.age.d18O,
#             color = specimen_name
#         ),
#         alpha = 0.3
#     ) +
#     geom_smooth(
#         aes(
#             x = DOY.d18O,
#             y = d13C,
#             fill = specimen_name,
#             color = specimen_name
#         ),
#         se = TRUE
#     ) +
#     geom_smooth(
#         aes(
#             x = DOY.d18O,
#             y = d13C
#         ),
#         se = TRUE,
#         fill = "black",
#         color = "black"
#     ) +
#     coord_cartesian(
#         xlim = c(0, 365),
#         ylim = c(-2.5, 3)
#     ) +
#     scale_x_continuous("Day of the year (d18O ShellChron)") +
#     scale_y_continuous("d13C (VSMOW)") +
#     scale_color_manual(values = specimencolors) +
#     scale_fill_manual(values = specimencolors) +
#     theme_bw()

# Seasonal_d13C_d13C <- ggplot(Saiwan_SI) +
#     geom_pointrange(
#         aes(
#             x = DOY.d13C,
#             y = d13C,
#             ymin = d13C - 2 * d13C_SD,
#             ymax = d13C + 2 * d13C_SD,
#             color = specimen_name
#         ),
#         fatten = 1,
#         alpha = 0.3
#     ) +
#     geom_errorbarh(
#         aes(
#             y = d13C,
#             xmin = DOY.d13C - CL95.age.d13C,
#             xmax = DOY.d13C + CL95.age.d13C,
#             color = specimen_name
#         ),
#         alpha = 0.3
#     ) +
#     geom_smooth(
#         aes(
#             x = DOY.d13C,
#             y = d13C,
#             fill = specimen_name,
#             color = specimen_name
#         ),
#         se = TRUE
#     ) +
#     geom_smooth(
#         aes(
#             x = DOY.d13C,
#             y = d13C
#         ),
#         se = TRUE,
#         fill = "black",
#         color = "black"
#     ) +
#     coord_cartesian(
#         xlim = c(0, 365),
#         ylim = c(-2.5, 3)
#     ) +
#     scale_x_continuous("Day of the year (d13C ShellChron)") +
#     scale_y_continuous("d13C (VSMOW)") +
#     scale_color_manual(values = specimencolors) +
#     scale_fill_manual(values = specimencolors) +
#     theme_bw()

# Seasonal_SI_combined <- ggarrange(
#     Seasonal_d18O_d18O,
#     Seasonal_d18O_d13C,
#     Seasonal_d13C_d18O,
#     Seasonal_d13C_d13C,
#     common.legend = TRUE
# )

# # ------------------------------------------------------------------------------
# # Plot seasonal growth rate spread using different ShellChron results
# Seasonal_GR_d18O <- ggplot(Saiwan_SI) +
#     geom_pointrange(
#         aes(
#             x = DOY.d18O,
#             y = mean.GR.d18O,
#             ymin = mean.GR.d18O - 2 * CL95.GR.d18O,
#             ymax = mean.GR.d18O + 2 * CL95.GR.d18O,
#             color = specimen_name
#         ),
#         fatten = 1,
#         alpha = 0.3
#     ) +
#     geom_errorbarh(
#         aes(
#             y = mean.GR.d18O,
#             xmin = DOY.d18O - CL95.age.d18O,
#             xmax = DOY.d18O + CL95.age.d18O,
#             color = specimen_name
#         ),
#         alpha = 0.3
#     ) +
#     geom_smooth(
#         aes(
#             x = DOY.d18O,
#             y = mean.GR.d18O,
#             fill = specimen_name,
#             color = specimen_name
#         ),
#         se = TRUE
#     ) +
#     geom_smooth(
#         aes(
#             x = DOY.d18O,
#             y = mean.GR.d18O
#         ),
#         se = TRUE,
#         fill = "black",
#         color = "black"
#     ) +
#     coord_cartesian(
#         xlim = c(0, 365),
#         ylim = c(0, 150)
#     ) +
#     scale_x_continuous("Day of the year (d18O ShellChron)") +
#     scale_y_continuous("Growth rate (um/d)") +
#     scale_color_manual(values = specimencolors) +
#     scale_fill_manual(values = specimencolors) +
#     theme_bw()

# Seasonal_GR_d13C <- ggplot(Saiwan_SI) +
#     geom_pointrange(
#         aes(
#             x = DOY.d13C,
#             y = mean.GR.d13C,
#             ymin = mean.GR.d13C - 2 * CL95.GR.d13C,
#             ymax = mean.GR.d13C + 2 * CL95.GR.d13C,
#             color = specimen_name
#         ),
#         fatten = 1,
#         alpha = 0.3
#     ) +
#     geom_errorbarh(
#         aes(
#             y = mean.GR.d13C,
#             xmin = DOY.d13C - CL95.age.d13C,
#             xmax = DOY.d13C + CL95.age.d13C,
#             color = specimen_name
#         ),
#         alpha = 0.3
#     ) +
#     geom_smooth(
#         aes(
#             x = DOY.d13C,
#             y = mean.GR.d13C,
#             fill = specimen_name,
#             color = specimen_name
#         ),
#         se = TRUE
#     ) +
#     geom_smooth(
#         aes(
#             x = DOY.d13C,
#             y = mean.GR.d13C
#         ),
#         se = TRUE,
#         fill = "black",
#         color = "black"
#     ) +
#     coord_cartesian(
#         xlim = c(0, 365),
#         ylim = c(0, 150)
#     ) +
#     scale_x_continuous("Day of the year (d13C ShellChron)") +
#     scale_y_continuous("Growth rate (um/d)") +
#     scale_color_manual(values = specimencolors) +
#     scale_fill_manual(values = specimencolors) +
#     theme_bw()

# Seasonal_GR_combined <- ggarrange(
#     Seasonal_GR_d18O,
#     Seasonal_GR_d13C,
#     common.legend = TRUE
# )

# ------------------------------------------------------------------------------
# Rotate seasons to find monthly d18O extremes
# ------------------------------------------------------------------------------

# Create data frame to store monthly data
Monthly_data <- data.frame(
    month = NA,
    N = NA,
    d18Omean = NA,
    d18Osd = NA,
    d13Cmean = NA,
    d13Csd = NA,
    mean.GR.d18Omean = NA,
    mean.GR.d18Osd = NA,
    mean.GR.d13Cmean = NA,
    mean.GR.d13Csd = NA,
    specimen = NA,
    age_model = NA,
    proxy = NA
)

# ------------------------------------------------------------------------------
# Loop through specimens
for(spec in unique(Saiwan_SI$specimen_name)){
    # Isolate data for this specimen
    specdat <- Saiwan_SI %>%
        filter(specimen_name == spec)

    # Use both age models
    for(agemod in c("DOY.d18O", "DOY.d13C")){
        if(!all(is.na(select(specdat, any_of(agemod))))){ # Catch instances where no age model is present
            # Loop through proxies and growth rate
            for(prox in c("d18O", "d13C", "mean.GR.d18O", "mean.GR.d13C")){
                print(paste("specimen =", spec, "age model =", agemod, "proxy =", prox))
                # Keep track of mean values and sample sizes for each combination of months
                months_try <- data.frame(
                    specimen = NA,
                    age_model = NA,
                    proxy = NA,
                    month = NA,
                    increment = NA,
                    N = NA,
                    value = NA
                )

                # Loop through roughly one twelfth of a year, day by day
                for(i in 1:31){
                    specdat <- specdat %>%
                        mutate(month = ceiling(((eval(as.symbol(agemod)) + i) %% 365) / (365 / 12)))

                    monthly_summary <- specdat %>%
                        group_by(month) %>%
                        summarize(
                            N = n(),
                            value = mean(eval(as.symbol(prox)))
                        )

                    # Report NA's in case one or more months is not represented in the data
                    if(nrow(monthly_summary) < 12){
                        for(j in 1:12){
                            if(!(j %in% monthly_summary$month)){ # Find missing months
                                monthly_summary <- rbind(monthly_summary,
                                    c(j, 0, NA) # Add NA rows for missing seasons
                                )
                            }
                        }
                    }

                    # Add means to matrix
                    monthly_summary <- monthly_summary %>%
                        mutate(
                            specimen = spec,
                            age_model = agemod,
                            proxy = prox,
                            increment = i
                        )
                    invisible(months_try <- full_join( # attempt to hide all the output of join functions
                            months_try,
                            monthly_summary
                        )
                    )
                    # Remove season classification from data before moving on to the next step
                    specdat$month <- NULL
                }

                # Find optimal seasonal definition and assign to data and calculate seasonal statistics
                months_try2 <- months_try %>%
                    filter(complete.cases(.)) %>%
                    group_by(specimen, age_model, proxy, increment) %>%
                    summarize(
                        lowest_month = min(value, na.rm = TRUE),
                        highest.month = max(value, na.rm = TRUE),
                        monthly_range = highest.month - lowest_month
                    )

                besti <- which.max(months_try2$monthly_range)
                specdat_summary <- specdat %>%
                    mutate(month = ceiling(((eval(as.symbol(agemod)) + besti) %% 365) / (365 / 12))) %>%
                    group_by(month) %>%
                    summarize(
                        N = n(),
                        d18Omean = mean(d18O, na.rm = TRUE),
                        d18Osd = sd(d18O, na.rm = TRUE),
                        d13Cmean = mean(d13C, na.rm = TRUE),
                        d13Csd = sd(d13C, na.rm = TRUE),
                        mean.GR.d18Omean = mean(mean.GR.d18O, na.rm = TRUE),
                        mean.GR.d18Osd = sd(mean.GR.d18O, na.rm = TRUE),
                        mean.GR.d13Cmean = mean(mean.GR.d13C, na.rm = TRUE),
                        mean.GR.d13Csd = sd(mean.GR.d13C, na.rm = TRUE)
                    ) %>%
                    mutate(
                        specimen = spec,
                        age_model = agemod,
                        proxy = prox
                    )
                
                # Add summary to Monthly data
                Monthly_data <- full_join(
                    Monthly_data,
                    specdat_summary
                )
            }
        }
    }
}

# Export data
write.csv(Monthly_data, "3_Combine_monthly_SI_data/3_Monthly_SI_summary.csv", row.names = FALSE)
write.csv(Saiwan_SI, "3_Combine_monthly_SI_data/3_Saiwan_SI_dated.csv", row.names = FALSE)

# # ------------------------------------------------------------------------------
# # Plot isotope profiles

# TS_plotlist <- list() # Create list to store plots of T. sanchezi samples

# # Loop through specimens
# for(spec in unique(Saiwan_SI$specimen_name[which(Saiwan_SI$species == "Torreites sanchezi")])){
#     # Isolate data for this specimen
#     specdat <- Saiwan_SI %>%
#         filter(
#             specimen_name == spec,
#             species == "Torreites sanchezi"
#         )

#     # Plot stable oxygen isotope results
#     d18Oplot <- ggplot(specdat) +
#     geom_pointrange(
#         aes(
#             x = D / 1000,
#             y = d18O,
#             ymin = d18O - 2 * d18O_SD,
#             ymax = d18O + 2 * d18O_SD
#         ),
#         color = "blue",
#         fatten = 0.8,
#         alpha = 0.5
#     ) +
#     geom_line(data = specdat %>%
#             group_by(D) %>%
#             summarize(mean_d18O = mean(d18O)),
#         aes(
#             x = D / 1000,
#             y = mean_d18O
#         ),
#         color = "blue"
#     ) +
#     scale_x_continuous("") +
#     scale_y_continuous(expression(paste(delta^18 * "O ", "(\u2030 VPDB)")),
#         breaks = seq(-7, -1, 1)
#     ) +
#     ggtitle(paste(spec, first(specdat$species))) +
#     theme_classic() +
#     coord_cartesian(
#         ylim = c(-7, -4)
#     )

#     # Plot stable carbon isotope results
#     d13Cplot <- ggplot(specdat) +
#     geom_pointrange(
#         aes(
#             x = D / 1000,
#             y = d13C,
#             ymin = d13C - 2 * d13C_SD,
#             ymax = d13C + 2 * d13C_SD
#         ),
#         color = "red",
#         fatten = 0.8,
#         alpha = 0.5
#     ) +
#     geom_line(data = specdat %>%
#             group_by(D) %>%
#             summarize(mean_d13C = mean(d13C)),
#         aes(
#             x = D / 1000,
#             y = mean_d13C
#         ),
#         color = "red"
#     ) +
#     scale_x_continuous("Distance [mm]") +
#     scale_y_continuous(expression(paste(delta^13 * "C ", "(\u2030 VPDB)")),
#         breaks = seq(-7, 2, 1)
#     ) +
#     ggtitle(paste(spec, first(specdat$species))) +
#     theme_classic() +
#     coord_cartesian(
#         ylim = c(-2, 2)
#     )

#     # Add plot to plotlist
#     TS_plotlist[[spec]] <- ggarrange(
#         d18Oplot,
#         d13Cplot,
#         ncol = 1,
#         nrow = 2
#     )
# }

# # Plot V. vesiculosus data
# # Isolate data for this specimen
# specdat <- Saiwan_SI %>%
#     filter(
#         specimen_name == "B6"
#     )

# # Plot stable oxygen isotope results
# d18Oplot <- ggplot(specdat) +
# geom_pointrange(
#     aes(
#         x = D / 1000,
#         y = d18O,
#         ymin = d18O - 2 * d18O_SD,
#         ymax = d18O + 2 * d18O_SD
#     ),
#     color = "blue",
#     fatten = 0.8,
#     alpha = 0.5
# ) +
# geom_line(data = specdat %>%
#         group_by(D) %>%
#         summarize(mean_d18O = mean(d18O)),
#     aes(
#         x = D / 1000,
#         y = mean_d18O
#     ),
#     color = "blue"
# ) +
# scale_x_continuous("") +
# scale_y_continuous(expression(paste(delta^18 * "O ", "(\u2030 VPDB)")),
#     breaks = seq(-7, -1, 1)
# ) +
# ggtitle(paste(spec, first(specdat$species))) +
# theme_classic() +
# coord_cartesian(
#     ylim = c(-6, -2.5)
# )

# # Plot stable carbon isotope results
# d13Cplot <- ggplot(specdat) +
# geom_pointrange(
#     aes(
#         x = D / 1000,
#         y = d13C,
#         ymin = d13C - 2 * d13C_SD,
#         ymax = d13C + 2 * d13C_SD
#     ),
#     color = "red",
#     fatten = 0.8,
#     alpha = 0.5
# ) +
# geom_line(data = specdat %>%
#         group_by(D) %>%
#         summarize(mean_d13C = mean(d13C)),
#     aes(
#         x = D / 1000,
#         y = mean_d13C
#     ),
#     color = "red"
# ) +
# scale_x_continuous("Distance [mm]") +
# scale_y_continuous(expression(paste(delta^13 * "C ", "(\u2030 VPDB)")),
#     breaks = seq(-7, 2, 1)
# ) +
# ggtitle("B6 Vaccinites vesiculosus") +
# theme_classic() +
# coord_cartesian(
#     ylim = c(-1, 2)
# )

# # Add plot to plotlist
# VV_plot <- ggarrange(
#     d18Oplot,
#     d13Cplot,
#     ncol = 1,
#     nrow = 2
# )

# # Plot O. figari data
# # Isolate data for this specimen
# specdat <- Saiwan_SI %>%
#     filter(
#         specimen_name == "B11"
#     )

# # Plot stable oxygen isotope results
# d18Oplot <- ggplot(specdat) +
# geom_pointrange(
#     aes(
#         x = D / 1000,
#         y = d18O,
#         ymin = d18O - 2 * d18O_SD,
#         ymax = d18O + 2 * d18O_SD
#     ),
#     color = "blue",
#     fatten = 0.8,
#     alpha = 0.5
# ) +
# geom_line(data = specdat %>%
#         group_by(D) %>%
#         summarize(mean_d18O = mean(d18O)),
#     aes(
#         x = D / 1000,
#         y = mean_d18O
#     ),
#     color = "blue"
# ) +
# scale_x_continuous("") +
# scale_y_continuous(expression(paste(delta^18 * "O ", "(\u2030 VPDB)")),
#     breaks = seq(-7, -1, 1)
# ) +
# ggtitle(paste(spec, first(specdat$species))) +
# theme_classic() +
# coord_cartesian(
#     ylim = c(-5, -2.5)
# )

# # Plot stable carbon isotope results
# d13Cplot <- ggplot(specdat) +
# geom_pointrange(
#     aes(
#         x = D / 1000,
#         y = d13C,
#         ymin = d13C - 2 * d13C_SD,
#         ymax = d13C + 2 * d13C_SD
#     ),
#     color = "red",
#     fatten = 0.8,
#     alpha = 0.5
# ) +
# geom_line(data = specdat %>%
#         group_by(D) %>%
#         summarize(mean_d13C = mean(d13C)),
#     aes(
#         x = D / 1000,
#         y = mean_d13C
#     ),
#     color = "red"
# ) +
# scale_x_continuous("Distance [mm]") +
# scale_y_continuous(expression(paste(delta^13 * "C ", "(\u2030 VPDB)")),
#     breaks = seq(-7, 2, 1)
# ) +
# ggtitle("B11 Oscillopha figari") +
# theme_classic() +
# coord_cartesian(
#     ylim = c(-1, 3)
# )

# # Add plot to plotlist
# OF_plot <- ggarrange(
#     d18Oplot,
#     d13Cplot,
#     ncol = 1,
#     nrow = 2
# )

# # Plot isotope profiles together
# SI_plots <- ggarrange(
#     ggarrange(
#         plotlist = TS_plotlist,
#         ncol = 4,
#         nrow = 2,
#         common.legend = TRUE
#     ),
#     ggarrange(
#         VV_plot,
#         OF_plot,
#         ncol = 1,
#         nrow = 2
#     ),
#     ncol = 2,
#     nrow = 1,
#     widths = c(0.8, 0.2)
# )