# Combine and plot growth rate data from Saiwan fossils
# Project "chemical_data"

require(tidyverse)
require(RColorBrewer)

# Load ShellChron ages and equalize mean day values without changeing relative ages
B10_18O_ShellChronage <- read.csv("2_Results_dating/B10_18O_ShellChron/Age_model_results.csv")
B10_13C_ShellChronage <- read.csv("2_Results_dating/B10_13C_ShellChron/Age_model_results.csv")
B10_13C_ShellChronage$mean.day <- B10_13C_ShellChronage$mean.day - mean(B10_13C_ShellChronage$mean.day) + mean(B10_18O_ShellChronage$mean.day)
B10_MgCa_ShellChronage <- read.csv("2_Results_dating/B10_MgCa_ShellChron/Age_model_results.csv")
B10_MgCa_ShellChronage$mean.day <- B10_MgCa_ShellChronage$mean.day - mean(B10_MgCa_ShellChronage$mean.day) + mean(B10_18O_ShellChronage$mean.day)
B10_SrCa_ShellChronage <- read.csv("2_Results_dating/B10_SrCa_ShellChron/Age_model_results.csv")
B10_SrCa_ShellChronage$mean.day <- B10_SrCa_ShellChronage$mean.day - mean(B10_SrCa_ShellChronage$mean.day) + mean(B10_18O_ShellChronage$mean.day)

B6_18O_ShellChronage <- read.csv("2_Results_dating/B6_18O_ShellChron/Age_model_results.csv")
B6_13C_ShellChronage <- read.csv("2_Results_dating/B6_13C_ShellChron/Age_model_results.csv")

B11_18O_ShellChronage <- read.csv("2_Results_dating/B11_18O_ShellChron/Age_model_results.csv")
B11_18O_ShellChronage$D <- B11_18O_ShellChronage$D / 1000 # Revert depth value to millimeters
B11_13C_ShellChronage <- read.csv("2_Results_dating/B11_13C_ShellChron/Age_model_results.csv")

H576_18O_ShellChronage <- read.csv("2_Results_dating/H576_18O_ShellChron/Age_model_results.csv")
H576_13C_ShellChronage <- read.csv("2_Results_dating/H576_13C_ShellChron/Age_model_results.csv")
H576_13C_ShellChronage$mean.day <- H576_13C_ShellChronage$mean.day - mean(H576_13C_ShellChronage$mean.day) + mean(H576_18O_ShellChronage$mean.day)

H579A_18O_ShellChronage <- read.csv("2_Results_dating/H579A_18O_ShellChron/Age_model_results.csv")
H579A_13C_ShellChronage <- read.csv("2_Results_dating/H579A_13C_ShellChron/Age_model_results.csv")
H579A_13C_ShellChronage$mean.day <- H579A_13C_ShellChronage$mean.day - mean(H579A_13C_ShellChronage$mean.day) + mean(H579A_18O_ShellChronage$mean.day)

H579B_18O_ShellChronage <- read.csv("2_Results_dating/H579B_18O_ShellChron/Age_model_results.csv")
H579B_13C_ShellChronage <- read.csv("2_Results_dating/H579B_13C_ShellChron/Age_model_results.csv")
H579B_13C_ShellChronage$mean.day <- H579B_13C_ShellChronage$mean.day - mean(H579B_13C_ShellChronage$mean.day) + mean(H579B_18O_ShellChronage$mean.day)

H579C_18O_ShellChronage <- read.csv("2_Results_dating/H579C_18O_ShellChron/Age_model_results.csv")
H579C_13C_ShellChronage <- read.csv("2_Results_dating/H579C_13C_ShellChron/Age_model_results.csv")
H579C_13C_ShellChronage$mean.day <- H579C_13C_ShellChronage$mean.day - mean(H579C_13C_ShellChronage$mean.day) + mean(H579C_18O_ShellChronage$mean.day)

H579D_18O_ShellChronage <- read.csv("2_Results_dating/H579D_18O_ShellChron/Age_model_results.csv")
H579D_13C_ShellChronage <- read.csv("2_Results_dating/H579D_13C_ShellChron/Age_model_results.csv")
H579D_13C_ShellChronage$mean.day <- H579D_13C_ShellChronage$mean.day - mean(H579D_13C_ShellChronage$mean.day) + mean(H579D_18O_ShellChronage$mean.day)

H579E_18O_ShellChronage <- read.csv("2_Results_dating/H579E_18O_ShellChron/Age_model_results.csv")
H579E_13C_ShellChronage <- read.csv("2_Results_dating/H579E_13C_ShellChron/Age_model_results.csv")
H579E_13C_ShellChronage$mean.day <- H579E_13C_ShellChronage$mean.day - mean(H579E_13C_ShellChronage$mean.day) + mean(H579E_18O_ShellChronage$mean.day)

H585_18O_ShellChronage <- read.csv("2_Results_dating/H585_18O_ShellChron/Age_model_results.csv")
H585_13C_ShellChronage <- read.csv("2_Results_dating/H585_13C_ShellChron/Age_model_results.csv")
H585_13C_ShellChronage$mean.day <- H585_13C_ShellChronage$mean.day - mean(H585_13C_ShellChronage$mean.day) + mean(H585_18O_ShellChronage$mean.day)

# Combine dating results per species and pivot to longer format
# B10
B10_dated <- full_join( # Add d18O ShellChron result
    select(
        B10_13C_ShellChronage,
        mean.day,
        CL95.day,
        D
    ), select(
        B10_18O_ShellChronage,
        mean.day,
        CL95.day,
        D
    ),
    by = "D",
    suffix = c(
        ".SG.d18O",
        ".SG.d13C"
        )
) %>%
full_join( # Add MgCa ShellChron result
    ., select(
    B10_MgCa_ShellChronage,
    mean.day,
    CL95.day,
    D
),
by = "D"
) %>%
full_join( # Add SrCa ShellChron result
    ., select(
    B10_SrCa_ShellChronage,
    mean.day,
    CL95.day,
    D
),
by = "D",
suffix = c(
    ".SG.MgCa",
    ".SG.SrCa"
    )
) %>%
mutate(
    specimen_name = "B10",
    species = "Torreites sanchezi"
)

# Pivot to long format
B10_dated_long <- pivot_longer(
    B10_dated,
    !c(D, specimen_name, species),
    names_to = "variable",
    values_to = "day"
)

# Separate day values from uncertainties
B10_dated_long <- separate_wider_delim(
    B10_dated_long,
    variable,
    delim = ".",
    names = c("statistic",
        "unit",
        "method",
        "proxy"
    )
)

# Pivot to wider format by statistic
B10_dated_long <- pivot_wider(B10_dated_long,
    names_from = statistic,
    values_from = day
)

# ------------------------------------------------------------------------------
# B6
B6_dated <- full_join( # Add d18O ShellChron result
    select(
        B6_13C_ShellChronage,
        mean.day,
        CL95.day,
        D
    ), select(
        B6_18O_ShellChronage,
        mean.day,
        CL95.day,
        D
    ),
    by = "D",
    suffix = c(
        ".SG.d18O",
        ".SG.d13C"
        )
) %>%
mutate(
    specimen_name = "B6",
    species = "Vaccinites vesiculosus"
)

# Pivot to long format
B6_dated_long <- pivot_longer(
    B6_dated,
    !c(D, specimen_name, species),
    names_to = "variable",
    values_to = "day"
)

# Separate day values from uncertainties
B6_dated_long <- separate_wider_delim(
    B6_dated_long,
    variable,
    delim = ".",
    names = c("statistic",
        "unit",
        "method",
        "proxy"
    )
)

# Pivot to wider format by statistic
B6_dated_long <- pivot_wider(B6_dated_long,
    names_from = statistic,
    values_from = day
)

# ------------------------------------------------------------------------------
# B11
B11_dated <- full_join( # Add d18O ShellChron result
    select(
        B11_13C_ShellChronage,
        mean.day,
        CL95.day,
        D
    ), select(
        B11_18O_ShellChronage,
        mean.day,
        CL95.day,
        D
    ),
    by = "D",
    suffix = c(
        ".SG.d18O",
        ".SG.d13C"
        )
) %>%
mutate(
    specimen_name = "B11",
    species = "Oscillopha figari"
)

# Pivot to long format
B11_dated_long <- pivot_longer(
    B11_dated,
    !c(D, specimen_name, species),
    names_to = "variable",
    values_to = "day"
)

# Separate day values from uncertainties
B11_dated_long <- separate_wider_delim(
    B11_dated_long,
    variable,
    delim = ".",
    names = c("statistic",
        "unit",
        "method",
        "proxy"
    )
)

# Pivot to wider format by statistic
B11_dated_long <- pivot_wider(B11_dated_long,
    names_from = statistic,
    values_from = day
)

# ------------------------------------------------------------------------------
# H576
H576_dated <- full_join( # Add d18O ShellChron result
    select(
        H576_13C_ShellChronage,
        mean.day,
        CL95.day,
        D
    ), select(
        H576_18O_ShellChronage,
        mean.day,
        CL95.day,
        D
    ),
    by = "D",
    suffix = c(
        ".SG.d18O",
        ".SG.d13C"
        )
) %>%
mutate(
    specimen_name = "H576",
    species = "Torreites sanchezi"
)

# Pivot to long format
H576_dated_long <- pivot_longer(
    H576_dated,
    !c(D, specimen_name, species),
    names_to = "variable",
    values_to = "day"
)

# Separate day values from uncertainties
H576_dated_long <- separate_wider_delim(
    H576_dated_long,
    variable,
    delim = ".",
    names = c("statistic",
        "unit",
        "method",
        "proxy"
    )
)

# Pivot to wider format by statistic
H576_dated_long <- pivot_wider(H576_dated_long,
    names_from = statistic,
    values_from = day
)

# ------------------------------------------------------------------------------
# H579A
H579A_dated <- full_join( # Add d18O ShellChron result
    select(
        H579A_13C_ShellChronage,
        mean.day,
        CL95.day,
        D
    ), select(
        H579A_18O_ShellChronage,
        mean.day,
        CL95.day,
        D
    ),
    by = "D",
    suffix = c(
        ".SG.d18O",
        ".SG.d13C"
        )
) %>%
mutate(
    specimen_name = "H579A",
    species = "Torreites sanchezi"
)

# Pivot to long format
H579A_dated_long <- pivot_longer(
    H579A_dated,
    !c(D, specimen_name, species),
    names_to = "variable",
    values_to = "day"
)

# Separate day values from uncertainties
H579A_dated_long <- separate_wider_delim(
    H579A_dated_long,
    variable,
    delim = ".",
    names = c("statistic",
        "unit",
        "method",
        "proxy"
    )
)

# Pivot to wider format by statistic
H579A_dated_long <- pivot_wider(H579A_dated_long,
    names_from = statistic,
    values_from = day
)

# ------------------------------------------------------------------------------
# H579B
H579B_dated <- full_join( # Add d18O ShellChron result
    select(
        H579B_13C_ShellChronage,
        mean.day,
        CL95.day,
        D
    ), select(
        H579B_18O_ShellChronage,
        mean.day,
        CL95.day,
        D
    ),
    by = "D",
    suffix = c(
        ".SG.d18O",
        ".SG.d13C"
        )
) %>%
mutate(
    specimen_name = "H579B",
    species = "Torreites sanchezi"
)

# Pivot to long format
H579B_dated_long <- pivot_longer(
    H579B_dated,
    !c(D, specimen_name, species),
    names_to = "variable",
    values_to = "day"
)

# Separate day values from uncertainties
H579B_dated_long <- separate_wider_delim(
    H579B_dated_long,
    variable,
    delim = ".",
    names = c("statistic",
        "unit",
        "method",
        "proxy"
    )
)

# Pivot to wider format by statistic
H579B_dated_long <- pivot_wider(H579B_dated_long,
    names_from = statistic,
    values_from = day
)

# ------------------------------------------------------------------------------
# H579C
H579C_dated <- full_join( # Add d18O ShellChron result
    select(
        H579C_13C_ShellChronage,
        mean.day,
        CL95.day,
        D
    ), select(
        H579C_18O_ShellChronage,
        mean.day,
        CL95.day,
        D
    ),
    by = "D",
    suffix = c(
        ".SG.d18O",
        ".SG.d13C"
        )
) %>%
mutate(
    specimen_name = "H579C",
    species = "Torreites sanchezi"
)

# Pivot to long format
H579C_dated_long <- pivot_longer(
    H579C_dated,
    !c(D, specimen_name, species),
    names_to = "variable",
    values_to = "day"
)

# Separate day values from uncertainties
H579C_dated_long <- separate_wider_delim(
    H579C_dated_long,
    variable,
    delim = ".",
    names = c("statistic",
        "unit",
        "method",
        "proxy"
    )
)

# Pivot to wider format by statistic
H579C_dated_long <- pivot_wider(H579C_dated_long,
    names_from = statistic,
    values_from = day
)

# ------------------------------------------------------------------------------
# H579D
H579D_dated <- full_join( # Add d18O ShellChron result
    select(
        H579D_13C_ShellChronage,
        mean.day,
        CL95.day,
        D
    ), select(
        H579D_18O_ShellChronage,
        mean.day,
        CL95.day,
        D
    ),
    by = "D",
    suffix = c(
        ".SG.d18O",
        ".SG.d13C"
        )
) %>%
mutate(
    specimen_name = "H579D",
    species = "Torreites sanchezi"
)

# Pivot to long format
H579D_dated_long <- pivot_longer(
    H579D_dated,
    !c(D, specimen_name, species),
    names_to = "variable",
    values_to = "day"
)

# Separate day values from uncertainties
H579D_dated_long <- separate_wider_delim(
    H579D_dated_long,
    variable,
    delim = ".",
    names = c("statistic",
        "unit",
        "method",
        "proxy"
    )
)

# Pivot to wider format by statistic
H579D_dated_long <- pivot_wider(H579D_dated_long,
    names_from = statistic,
    values_from = day
)

# ------------------------------------------------------------------------------
# H579E
H579E_dated <- full_join( # Add d18O ShellChron result
    select(
        H579E_13C_ShellChronage,
        mean.day,
        CL95.day,
        D
    ), select(
        H579E_18O_ShellChronage,
        mean.day,
        CL95.day,
        D
    ),
    by = "D",
    suffix = c(
        ".SG.d18O",
        ".SG.d13C"
        )
) %>%
mutate(
    specimen_name = "H579E",
    species = "Torreites sanchezi"
)

# Pivot to long format
H579E_dated_long <- pivot_longer(
    H579E_dated,
    !c(D, specimen_name, species),
    names_to = "variable",
    values_to = "day"
)

# Separate day values from uncertainties
H579E_dated_long <- separate_wider_delim(
    H579E_dated_long,
    variable,
    delim = ".",
    names = c("statistic",
        "unit",
        "method",
        "proxy"
    )
)

# Pivot to wider format by statistic
H579E_dated_long <- pivot_wider(H579E_dated_long,
    names_from = statistic,
    values_from = day
)

# ------------------------------------------------------------------------------
# H585
H585_dated <- full_join( # Add d18O ShellChron result
    select(
        H585_13C_ShellChronage,
        mean.day,
        CL95.day,
        D
    ), select(
        H585_18O_ShellChronage,
        mean.day,
        CL95.day,
        D
    ),
    by = "D",
    suffix = c(
        ".SG.d18O",
        ".SG.d13C"
        )
) %>%
mutate(
    specimen_name = "H585",
    species = "Torreites sanchezi"
)

# Pivot to long format
H585_dated_long <- pivot_longer(
    H585_dated,
    !c(D, specimen_name, species),
    names_to = "variable",
    values_to = "day"
)

# Separate day values from uncertainties
H585_dated_long <- separate_wider_delim(
    H585_dated_long,
    variable,
    delim = ".",
    names = c("statistic",
        "unit",
        "method",
        "proxy"
    )
)

# Pivot to wider format by statistic
H585_dated_long <- pivot_wider(H585_dated_long,
    names_from = statistic,
    values_from = day
)

# ------------------------------------------------------------------------------
# Combine all growth data into one dataframe
Age_model_results <- bind_rows(
    B6_dated_long,
    B10_dated_long,
    B11_dated_long,
    H576_dated_long,
    H579A_dated_long,
    H579B_dated_long,
    H579C_dated_long,
    H579D_dated_long,
    H579E_dated_long,
    H585_dated_long
) %>% # start each curve with D = 0
    group_by(specimen_name) %>%
    mutate(
        D_zero = D - min(D)
    )

# Export growth rate data
write.csv(Age_model_results, "2_Results_dating/Age_model_results.csv", row.names = FALSE)

# ------------------------------------------------------------------------------
# Create tiered color scale
specimencolors <- c(
    brewer.pal(3, "Blues")[-3],
    brewer.pal(4, "Greens"),
    brewer.pal(3, "Reds")[-3],
    brewer.pal(9, "Greens"),
    brewer.pal(5, "Greens")
)
names(specimencolors) <- apply(unique(select(Age_model_results, species, specimen_name, proxy)) , 1, paste, collapse = "_")

# Plot growth curves for all specimens and proxies
Gcurves <- ggplot(Age_model_results) +
    geom_vline(xintercept = seq(-4, 11, 1) * 365) + # Add year lines
    geom_smooth( # Add smoothed growth curves per species
        aes(
            x = mean,
            y = D_zero / 1000,
            color = paste(species, specimen_name, proxy, sep = "_"),
            fill = paste(species, specimen_name, proxy, sep = "_")
        ),
        method = "loess",
        span = 0.2
    ) +
    geom_point( # Add datapoints
        aes(
            x = mean,
            y = D_zero / 1000,
            color = paste(species, specimen_name, proxy, sep = "_")
        ),
        alpha = 0.5
    ) +
    geom_errorbar( # Add vertical (height) uncertainties, assuming 500 um sampling
        aes(
            x = mean,
            ymin = D_zero / 1000 - 0.5,
            ymax = D_zero / 1000 + 0.5,
            color = paste(species, specimen_name, proxy, sep = "_")
        ),
        alpha = 0.5
    ) +
    geom_errorbarh( # Add horizontal (age) uncertainties based on ShellChron
        aes(
            xmin = mean - CL95,
            xmax = mean + CL95,
            y = D_zero / 1000,
            color = paste(species, specimen_name, proxy, sep = "_")
        ),
        alpha = 0.5
    ) +
    scale_x_continuous("age (days)",
        limits = c(-3, 11) * 365) +
    scale_y_continuous("Height (mm)",
        limits = c(0, 180)) +
    scale_color_manual(values = specimencolors) +
    scale_fill_manual(values = specimencolors) +
    guides(fill = guide_legend(ncol = 1),
        color = guide_legend(ncol = 1)
    ) +
    labs(fill = "specimen",
        color = "specimen"
    ) +
    theme_minimal()

# Plot growth curves for all specimens for d18O age models
Gcurves_d18O <- ggplot(filter(Age_model_results, proxy == "d18O")) +
    geom_vline(xintercept = seq(-4, 11, 1) * 365) + # Add year lines
    geom_smooth( # Add smoothed growth curves per species
        aes(
            x = mean,
            y = D_zero / 1000,
            color = paste(species, specimen_name, proxy, sep = "_"),
            fill = paste(species, specimen_name, proxy, sep = "_")
        ),
        method = "loess",
        span = 0.2
    ) +
    geom_point( # Add datapoints
        aes(
            x = mean,
            y = D_zero / 1000,
            color = paste(species, specimen_name, proxy, sep = "_")
        ),
        alpha = 0.5
    ) +
    geom_errorbar( # Add vertical (height) uncertainties, assuming 500 um sampling
        aes(
            x = mean,
            ymin = D_zero / 1000 - 0.5,
            ymax = D_zero / 1000 + 0.5,
            color = paste(species, specimen_name, proxy, sep = "_")
        ),
        alpha = 0.5
    ) +
    geom_errorbarh( # Add horizontal (age) uncertainties based on ShellChron
        aes(
            xmin = mean - CL95,
            xmax = mean + CL95,
            y = D_zero / 1000,
            color = paste(species, specimen_name, proxy, sep = "_")
        ),
        alpha = 0.5
    ) +
    scale_x_continuous("age (days)",
        limits = c(-3, 11) * 365) +
    scale_y_continuous("Height (mm)",
        limits = c(0, 180)) +
    scale_color_manual(values = specimencolors) +
    scale_fill_manual(values = specimencolors) +
    guides(fill = guide_legend(ncol = 1),
        color = guide_legend(ncol = 1)
    ) +
    labs(fill = "specimen",
        color = "specimen"
    ) +
    theme_minimal()

# Plot growth curves for all specimens for d13C age models
Gcurves_d13C <- ggplot(filter(Age_model_results, proxy == "d13C")) +
    geom_vline(xintercept = seq(-4, 11, 1) * 365) + # Add year lines
    geom_smooth( # Add smoothed growth curves per species
        aes(
            x = mean,
            y = D_zero / 1000,
            color = paste(species, specimen_name, proxy, sep = "_"),
            fill = paste(species, specimen_name, proxy, sep = "_")
        ),
        method = "loess",
        span = 0.2
    ) +
    geom_point( # Add datapoints
        aes(
            x = mean,
            y = D_zero / 1000,
            color = paste(species, specimen_name, proxy, sep = "_")
        ),
        alpha = 0.5
    ) +
    geom_errorbar( # Add vertical (height) uncertainties, assuming 500 um sampling
        aes(
            x = mean,
            ymin = D_zero / 1000 - 0.5,
            ymax = D_zero / 1000 + 0.5,
            color = paste(species, specimen_name, proxy, sep = "_")
        ),
        alpha = 0.5
    ) +
    geom_errorbarh( # Add horizontal (age) uncertainties based on ShellChron
        aes(
            xmin = mean - CL95,
            xmax = mean + CL95,
            y = D_zero / 1000,
            color = paste(species, specimen_name, proxy, sep = "_")
        ),
        alpha = 0.5
    ) +
    scale_x_continuous("age (days)",
        limits = c(-3, 11) * 365) +
    scale_y_continuous("Height (mm)",
        limits = c(0, 180)) +
    scale_color_manual(values = specimencolors) +
    scale_fill_manual(values = specimencolors) +
    guides(fill = guide_legend(ncol = 1),
        color = guide_legend(ncol = 1)
    ) +
    labs(fill = "specimen",
        color = "specimen"
    ) +
    theme_minimal()