# Add chronology to rudist data using ShellChron
# Project "chemical_data"

# devtools::install_github("nielsjdewinter/ShellChron")

require(ShellChron)
require(signal)
require(tidyverse)

All_sample_data <- read.csv("H579A_SI.csv")

# Apply ShellChron to create chronology based on d13Cc
All_sample_data$ShellChron_day_cumulative <- NA # Create cu\olumn to store cumulative day age model results
All_sample_data$ShellChron_DOY <- NA # Create column to store Day of the Year results from ShellChron
All_sample_data$ShellChron_DOY_err <- NA # Create column to store uncertainties on Day of the Year results from ShellChron

# Smooth d13C curve for better performance (not used here)
All_sample_data$d13C_smooth <- sgolayfilt(All_sample_data$d13C, p = 3, n = 7)

# ------------------------------------------------------------------------------
# Create SG input for d13C data
H579A_13C <- All_sample_data
H579A_13Csg <- H579A_13C %>%
    select(
        D = D,
        d18Oc = d13C
    ) %>%
    mutate(
        d18Oc_err = 0.08,
        D = D * 1000,
        D_err = 500,
        YEARMARKER = 0
    )

H579A_13Csg <- H579A_13Csg[order(H579A_13Csg$D), ] # order by depth
rownames(H579A_13Csg) <- seq(1, nrow(H579A_13Csg)) # add new rownames to more easily reference data
H579A_13Csg$YEARMARKER[c(3, 18, 43, 74, 85, 98, 107, 112)] <- 1 # Add manual year markers

# Plot to check
H579A_13C_plot <- ggplot(H579A_13Csg) +
geom_point(aes(x = D,
    y = d18Oc)) +
geom_line(data = H579A_13Csg %>%
    group_by(D) %>%
    summarize(mean_d13C = mean(d18Oc)),
    aes(x = D,
        y = mean_d13C)) +
geom_vline(data = filter(H579A_13Csg, YEARMARKER == 1),
    aes(xintercept = D))

# Run ShellChron model
wrap_function(
    input_from_file = FALSE, # Should input be read from a file?
    object_name = H579A_13Csg, # Name of object with input (only if input_from_file = FALSE)
    transfer_function = "KimONeil97", # Set transfer function of the record, default is Kim and O'Neil 1997.
    t_int = 1, # Set time interval in days
    T_per = 365, # Set annual time period in days (default = 365)
    d18Ow = 0.15, # Set d13Cw value or vector (default = constant year-round at 0 VSMOW). Alternative options are either one value (assumed constant year-round) or a vector with length T_per / t_int and interval t_int specifying d13Cw evolution through one year.
    t_maxtemp = 182.5, # Define the day of the year at which temperature is heighest. Default = Assume that the day of maximum temperature is helfway through the year
    SCEUApar = c(1, 25, 10000, 5, 0.01, 0.01), # Set parameters for SCEUA optimization (iniflg, ngs, maxn, kstop, pcento, peps)
    sinfit = TRUE, # Apply sinusoidal fitting to guess initial parameters for SCEUA optimization? (TRUE/FALSE)
    MC = 1000, # Number of MC simulations to include measurement error into error analysis. Default = 1000 (if MC = 0, error on D and d13C measurements not considered)
    plot = TRUE, # Should intermediate plots be given to track progress? WARNING: plotting makes the script much slower, especially for long datasets.
    plot_export = TRUE, # Should a plot of the results be saved as PDF?
    export_raw = TRUE, # Should the results of all individual model runs be exported as CSV files?
    export_path = "E:/Dropbox/Research/Manuscripts/Rudist B10 growth Daydacna_ShellChron/chemical data/H579A_13C_ShellChron"
)
