# Plot modern SST range and mollusk occurrences
# Project "Saiwan_data_processing"
# Author: Niels J. de Winter
# Study: "Living on the edge: Response of rudist bivalves (Hippuritida) to hot and highly seasonal climate in the low-latitude Saiwan site, Oman"

require(tidyverse)
require(RColorBrewer)
require(ggbeeswarm)
require(ggridges)
require(ggpubr)

# Load data
globalSST <- read.csv("10_NOAA_data/globalSST_data.csv")
bivalve_grid <- read.csv("10_OBIS_data/bivalve_monthly_extremes.csv")
Monthly_data <- read.csv("8_Results_monthly_extremes/Monthly_SI_summary2.csv")

# Plot maximum SSTs
maxSSTmap <- ggplot() + 
    geom_tile(data = filter(globalSST, !complete.cases(globalSST), year == 1981, month == "Jan"), # Mask land
        aes(
            x = longitude,
            y = latitude
        ),
        fill = "grey"
    ) +
    geom_tile(data = filter(bivalve_grid, complete.cases(bivalve_grid)), # Isolate rows with bivalve occurrences
        aes(
            x = longitude,
            y = latitude,
            fill = maxT
        )
    ) +
    scale_fill_gradientn(
        colours = brewer.pal(3, "YlOrRd"),
        values = c(0, .5, 1),
        limits = c(-5, 35)
    ) +
    ggtitle("Maximum sea surface temperatures") +
    theme_bw()

# Plot minimum SSTs
minSSTmap <- ggplot() + 
    geom_tile(data = filter(globalSST, !complete.cases(globalSST), year == 1981, month == "Jan"), # Mask land
        aes(
            x = longitude,
            y = latitude
        ),
        fill = "grey"
    ) +
    geom_tile(data = filter(bivalve_grid, complete.cases(bivalve_grid)), # Isolate rows with bivalve occurrences
        aes(
            x = longitude,
            y = latitude,
            fill = minT
        )
    ) +
    scale_fill_gradientn(
        colours = rev(brewer.pal(3, "Blues")),
        values = c(0, .5, 1),
        limits = c(-5, 35)
    ) +
    ggtitle("Minimum sea surface temperatures") +
    theme_bw()

# Plot abundance of bivalves in OBIS data
Abundancemap <- ggplot() + 
    geom_tile(data = filter(globalSST, !complete.cases(globalSST), year == 1981, month == "Jan"), # Mask land
        aes(
            x = longitude,
            y = latitude
        ),
        fill = "grey"
    ) +
    geom_tile(data = filter(bivalve_grid, complete.cases(bivalve_grid), N != 0), # Isolate rows with bivalve occurrences
        aes(
            x = longitude,
            y = latitude,
            fill = log10(N)
        )
    ) +
    scale_fill_gradientn(
        colours = brewer.pal(3, "Greens"),
        values = c(0, .5, 1),
        limits = c(0, max(log10(bivalve_grid$N)))
    ) +
    ggtitle("Abundance of bivalves") +
    theme_bw()

# Pivot bivalve grid for violin plot
bivalve_grid_long <- pivot_longer(
    bivalve_grid,
    cols = c(maxT, minT),
    names_to = "category",
    values_to = "T"
)

# Process Monthly Saiwan data
Monthly_data2 <- select(Monthly_data,
    Tmean,
    N,
    species
) %>%
rename(
    T = Tmean,
    category = species
)

# Merge with modern bivalve data
bivalve_grid_long <- full_join(
    bivalve_grid_long,
    Monthly_data2
)

# Create manual color scale
Category_colors <- c(
    "maxT" = "darkred",
    "minT" = "lightblue",
    "Vaccinites vesiculosus" = "blue",
    "Oscillopha figari" = "red",
    "Torreites sanchezi" = "darkgreen"
)

# Plot bivalve abundance against max and min temperature
Extreme_temperatures <- ggplot(data = filter(bivalve_grid_long, N != 0)) +
# Add gradient of maximum tolerated temperatures from Roebuck Bay (Compton et al., 2007)
geom_rect(
    alpha = 0.01,
    xmin = 32.72,
    xmax = 41.76,
    ymin = -Inf,
    ymax = Inf,
    fill = "lightcoral",
    color = NA
) +
geom_density_ridges(
    aes(
        x = T,
        y = category,
        fill = category
    ),
    alpha = 0.5,
    scale = 0.95,
    rel_min_height = 0.01,
    quantile_lines = TRUE
) +
geom_quasirandom(
    aes(
        x = T,
        y = category,
        size = log10(N)
    ),
    alpha = 0.05,
    method = "maxout",
    orientation = "x"
) +
# geom_freqpoly(
#     aes(
#         x = T,
#         color = category
#     ),
#     binwidth = 1,
#     position = "dodge",
#     linewidth = 2,
#     alpha = 0.5
# ) +
# geom_histogram(
#     aes(
#         x = T,
#         fill = category
#     ),
#     binwidth = 1,
#     position = "dodge"
# ) +
scale_x_continuous("Monthly temperature (degrees C)",
    breaks = seq(0, 50, 10),
    limits = c(0, 50)
) +
scale_y_discrete("",
    labels = c(
        "Maximum temperature\nmodern bivalves",
        "Minimum temperature\nmodern bivalves",
        "Monthly temperatures\nO. figari",
        "Monthly temperatures\nT. sanchezi",
        "Monthly temperatures\nV. vesiculosus"
    )
) +
scale_fill_manual(
    values = Category_colors,
    guide = "none"
) +
theme_bw() +
ggtitle("Histogram of temperatures at which modern\nand Saiwan bivalves mineralize their shells") +
theme(legend.position = "none")

# Combine plots
Monthly_data_compilation <- ggarrange(
    ggarrange(
        maxSSTmap,
        minSSTmap,
        Abundancemap,
        ncol = 1,
        labels = c("A", "B", "C")
    ),
    Extreme_temperatures,
    ncol = 2,
    labels = c("A", "D")
)