
#include <iostream>
#include "global.h"
#include "random.h"
#include "problem.h"
#include "mutation.h"
#include <cmath>

using namespace std;

static bool weakDom(IntVec const &v, IntVec const &w) { // true if v weakly dominates w
	size_t d = v.size();
	for (size_t i = 0; i < d; i++) {
		if (v[i] > w[i]) return(false);
	}
	return(true);
}
static bool dom(IntVec const &v, IntVec const &w) { // true if v dominates w
	size_t d = v.size();
	size_t cnt = 0;
	for (size_t i = 0; i < d; i++) {
		if (v[i] > w[i]) return(false);
		if (v[i] < w[i]) cnt++;
	}
	return(cnt > 0);
}
static int32_t distance(Pop_t const &pop, int32_t delta) { // distance to Pareto front
	int32_t dist = INT32_MAX;
	for (size_t i = 0; i < pop.size(); i++) {
		IntVec f = get<1>(pop[i]); // pop[i].second;
		int32_t d = f[0] + f[1] - delta;
		if (d < dist) dist = d;
	}
	return dist;
}
static bool has_found_PF(Pop_t const &pop, int32_t delta, int &noOfOptInd) {
	IntSet indexSet;
	for (size_t i = 0; i < pop.size(); i++) {
		IntVec f = get<1>(pop[i]);
		int32_t dist = f[0] + f[1] - delta;
		if (dist > 0) continue;
		indexSet.insert(f[0]);
	}
	noOfOptInd = static_cast<int32_t>(indexSet.size());
	return(noOfOptInd == delta + 1);
}

static void copyPop(Pop_t& pnew, Pop_t& pold) {
	for (size_t i = 0; i < pold.size(); i++)
		pnew.push_back(pold[i]);
}
static void copyPop(Pop_t& pnew, Pop_t& pold, IntSet &ignore) {
	if (ignore.size() == 0) copyPop(pnew, pold);
	else {
		auto it = ignore.begin();
		for (size_t i = 0; i < pold.size(); i++) {
			if (it != ignore.end() && i == *it) {
				it++;
				continue;
			}
			pnew.push_back(pold[i]);
		}
	}
}
static void copyVec(DblVec& vnew, IntVec& vold) { // and cast int to double
	for (size_t i = 0; i < vnew.size(); i++) {
		vnew[i] = vold[i];
	}
}
static void addVec(DblVec& dest, DblVec const& add) {
	for (size_t i = 0; i < dest.size(); i++) {
		dest[i] += add[i];
	}
}
static void subVec(IntVec& dest, IntVec const& sub) {
	for (size_t i = 0; i < dest.size(); i++) {
		dest[i] -= sub[i];
	}
}
static void multVec(DblVec& dest, double c) {
	for (size_t i = 0; i < dest.size(); i++) {
		dest[i] *= c;
	}
}
double norm1(DblVec const& x) {
	double sum = 0;
	for (size_t i = 0; i < x.size(); i++) sum += abs(x[i]);
	return(sum);
}
static void updateCSA(Ind_t const& ind, Ind_t& kid, double c, double beta) {
	const double pi = 3.14159265358979323846;
	const double ezbar = 2 * sqrt((2 / pi) * c / (2 - c));
	DblVec zp = { 0, 0 };
	DblVec csa = get_p(kid);
	IntVec x = get_xvec(ind);
	IntVec y = get_xvec(kid);
	subVec(y, x); // yields Z+
	copyVec(zp, y);  // cast to double
	multVec(zp, 1. / get_s(ind)); // normalize
	multVec(csa, 1 - c);
	multVec(zp, c);
	addVec(csa, zp);
	get_p(kid) = csa;
	// update step size
	double zlen = norm1(csa);	
	double quot = zlen / ezbar - 1;
	get_s(kid) = get_s(ind) * exp(beta * quot);
}
inline double qval(double ss) {
	return (1 - ss / (1 + sqrt(1 + ss * ss)));
}


int main(int argc, char** argv)
{
	int32_t d = 2; // number of objectives (fixed at the moment)

	size_t maxFE = 1000000;	// maximum number of function evaluations
	size_t cntFE = 0;

	//environment parameters:
	//	pid   [int]	  problem id
	//	xid   [int]   starting point id
	//	s0    [float] initial step size > 0
	//  mt    [char] mutation type (S: subdimensional / F: fulldimensional)
	//	c     [float] weight of weighted mean; in (0,1)
	//  beta  [float] exponent of the step size update
	//  redo  [char*] if true, redo sampling in case of zero mutation

	if (argc < 8) {
		cerr << "usage: " << argv[0] << " pid xid s0 muttype c beta redo" << endl;
		exit(1);
	}

	int pid = atoi(argv[1]);      // problem id
	int xid = atoi(argv[2]);	  // id of starting point
	double s = atof(argv[3]);	  // initial mean step size
	char mt = argv[4][0];         // mutation type (S: subdimensional / F: fulldimensional)
	double c = atof(argv[5]);     // weighting factor
	double beta = atof(argv[6]);  // exponent
	string str(argv[7]);
	bool redo = (str == "true") || (str == "T") || (str == "1");  // if true, redo sampling in case of zero mutation

	if (s <= 0 || c < 0 || c > 1 || beta < 0 || (mt != 'S' && mt != 'F')) { // incomplete check
		cerr << argv[0] << ": check your hyperparams!" << endl;
		exit(1);
	}

	// define problem
	BOP bop(pid);
	int32_t n = bop.getDim();
	int32_t delta = bop.getSizePF() - 1;

	Mutation* mut = nullptr;
	if (mt == 'F') mut = new MutationLaplaceFullspace();
	if (mt == 'S') mut = new MutationLaplaceSubspace();

	// set starting point
	IntVec x = bop.get_x(xid);

	// set initial step size
	double ss = (mt == 'F') ? s / n : s;

	// set initial CSA vector
	DblVec csaVec = { 0., 0. };

	// initial individual			
	IntVec fx = { bop.f1(x), bop.f2(x) };
	cntFE++;
	Ind_t ind = make_tuple(x, fx, ss, csaVec);

	// initial population
	Pop_t pop[2];
	size_t ff = 0; // flipflop
	pop[ff] = { ind };
	pop[1 - ff].clear();

	int32_t na = 0; // number of accepted offspring (not used)
	int32_t ni = 0; // number of improving offspring (not used)

	Irnd irnd;	// RNG for selecting parent

	// 1st reporting
	int noOfOptInd = 0;
	bool found_PF = has_found_PF(pop[ff], delta, noOfOptInd);
	int32_t dist = distance(pop[ff], delta);
	cout << cntFE << " " << dist << " " << ss << " " << pop[ff].size() << " " << noOfOptInd << endl;

	while (cntFE <= maxFE) {
		int32_t mu = static_cast<int32_t>(pop[ff].size());
		size_t k = irnd.rnd(0, mu - 1);
		ind = pop[ff][k];
		IntVec y(get_xvec(ind));	// y is copy of randomly chosen individual
		double q = qval(get_s(ind));
		if (redo) {			
			do {
				mut->mutate(q, y);
			} while (y == get_xvec(ind));
		}
		else {
			mut->mutate(q, y);
			if (y == get_xvec(ind)) { // no change by mutation
				//cntFE++;
				//continue; // don't evaluate but increment FE
				//### todo: can skip adaptation this way!
			}
		}
		IntVec fy{ bop.f1(y), bop.f2(y) }; // evaluate y
		cntFE++;
		Ind_t kid = ind;	// copy the history of the individual
		get_xvec(kid) = y;  // store new position
		get_fx(kid) = fy;   // store new fitness

		bool isAccepted = false;

		// selection stage 1
		IntSet E, B, W, I;
		for (int32_t i = 0; i < mu; i++) {
			Ind_t z = pop[ff][i];
			IntVec fz = get_fx(z);
			if (fz == fy) E.insert(i);			// fz equal fy
			else if (dom(fz, fy)) B.insert(i);  // fz better than fy
			else if (dom(fy, fz)) W.insert(i);  // fy better than fz
			else I.insert(i);					// fy and fz incomparable
		}
		// selection stage 2
		if (E.size() > 0) {		// remark: E can contain at most a single element
			copyPop(pop[1 - ff], pop[ff]);
			int32_t k = *E.begin();
			if (get_xvec(pop[1 - ff][k]) != get_xvec(kid)) {
				updateCSA(ind, kid, c, beta);
				na = na + 1; // only count if different x!
				pop[1 - ff][k] = kid; // replace
			}
			ss = get_s(kid);
		}
		else if (W.size() > 0) {
			updateCSA(ind, kid, c, beta);
			copyPop(pop[1 - ff], pop[ff], W); // copy pop but skip over individuals in W
			pop[1 - ff].push_back(kid);
			ni = ni + 1;
			ss = get_s(kid);
		}
		else if (B.size() > 0) {
			updateCSA(ind, kid, c, beta); // really?
			copyPop(pop[1 - ff], pop[ff]);
			ss = get_s(kid);
		}
		else {
			updateCSA(ind, kid, c, beta);
			copyPop(pop[1 - ff], pop[ff]);
			pop[1 - ff].push_back(kid);
			na = na + 1;
			ss = get_s(kid);
		}

		// tidy up
		ff = 1 - ff; // now pop[ff] is the current population
		pop[1 - ff].clear(); // delete old population

		// reporting
		noOfOptInd = 0;
		found_PF = has_found_PF(pop[ff], delta, noOfOptInd);
		dist = distance(pop[ff], delta);
		//cout << ni << " " << na << " : ";
		cout << cntFE << " " << dist << " " << ss << " " << mu << " " << noOfOptInd << endl;

		if (found_PF) break;
	}

	return 0;
}
