library(raster)
library(abind)
library(tidyverse)
library(readxl)
library(lubridate)
library(ggplot2)
library(cubature)
library(mosaic)
library(stats)
library(suncalc)
library(units)
################## FUNCTION: raster_as ####
raster_as=function(matrix_to_rasterise, reference_raster){
  output=raster(matrix_to_rasterise)
  extent(output)=extent(reference_raster)
  crs(output)=crs(reference_raster)
  
  return(output)
}

################## FUNCTION: BINDING COLUMN OF DIFFERENT LENGTH ##########################
bind_dif_columns<-function(array_to_add, array_destination){
  
  if(nrow(array_to_add)>nrow(array_destination)){
    
    dif=nrow(array_to_add)-nrow(array_destination)
    add=matrix(data= NaN,dif,ncol(array_destination))
    array_destination=smartbind(array_destination,add)
    array_destination=cbind(array_destination,array_to_add)
  }
  if(nrow(array_to_add)==nrow(array_destination)){
    array_destination=cbind(array_destination,array_to_add)
  }
  else{
    dif=nrow(array_destination)-nrow(array_to_add)
    add=matrix(data= NaN,dif,ncol(array_to_add))
    array_to_add=smartbind(array_to_add,add)
    array_destination=cbind(array_destination,array_to_add)
  }
  return(array_destination)
}

################## FUNCTION: GAP_DETECTION ###########################

gap_detection<-function(array_name, threshold_pr, threshold_hours){
  
  l=1
  m=1 #beginning of the gap
  j=1
  GAPS=data.frame() # RETURNS: initial and final position of the gaps larger than a threshold, and the length of the gap
  
  while (m < length(array_name)){
    if(array_name[m]<threshold_pr){
      k=1
      while (1==1){ 
        if(array_name[m+k]<threshold_pr & (m+k+1)<=length(array_name)){
          k=k+1
        }
        else{
          if(k>threshold_hours){
            co=c(m, m+k, k)
            for (l in 1:3){
              GAPS[j,l]=co[l]
            }
            m=m+k
            j=j+1
            break 
          }
          else {
            m=m+k
            break
          }
        }
      }
    }
    else {
      m=m+1
    }
  } 
  colnames(GAPS)=c("index start", "index end","length")
  return(GAPS) 
}

################## FUNCTION: CANOPY_INTERCEPTION ###########################

canopy_interception<-function(GAPS_matrix, threshold_pr, PR_array){
  
  add_end_line=GAPS_matrix[1,]
  add_end_line[,]=0
  add_end_line[1,1]=length(PR_array)
  GAPS_matrix=rbind(GAPS_matrix,add_end_line)
  
  
  for(h in 1:(length(GAPS_matrix$`index start`)-1)){
    PR_array[GAPS_matrix$`index start`[h]:(GAPS_matrix$`index end`[h]-1)]=0
    
    if(PR_array[GAPS_matrix$`index end`[h]]>=threshold_pr){
      PR_array[GAPS_matrix$`index end`[h]]=PR_array[GAPS_matrix$`index end`[h]]-threshold_pr
    }else{
      def=threshold_pr
      d=0
      event_index=c()
      while(def>0 & (GAPS_matrix$`index end`[h]+d)<length(PR_array) & (GAPS_matrix$`index end`[h]+d)< GAPS_matrix$`index end`[h+1]){
        event_index[d]=GAPS_matrix$`index end`[h]+d
        if(PR_array[GAPS_matrix$`index end`[h]+d]>=def){
          PR_array[GAPS_matrix$`index end`[h]+d]=PR_array[GAPS_matrix$`index end`[h]+d]-def
          def=0
        }else{
          
          def=def-PR_array[GAPS_matrix$`index end`[h]+d]
          PR_array[GAPS_matrix$`index end`[h]+d]=0
        }
        d=d+1
        
      }
    }
    
  }
  if(PR_array[length(PR_array)]<0){
    PR_array[length(PR_array)]=0
  }
  return(PR_array)
}


################## FUNCTION: I requirement limit #######
B_limit=function(){
  B_limit=as.character(menu(c("Critical Point","Field Capacity"),title="Up to which Water Content Irrigation contributes/contributed to?"))
  return(B_limit)
}

################## FUNCTION: Irrigation withdrawals #######
I_units=function(){
  Irr_req=as.character(menu(c("mm","m3","both"),title="which Units do you want irrigation volumes to be expressed in?"))
  return(Irr_req)
}






#################################### WaterCROP applied to maize in Italy  ###########################################
# USED DATA:
# Italian territory: raster of 0s and 1s to shape out the italian territory ->  italy.tif 
# Climate zones: raster with climate zones on italy extension -> climate_zones.tif
# ET0 values: Evapotranspiration data per month expressed in daily values [mm/day] -> ET0mm_day.Mat
# Precipitation values: ERA5 precipitation data  -> 
# Kc: crop coefficient ->
# Irrigated area:  raster with extension of the irrigated area per pixel per crop for 2010. ita_crop extension. [ha] -> 
# Sowing date for irrigated crop: raster file with sowing dates in case of irrigated crop in 2010 -> ì
# Harvesting time for rainfed crops: raster file with harvesting dates in case of irrigated crop in 2010 -> 
# Length Growing Period: raster file with lgp data in case of irrigated crop in 2010 -> 
# Available Water Content: 


######### UPLOAD USER CHOICES ###########################################################
crop="maize" #(e.g. "rice","olive","vine","barley","sorghum","wheat","potato","sugarbeet","temperate fruit","other pulses","other cereals")
B_l=B_limit()# chose the blue water (B) limit, 1=Critical Point, 2=Field Capacity
from_B_to_I=I_units() #chose the units for irrigation water output: mm, m3, both

year_start=as.numeric(readline(prompt = "Enter the first year of interest:")) # e.g. interested in hy=1998-1999, enter 1998
year_end=as.numeric(readline(prompt = "Enter the last year of interest:")) #e.g. interested in hy=1998-1999, enter 1999


saving_location=paste0("C:/Users/Nike/OneDrive/Desktop/WaterCROP pulito/output/")#set the saving location
input_location="C:/Users/Nike/OneDrive/Desktop/WaterCROP pulito/input/" #set the location of the general folder where input folders are. note: Input folders are "functional", "climate","crop","irrigation","soil"


######### UPLOAD FUCTIONAL FILES #######################################################

#Read raster file of italian territory
ita=raster(paste0(input_location,"functional/italy.tif"))#raster of 0s and 1s to define the italian territory

load(paste0(input_location,"functional/in_quali_celle_grandi.RData"))#in_quali_celle_Grandi: relation between model cells and ERA5 cells
m_index=read.table(paste0(input_location,"functional/matrix_index_cells.txt"), sep=",") #matrix of indexes of cells

file_code_crop=read.table(paste0(input_location,"functional/crop_code_file.txt"), sep=",", header = TRUE)
code=file_code_crop$code[which(file_code_crop$crop==crop)]





############## WATERCROP V2 2024 CODE #############################################################################################

years=seq(year_start,year_end,by=1)
for(y in 2:length(years)){

year_1= years[y-1]
year_2= years[y]

print(paste0("working on year: ", year_1,"_",year_2))

### print info metadata
info=c()
info[1]=paste0("crop: ",code)
info[2]=paste0("Precipitation: hourly")
info[3]=paste0("hydrological year: ",year_1,"-",year_2)
if(B_l==1){
  info[4]=paste0("Irrigation limit: Critical Point")
}else{
  info[4]=paste0("Irrigation limit: Field Capacity")
}
time=Sys.time()
time=gsub(":",".",time)

write.table(info,paste0(saving_location,"metadata-",time,".txt"),row.names = FALSE, col.names = FALSE)





######### UPLOAD CLIMATE DATA #########
#Precipitation
load(paste0(input_location,"climate/precipitation/Precipitation_hourly_",year_1,".RData")) #stack_raster
PR_1=stack_raster*1000 # 1000 is used to pass from [m] to [mm]

load(paste0(input_location,"climate/precipitation/Precipitation_hourly_",year_2,".RData"))#
PR_2=stack_raster*1000 #1000 is used to pass from [m] to [mm]

M_index=matrix((1:(dim(PR_1)[1]*dim(PR_1)[2])),nrow=dim(PR_1)[1],ncol=dim(PR_1)[2], byrow = TRUE)# matrix of indexes of cell of ERA5 data


# climate zones
climate_zones=raster(paste0(input_location,"climate/climate_zones.tif")) #each cell contains a number from 1 to 10 according to the climate zone




######### UPLOAD SOIL DATA #########
pedo=as.data.frame(read_excel(paste0(input_location,"soil/soil_data.xlsx", sheet="pedologic_values", range="A1:H9"))) #soil hydraulic characteristics
soil=raster(paste0(input_location,"soil/Resampled_soils.tif")) #soil type
awc=raster(paste0(input_location,"soil/awc.tif")) #awc= Available Water Content 


  
#### UPLOAD CROP-RELATED DATA ####
load(paste0(input_location,"crop/ET0_matrix.rda")) #Potential Evapotranspiration

area_irr=raster(paste0(input_location, "crop/MAIZ_ha_irrigated.tif")) #irrigated areas per cell [ha]
data_semina_irr=raster(paste0(input_location, "crop/MAIZ_sowing_date_irr.tif")) #sowing dates
lgp_irrigated=raster(paste0(input_location, "crop/MAIZ_lgp_irr.tif")) #growing period lengths
lgp_irrigated=as.matrix(lgp_irrigated)
  
kc=as.data.frame(read_excel(paste0(input_location,"crop/kc_lgp.xlsx"), sheet = crop,range = "C16:I26")) #dataframe with kc values and partitions of lgp
colnames(kc)=c("kc_ini","kc_mid","kc_end","lgp_1","lgp_2","lgp_3","lgp_4")
  

#################  
  possible_crops=c("maize","rice","olive","vine","barley","sorghum","wheat","potato","sugarbeet","temperate fruit","other pulses","other cereals")
  p=which(possible_crops==crop)
  {switch_crop=p}
  #1="maize",2="rice",3="olive",4="vine",5="barley",6="sorghum",7="wheat")
  
  switch (switch_crop, 
          
          "1"={#Maiz
            rd_ini=0.3 #initial seed positioning depth [m]. Source: FAO56
            rd_max_irrigated=1 #max roothing depth [m]. Source: FAO56
            depl_fraction=0.55 #depletion fraction
            plant_type="grass"
          },
          
          "2"={ #Rice FAO
            rd_ini=0.3 
            rd_max_irrigated=0.5  
            depl_fraction=0.2
            plant_type="grass"
          },
          
          "3"={ #olive
            rd_ini=0.4
            rd_max_irrigated= 1.2 
            depl_fraction=0.65
            plant_type="tree"
          },

          "4"={ #vine
            rd_ini= 0.4
            rd_max_irrigated= 1
            depl_fraction=0.40
            plant_type="tree"
          },
          
          "5"={#barley
            rd_ini=0.2 
            rd_max_irrigated= 1 
            depl_fraction= 0.5
            plant_type="grass"
          },
          
          "6"={
            #sorghum
            rd_ini=0.25
            rd_max_irrigated=1
            depl_fraction= 0.55
            plant_type="grass"
          },
          
          "7"={
            #Wheat
            rd_ini=0.3 
            rd_max_irrigated=1.5 
            depl_fraction=0.55 
            plant_type="grass"
          },
  )
############################# 
  
  #initialization of final MATRICES 
  ETgreen_lgp_irr=matrix(NaN,139,142) #Growing-period Green-Water
  B_lgp=matrix(NaN,139,142) #Growing-period Blue water
  ETa_lgp_irr=matrix(NaN,139,142) #Growing-period Actual Evapotranspiration
  PR_lgp_irr=matrix(NaN,139,142) #Growing-period Precipitation 
  L_lgp_irr=matrix(NaN,139,142) #Growing-period Leakage 
  R_lgp_irr=matrix(NaN,139,142) #Growing-period Runoff
  


  #monthly ET0 data will be disaggregated into daily data. TOOL:interpolation 
  #to use the interpolation method also at the beginning and end of the year, we use December values and January values respectively
  new_calendar=seq(as.Date("16-12-2009",format="%d-%m-%Y"),as.Date("16-01-2011",format="%d-%m-%Y"),"day")#random year just to create the calendar
  central_dates2=as.Date(c("16-12-2009","16-01-2010", "15-02-2010","16-03-2010","15-04-2010","16-05-2010","15-06-2010","16-07-2010","16-08-2010","15-09-2010","16-10-2010","15-11-2010","16-12-2010","16-01-2011"),format="%d-%m-%Y")
  
  
  for(r in 1:139){
    for(c in 1:142){
      
      print(paste0("Working on cell: ",r,"-",c))
      
      if(ita[r,c]==0){ #excluding the not-Italian cells from the computational process  
        next
      }
      
      if(!is.na(area_irr[r,c]) & area_irr[r,c]>0 & climate_zones[r,c]>0 & awc[r,c]>0 & tot_matrix_ET[r,c,2]>0 & lgp_irrigated[r,c]>0 & data_semina_irr[r,c]>0) { #consider only irrigated areas

        
        # STEP 0: DEFINITION OF CLIMATIC ZONE of the cell 
        climate=climate_zones[r,c,1]
        if( climate>10 || climate <1){
          climate="z"
        }
        climate=as.character(climate)
        
        #### SWITCH cases CLIMATE ZONES  ######
        switch (climate,
                "z"={ #DEFAULT
                  kc_ini=0
                  kc_mid=0
                  kc_end=0
                  lgp_1=0
                  lgp_2=0
                  lgp_3=0
                  lgp_4=0
                  lgp_1_irr=0
                  lgp_2_irr=0
                  lgp_3_irr=0
                  lgp_4_irr=0
                 
                },
                "1"={#TROPICS
                  kc_ini=as.numeric(kc$kc_ini[1])
                  kc_mid=as.numeric(kc$kc_mid[1])
                  kc_end=as.numeric(kc$kc_end[1])
                  
                  
                  lgp_2_irr=round(lgp_irrigated[r,c]*as.numeric(kc$lgp_2[1]))
                  lgp_3_irr=round(lgp_irrigated[r,c]*as.numeric(kc$lgp_3[1]))
                  lgp_4_irr=round(lgp_irrigated[r,c]*as.numeric(kc$lgp_4[1]))
                  lgp_1_irr=lgp_irrigated[r,c]-lgp_2_irr-lgp_3_irr-lgp_4_irr
               
                },
                
                "2"={#Sub tropics-summer rainfall
                  
                  kc_ini=as.numeric(kc$kc_ini[2])
                  kc_mid=as.numeric(kc$kc_mid[2])
                  kc_end=as.numeric(kc$kc_end[2])
                  
                  lgp_2_irr=round(lgp_irrigated[r,c]*as.numeric(kc$lgp_2[2]))
                  lgp_3_irr=round(lgp_irrigated[r,c]*as.numeric(kc$lgp_3[2]))
                  lgp_4_irr=round(lgp_irrigated[r,c]*as.numeric(kc$lgp_4[2]))
                  lgp_1_irr=lgp_irrigated[r,c]-lgp_2_irr-lgp_3_irr-lgp_4_irr
                  
                  
                },
                
                "3"={#Sub tropics-winter rainfall (Mediterraneo)
                  kc_ini=as.numeric(kc$kc_ini[3])
                  kc_mid=as.numeric(kc$kc_mid[3])
                  kc_end=as.numeric(kc$kc_end[3])
                  
                    
                  lgp_2_irr=round(lgp_irrigated[r,c]*as.numeric(kc$lgp_2[3]))
                  lgp_3_irr=round(lgp_irrigated[r,c]*as.numeric(kc$lgp_3[3]))
                  lgp_4_irr=round(lgp_irrigated[r,c]*as.numeric(kc$lgp_4[3]))
                  lgp_1_irr=lgp_irrigated[r,c]-lgp_2_irr-lgp_3_irr-lgp_4_irr
                  
                },
                
                "4"={#Temperate (oceanic)
                  kc_ini=as.numeric(kc$kc_ini[4])
                  kc_mid=as.numeric(kc$kc_mid[4])
                  kc_end=as.numeric(kc$kc_end[4])
                  
                  
                  lgp_2_irr=round(lgp_irrigated[r,c]*as.numeric(kc$lgp_2[4]))
                  lgp_3_irr=round(lgp_irrigated[r,c]*as.numeric(kc$lgp_3[4]))
                  lgp_4_irr=round(lgp_irrigated[r,c]*as.numeric(kc$lgp_4[4]))
                  lgp_1_irr=lgp_irrigated[r,c]-lgp_2_irr-lgp_3_irr-lgp_4_irr

                },
                
                "5"={ #Temperate (sub-continental
                  kc_ini=as.numeric(kc$kc_ini[5])
                  kc_mid=as.numeric(kc$kc_mid[5])
                  kc_end=as.numeric(kc$kc_end[5])
                 
                  lgp_2_irr=round(lgp_irrigated[r,c]*as.numeric(kc$lgp_2[5]))
                  lgp_3_irr=round(lgp_irrigated[r,c]*as.numeric(kc$lgp_3[5]))
                  lgp_4_irr=round(lgp_irrigated[r,c]*as.numeric(kc$lgp_4[5]))
                  lgp_1_irr=lgp_irrigated[r,c]-lgp_2_irr-lgp_3_irr-lgp_4_irr
                 
                },
                
                
                "6"={#Temperate (continenta)
                  kc_ini=as.numeric(kc$kc_ini[6])
                  kc_mid=as.numeric(kc$kc_mid[6])
                  kc_end=as.numeric(kc$kc_end[6])
                  
                  
                  lgp_2_irr=round(lgp_irrigated[r,c]*as.numeric(kc$lgp_2[6]))
                  lgp_3_irr=round(lgp_irrigated[r,c]*as.numeric(kc$lgp_3[6]))
                  lgp_4_irr=round(lgp_irrigated[r,c]*as.numeric(kc$lgp_4[6]))
                  lgp_1_irr=lgp_irrigated[r,c]-lgp_2_irr-lgp_3_irr-lgp_4_irr

                },
                
                "7"={ #Boreal (oceanic)
                  kc_ini=as.numeric(kc$kc_ini[7])
                  kc_mid=as.numeric(kc$kc_mid[7])
                  kc_end=as.numeric(kc$kc_end[7])
                  
                   
                  lgp_2_irr=round(lgp_irrigated[r,c]*as.numeric(kc$lgp_2[7]))
                  lgp_3_irr=round(lgp_irrigated[r,c]*as.numeric(kc$lgp_3[7]))
                  lgp_4_irr=round(lgp_irrigated[r,c]*as.numeric(kc$lgp_4[7]))
                  lgp_1_irr=lgp_irrigated[r,c]-lgp_2_irr-lgp_3_irr-lgp_4_irr

                },
                
                "8"={ #Boreal (sub-continental)
                  kc_ini=as.numeric(kc$kc_ini[8])
                  kc_mid=as.numeric(kc$kc_mid[8])
                  kc_end=as.numeric(kc$kc_end[8])
                  
                  
                  lgp_2_irr=round(lgp_irrigated[r,c]*as.numeric(kc$lgp_2[8]))
                  lgp_3_irr=round(lgp_irrigated[r,c]*as.numeric(kc$lgp_3[8]))
                  lgp_4_irr=round(lgp_irrigated[r,c]*as.numeric(kc$lgp_4[8]))
                  lgp_1_irr=lgp_irrigated[r,c]-lgp_2_irr-lgp_3_irr-lgp_4_irr

                },
                
                "9"={#Boreal (continental)
                  kc_ini=as.numeric(kc$kc_ini[9])
                  kc_mid=as.numeric(kc$kc_mid[9])
                  kc_end=as.numeric(kc$kc_end[9])
                  
                  lgp_2_irr=round(lgp_irrigated[r,c]*as.numeric(kc$lgp_2[9]))
                  lgp_3_irr=round(lgp_irrigated[r,c]*as.numeric(kc$lgp_3[9]))
                  lgp_4_irr=round(lgp_irrigated[r,c]*as.numeric(kc$lgp_4[9]))
                  lgp_1_irr=lgp_irrigated[r,c]-lgp_2_irr-lgp_3_irr-lgp_4_irr
                  
                },
                
                "10"={# ARCTIC
                  kc_ini=as.numeric(kc$kc_ini[10])
                  kc_mid=as.numeric(kc$kc_mid[10])
                  kc_end=as.numeric(kc$kc_end[10])
                  
                
                  lgp_2_irr=round(lgp_irrigated[r,c]*as.numeric(kc$lgp_2[10]))
                  lgp_3_irr=round(lgp_irrigated[r,c]*as.numeric(kc$lgp_3[10]))
                  lgp_4_irr=round(lgp_irrigated[r,c]*as.numeric(kc$lgp_4[10]))
                  lgp_1_irr=lgp_irrigated[r,c]-lgp_2_irr-lgp_3_irr-lgp_4_irr
                  
                }
        )
        ###############     
        
    
        #STEP 1: INITIALISATION OF DAILY DATA
        #monthly data (ET0) are disaggregated into daily data. TOOL:interpolation 
        #to use the interpolation method also at the beginning and end of the year, we use December values and January values respectively
        
        data_daily=data.frame(matrix(ncol=17,nrow=length(new_calendar), dimnames=list(NULL, c("new_calendar","day_number","ET0_daily","ETa","kc","ks_e","runoff","leakage","I","WC_C","WC_FC","WC_SAT","WC_m","WC_e", "PR", "PR_canopy","PR_eff"))))
        data_daily$new_calendar=new_calendar
        
        s=which(tot_matrix_ET[r,c,]<0)#checks: if ET0<0 --> ET0==0
        tot_matrix_ET[r,c,s]=0
        
        #ET0 at daily resolution: interpolation on the interpolation_dates.
        data_daily$ET0_daily[which((data_daily$new_calendar %in% central_dates2)==TRUE)]=tot_matrix_ET[r,c,] #in correspondence to central dates: save available values
        dummy=approx(which((data_daily$new_calendar %in% central_dates2)==TRUE), tot_matrix_ET[r,c,],which((data_daily$new_calendar %in% central_dates2)==FALSE)) #obtain interpolation values on the other dates
        data_daily$ET0_daily[which((data_daily$new_calendar %in% central_dates2)==FALSE)]=dummy$y #merge already-available-values and newly interpolated values
        
        #CUT: cut of the dataframe to consider only the year 2010
        starting_date=as.Date("01-01-2010" , format="%d-%m-%Y")
        ending_date=as.Date("31-12-2010" , format="%d-%m-%Y")
        just_save=data_daily
        data_daily=data_daily[which(data_daily$new_calendar==starting_date):which(data_daily$new_calendar==ending_date),]
        data_daily=rbind(data_daily,data_daily)
        data_daily$day_number=c(seq(from=1, to=365, by=1),seq(from=1, to=365, by=1))
        
        
        
        #STEP 2: INITIALISATION OF HOURLY DATASET
        #Data at hourly resolution
        data_hourly=data.frame(matrix(ncol=11,nrow=365*2*24, dimnames=list(NULL, c("day-hour","PR","PR_canopy","PR_eff", "runoff","leakage", "WC","ks","ET0", "ETa", "B_hourly"))))
        data_hourly$day.hour=c(rep(seq(from=1, to=365, by=1), each=24),rep(seq(from=1, to=365, by=1), each=24))
        
        #Hourly PR
        if(length(in_quali_celle_Grandi[[m_index[r,c]]]$cella.grande)==1){ #case for which model-cell falls within only 1 ERA5 cell
            w=which(M_index==in_quali_celle_Grandi[[m_index[r,c]]]$cella.grande[1], arr.ind = TRUE)
            
            pr_1=as.numeric(PR_1[w[1], w[2],]) #precipitation of first year
            pr_1[pr_1<0.00000001]=0
      
            pr_2=as.numeric(PR_2[w[1], w[2],]) #precipitation of second year
            pr_2[pr_2<0.00000001]=0
            
            
            data_hourly$PR=c(pr_1[1:8760], pr_2[1:8760]) #rough way to avoid leap-year problems
          }else{
            dummy=c()  #case: model cell falls within multiple ERA5 cells
            for(x in 1:length(in_quali_celle_Grandi[[m_index[r,c]]]$cella.grande)){
              w=which(M_index==in_quali_celle_Grandi[[m_index[r,c]]]$cella.grande[x], arr.ind = TRUE)
              pr_1=as.numeric(PR_1[w[1], w[2],])*as.numeric(in_quali_celle_Grandi[[m_index[r,c]]]$PERC[x]) #weighted precipitation of first year
              pr_1[pr_1<0.00000001]=0
              pr_2=as.numeric(PR_2[w[1], w[2],])*as.numeric(in_quali_celle_Grandi[[m_index[r,c]]]$PERC[x]) #weighted precipitation of second year
              pr_2[pr_2<0.00000001]=0
              
              if(x==1){
                dummy=c(pr_1[1:8760], pr_2[1:8760])
              }else{
                dummy=dummy+c(pr_1[1:8760], pr_2[1:8760])
              }
            }
            data_hourly$PR=dummy
          }
        
        
        #Canopy Interception of rain
        if(plant_type=="grass"){
          threshold_pr=0.5  #intercepted rain in case of grass crop [mm/h]
        }else{
          threshold_pr=2 #intercepted rain in case of tree crop [mm/h]
        }
        GAPS_matrix=gap_detection(data_hourly$PR, 0.01, 5) #(array_name, threshold_pr (mm to start considering precipitation a rain event), threshold_hours (minimum amount of hours between events))
        data_hourly$PR_canopy=canopy_interception(GAPS_matrix, threshold_pr, data_hourly$PR) #(GAPS_matrix, threshold_pr, PR_array)


        
        #STEP 3: COMPUTATION - BLUE WATER
        
          #DEFINITION OF GROWING PERIOD LENGTH
          lgp_irr=lgp_1_irr+lgp_2_irr+lgp_3_irr+lgp_4_irr #length of growing period in case irrigated
          
          #array index of the day vs. Growing period
          day_start=data_semina_irr[r,c]
          if((day_start+lgp_irr)<=365){
            gp_index=c(seq(day_start,(day_start+(lgp_irr-1))))
          }else{
            p2=(lgp_irr-1)-(365-day_start)
            gp_index=c(seq(day_start,365), seq(1,p2))
          }
          
          #CUT daily data during growing period
          w=which(data_daily$day_number==day_start)[1]
          data_daily=data_daily[w:((w-1)+lgp_irr),]
          rownames(data_daily) <- NULL
          
          #CUT Hourly data during Growing period
          w=which(data_hourly$day.hour==day_start)[1]
          data_hourly=data_hourly[w:((w-1)+lgp_irr*24),]
          rownames(data_hourly) <- NULL

          

          #DEFINITION OF HOURLY ET0 according to location and day of the year 
          coordinates=c()
          coordinates[1]=xyFromCell(ita,m_index[r,c])[1]#longitude(E)
          coordinates[2]=xyFromCell(ita,m_index[r,c])[2]#latitude(N)
          
          sowing_date=as.Date(day_start,format="%d-%m-%Y", origin=starting_date)
          harvesting_date=sowing_date+lgp_irr-1
          calendar=seq(as.Date(sowing_date,format="%d-%m-%Y"),as.Date(harvesting_date,format="%d-%m-%Y"),"day")
          
          SunSetSunRise=getSunlightTimes(date=calendar, lat=coordinates[2], lon=coordinates[1],keep = c("sunrise", "sunset"), tz = "UTC") #!!!! WATCH OUT: it's at UTC, Italy is at UTC+1=CET

          SunSetSunRise$sunrise=round_date(SunSetSunRise$sunrise, unit="hour")
          SunSetSunRise$sunset=round_date(SunSetSunRise$sunset, unit="hour")
          
          sr=c()
          ss=c()
          for(d in 1:length(SunSetSunRise$date)){
            sr[d]=sub(":00:00","",sub(" ","",sub(as.character(SunSetSunRise$date[d]),"",as.character(SunSetSunRise$sunrise[d]))))
            ss[d]=sub(":00:00","",sub(" ","",sub(as.character(SunSetSunRise$date[d]),"",as.character(SunSetSunRise$sunset[d]))))
          }
          
          SunSetSunRise$sunrise=as.numeric(sr)+1 #to have it at CET 
          SunSetSunRise$sunset=as.numeric(ss)+1 #to have it at CET
          
          N=SunSetSunRise$sunset- SunSetSunRise$sunrise #hours for sunlight
          SunSetSunRise=cbind.data.frame(SunSetSunRise,N)
          
          
          for(j in 1:lgp_irr){
            h=seq(from=0, to=(N[j]-1), by=1) #equivalent in radiant of 24 hours
            B=function(x){sin(pi*(x)/N[j])} #aprroximation of ETa during the day with a sine function
            B_int=integrate(B, lower=0, upper=N[j])
            A=data_daily$ET0_daily[j]/B_int$value
            
            
            dummy_ET0_hourly=rep(0,24) #ET0 from sunset to sunrise is 0
            dummy_ET0_hourly[SunSetSunRise$sunrise[j]:(SunSetSunRise$sunset[j]-1)]= A*{sin(pi*(h)/N[j])}#hourly ETO trend for 1 day
            
            print(paste0("day: ", j))
            print(paste0("daily: ",data_daily$ET0_daily[j]))
            print(paste0("daily: ",sum(dummy_ET0_hourly)))
            
            if(j==1){
              dummy_ET0=dummy_ET0_hourly
            }else{
              dummy_ET0=c(dummy_ET0,dummy_ET0_hourly)
            }
          }
          data_hourly$ET0=dummy_ET0
          
          
          
    
          #DEFINITION OF kc
          #daily Kc
          data_daily$kc[1:lgp_1_irr]=kc_ini
          
          for (g in (lgp_1_irr+1):(lgp_1_irr+lgp_2_irr)){
            data_daily$kc[g]=(kc_mid-kc_ini)/lgp_2_irr*(g-lgp_1_irr)+kc_ini
          }
          data_daily$kc[(lgp_2_irr+lgp_1_irr+1):(lgp_2_irr+lgp_1_irr+lgp_3_irr)]=kc_mid
          
          for (g in (lgp_2_irr+lgp_1_irr+1+lgp_3_irr):(lgp_irr)){
            data_daily$kc[g]=(kc_end-kc_mid)/lgp_4_irr*(g-lgp_3_irr-lgp_2_irr-lgp_1_irr)+kc_mid
          }
          

          #DEFINITION OF ROOTHING DEPTH & TAWC & RAWC 
          rd_irr=c()
          rd_irr[1]=rd_ini
          
          for (g in 2:(lgp_1_irr+lgp_2_irr)){ #roots elongation
            rd_irr[g]=rd_ini+(rd_max_irrigated-rd_ini)/(lgp_1_irr+lgp_2_irr)*g
          }
          rd_irr[(lgp_1_irr+lgp_2_irr+1):lgp_irr]=rd_max_irrigated
          
          
          tawc_irr=awc[r,c]*rd_irr #water content at Wilting point: varies with Zn elongation [mm(water)/m(soildepth)] 

          f_irr=rep(depl_fraction, lgp_irr) #depletion fraction. constant during the whole growing period
          
          rawc_irr=tawc_irr*f_irr #Readily available water content: varies with Zn elongation
          
          
          #computation of critical Water Content: WC*
          class_soil=soil[r,c]
          ks_soil=pedo$Ks[which(pedo$`USDA code`==class_soil)]*10/(24) #from cm/day to mm/hour so that integrating over the hour, leakage is mm/hour
          b=pedo$b[which(pedo$`USDA code`==class_soil)]
          beta=2*b+4
          teta_fc=pedo$teta_fc_mean[which(pedo$`USDA code`==class_soil)]
          teta_sat=pedo$teta_sat[which(pedo$`USDA code`==class_soil)]
          
          teta_c=(teta_fc-(rawc_irr/(1000*rd_irr)))[1] #[1]: it is sufficient to keep the first value as the array contains the same value
          sfc=teta_fc/teta_sat
          
          data_daily$WC_C=1000*rd_irr*teta_fc-rawc_irr #critical water content
          data_daily$WC_FC=1000*rd_irr*teta_fc #water content at field capacity
          data_daily$WC_SAT=1000*rd_irr*teta_sat #water content at saturation
          
          
          
          

          
          #STEP 3A:COMPUTATIONS DURING GROWING PERIOD
          L_f=makeFun((ks_soil/(exp(beta*(1-sfc))-1))*(exp(beta*(s-sfc))-1) ~ s) #Leakage function
          
          for (i in 1:lgp_irr){  #daily computation during the growing period
            
            #initialisation of morning-WC
            if(i==1){
              data_daily$WC_m[1]=data_daily$WC_C[1] #we assume that we start the first day with a WC=WC critical
            }else{
              data_daily$WC_m[i]=data_daily$WC_e[i-1]+data_daily$I[i-1]+data_daily$WC_FC[i]-data_daily$WC_FC[i-1] #every day, the last hour, the crop is irrigated: WC_m=WC_C
            }
            
            
            #STEP 3B: HOURLY COMPUTATION
            #initialise dummy-vectors for hourly leakage computation
            WC_0=c()  #water content at minute 0 of the hour
            WC_60=c()  #water content at minute 60 of the hour
            teta_0=c() #teta at minute 0 of the hour
            s_0=c() #relative soil moisture at minute 0 of the hour
            teta_60=c() #teta at minute 60 of the hour
            s_60=c() #relative soil moisture at minute 60 of the hour
            ETa_hourly=c() #hourly actual evapotranspiration
            L_hourly=c() #hourly leakage
            runoff_hourly=c() #hourly runoff
            PR_eff_hourly=c() #hourly effective precipitation
            ks_hourly=c() #hourly stress coefficient
            B_hourly=rep(0,24) #hourly Blue water (0 every hour but the last)
            
            dummy_PR_canopy=data_hourly$PR_canopy[data_hourly$day.hour==gp_index[i]] #Precipitation of the day
            dummy_ET0=data_hourly$ET0[data_hourly$day.hour==gp_index[i]] #Potential evapotranspiration
            
            
            #hourly computation           
            for(h in 1:24){ #from 00am to 12pm
    
              if(h==1){
                WC_0[h]=data_daily$WC_m[i] #beginning of the first hour of the day coincide with the WC computed for the morning-previous evening
              }else{
                WC_0[h]=WC_60[h-1] #every other hour, the beginning is the end of the previous hour
              }
              teta_0[h]=(WC_0[h]/(rd_irr[i]*1000)) #water content wrt teta=0
              s_0[h]=teta_0[h]/teta_sat #relative water content
              
              
              #Computation of hourly ETa starting from the computation of hourly ks
              RAWC=1000*rd_irr[i]*teta_fc- WC_0[h] #RAWC[mm/hour], WC[mm/hour], teta [mm/mm]
              ks_hourly[h]=(tawc_irr[i]-RAWC)/(tawc_irr[i]-rawc_irr[i])
              
              if(ks_hourly[h]>1){
               ks_hourly[h]=1
              }
              
              ETa_hourly[h]=dummy_ET0[h]*data_daily$kc[i]*ks_hourly[h] #computation of actual evapotranspiration
              if(dummy_PR_canopy[h]>0){
                ETa_hourly[h]=0 #if it rains, the evapotranpiration is shut down due to humidity
              }
              
              
              #to account for possible runoff: if the hourly precipitation is larger than the available WC, then runoff is formed
              if(data_daily$WC_SAT[i]>WC_0[h]){#terrain is not in saturated conditions
                if((data_daily$WC_SAT[i]-WC_0[h])>(dummy_PR_canopy[h])){ #available free space is larger than precipitation
                  PR_eff_hourly[h]=dummy_PR_canopy[h] #effective precipitation= hourly precipitation=daily precipitation equally divided along the hours
                  runoff_hourly[h]=0
                  
                }else{
                  PR_eff_hourly[h]=data_daily$WC_SAT[i]-WC_0[h] #available free space is smaller than precipitation -> effective precipitation=free space
                  runoff_hourly[h]=(dummy_PR_canopy[h])-PR_eff_hourly[h]
                }
              }else{#terrain is in saturated conditions
                PR_eff_hourly[h]=0 #no precipitation can be stored
                runoff_hourly[h]=dummy_PR_canopy[h]
              }
              
              if(s_0[h]>sfc){
                L_hourly[h]=L_f(s_0[h])
              }else{
                L_hourly[h]=0 #there's no leakage if WC is below WC_FC
              }
              
              
              WC_60[h]=WC_0[h]+PR_eff_hourly[h]-ETa_hourly[h]-L_hourly[h] #first update of soil WC at minute 60 of the hour
              teta_60[h]=(WC_60[h]/(rd_irr[i]*1000)) #update of teta
              s_60[h]=teta_60[h]/teta_sat
              
              if(B_l==1){
                daily_limit=data_daily$WC_C[i]
              }else{
                daily_limit=data_daily$WC_FC[i]
              }
              
              
              if(h==24 & WC_60[h]<daily_limit){
                  B_hourly[h]=daily_limit-WC_60[h]
              }
              
            }# end of loop: hours (h)
            
            data_hourly$PR_eff[data_hourly$day.hour==gp_index[i]]=PR_eff_hourly
            data_hourly$runoff[data_hourly$day.hour==gp_index[i]]=runoff_hourly
            data_hourly$leakage[data_hourly$day.hour==gp_index[i]]=L_hourly
            data_hourly$WC[data_hourly$day.hour==gp_index[i]]=WC_60
            data_hourly$ks[data_hourly$day.hour==gp_index[i]]=ks_hourly
            data_hourly$ETa[data_hourly$day.hour==gp_index[i]]=ETa_hourly
            data_hourly$B_hourly[data_hourly$day.hour==gp_index[i]]=B_hourly
            
            data_daily$PR[i]=sum(data_hourly$PR[data_hourly$day.hour==gp_index[i]])
            data_daily$PR_canopy[i]=sum(dummy_PR_canopy)
            data_daily$WC_e[i]=WC_60[24] #salvo il dato di WC serale prima dell'irrigazione
            data_daily$PR_eff[i]=sum(PR_eff_hourly)
            data_daily$runoff[i]=sum(runoff_hourly)
            data_daily$leakage[i]=sum(L_hourly)
            data_daily$ks_e[i]=ks_hourly[24]
            data_daily$ETa[i]=sum(ETa_hourly)
            data_daily$I[i]=sum(B_hourly)
            
            
            
          }# end of loop: growing period loop (i)
          
          
          
          ETgreen_lgp_irr[r,c]=sum(data_daily$PR_eff)     
          B_lgp[r,c]=sum(data_daily$I)
          ETa_lgp_irr[r,c]=sum(data_daily$ETa)  
          PR_lgp_irr[r,c]=sum(data_daily$PR)
          R_lgp_irr[r,c]=sum(data_daily$runoff)
          L_lgp_irr[r,c]=sum(data_daily$leakage)
          
        }# end of condition if: irrigated area
    }#end of loop: columns (c)
  }# end of loop: rows (r)

  
  #rasterization of results
  ETgreen_lgp_irr_r=raster_as(ETgreen_lgp_irr, ita)
  B_lgp_r=raster_as(B_lgp,ita)
  ETa_lgp_irr_r=raster_as(ETa_lgp_irr, ita)
  PR_lgp_irr_r=raster_as(PR_lgp_irr, ita)
  L_lgp_irr_r=raster_as(L_lgp_irr,ita)
  R_lgp_irr_r=raster_as(R_lgp_irr,ita)
  
  
  writeRaster(ETgreen_lgp_irr_r, paste0(saving_location,year_1,"-",year_2,"/", code,"_B",B_l,"_ETgreen_",year_1,"-",year_2,"_lgp_irr_WCv2.tif"))
  print()
  writeRaster(B_lgp_r, paste0(saving_location,year_1,"-",year_2,"/", code,"_B",B_l,"_B_",year_1,"-",year_2,"_lgp_irr_WCv2.tif"))
  print()
  writeRaster(ETa_lgp_irr_r, paste0(saving_location,year_1,"-",year_2,"/", code,"_B",B_l,"_ETa_",year_1,"-",year_2,"_lgp_irr_WCv2.tif"))
  print()
  writeRaster(PR_lgp_irr_r, paste0(saving_location,year_1,"-",year_2,"/", code,"_B",B_l,"_PR_",year_1,"-",year_2,"_lgp_irr_WCv2.tif"))
  print()
  writeRaster(L_lgp_irr_r, paste0(saving_location,year_1,"-",year_2,"/", code,"_B",B_l,"_L_",year_1,"-",year_2,"_lgp_irr_WCv2.tif"))
  print()
  writeRaster(R_lgp_irr_r, paste0(saving_location,year_1,"-",year_2,"/", code,"_B",B_l,"_R_",year_1,"-",year_2,"_lgp_irr_WCv2.tif"))
  print()
  
  
  
  
  #STEP 4: IRRIGATION COMPUTATION
  print("Working on B2I")
  
  
  #### UPLOAD IRRIGATION DATA ###########
  load(paste0(input_location,"irrigation/irrigation_system_per_cell_ha.RData")) #irr_system_per_cell: hectars irrigated per cell with different irrigation systems
  
  blue_map_r=raster(paste0(saving_location,year_1,"-",year_2,"/",code,"_B",B_l,"_B_",year_1,"-",year_2,"_lgp_irr_WCv2.tif"))
  blue_map=as.matrix(blue_map_r)
  efficienze=c(0.55,0.25,0.75,0.90, 0.7)#scorriemnto&infiltrazione, #sommersione, #aspersione,#microirrigazione. #altro
  
  
  {switch_crop=p}
  switch (switch_crop,  #list of irrigation systems used for each cultivation
          
          "1"={ #Maiz
            
            possible_sys=c("scorrimento superficiale ed infiltrazione laterale","aspersione (a pioggia)","microirrigazione")
            
          },
          
          "2"={#rice
            
            possible_sys=c("sommersione")
            
          },
          
          "3"={ #olive
            
            possible_sys="microirrigazione"
            
          },
          
          "4"={ #vine
            
            possible_sys="microirrigazione"
            
          },
          
          "5"={#barley
            
            possible_sys=c("aspersione (a pioggia)", "microirrigazione")
            
          },
          
          "6"={ #sorghum
            
            possible_sys=c("asperione (a pioggia)","microirrigazione")
            
          },
          
          "7"={ #Wheat
            
            possible_sys=c("aspersione (a pioggia)", "microirrigazione")
            
          },
          
  )
  
  types=cbind.data.frame(names(irr_systems_per_cell[[203]][1:5]),efficienze,seq(1:length(efficienze))) #get irrigation system names
  colnames(types)=c("irrigation_system","efficiency","index")
  
  
  #MEAN IRRIGATION VOLUMES
  mean_irr=matrix(NaN, 139,142)
  dummy2=irr_systems_per_cell
  
  for(r in 1:dim(m_index)[1]){
    for(c in 1:dim(m_index)[2]){
      t=m_index[r,c] 
      dummy2[[t]][1:6]=0
      
      if(length(irr_systems_per_cell[[t]])>0 & !is.na(blue_map[r,c])){ #takes into account only the irrigated areas for the crop
        
        k=which(irr_systems_per_cell[[t]][1:4]>0)#5=other system, 6=total : 
        n=names(irr_systems_per_cell[[t]])[k]#just to get names of available systems
        
        f=intersect(possible_sys,n) #intersection between possible irrigation systems and present irrigation systems
        
        if(length(f)==0){#there's no possible&present irrigation system 
          
          dummy2[[t]][5]=1/efficienze[5] #attribute irrigation to "other system" 
        }
        if(length(f)==1){#there's only 1 possible&present irrigation system
          l=types$index[which(types$irrigation_system %in% f)]
          dummy2[[t]][l]=1/efficienze[l]
          
        }
        if(length(f)>1){#there are several possible&present irrigation systems and so the partitioning of the irrigation has to be doen among these systems 
          l=types$index[which(types$irrigation_system %in% f)]
          dummy2[[t]][c(l)]=irr_systems_per_cell[[t]][c(l)]
          dummy2[[t]][6]=sum(dummy2[[t]])
          
          for(g in 1:length(l)){
            dummy2[[t]][l[g]]=(irr_systems_per_cell[[t]][l[g]]/(efficienze[l[g]]*dummy2[[t]][6]))# % of irrigation per system*  required mm of water/ efficiency of irrigation system
          }
          dummy2[[t]][6]=0
          
        }
        
        mean_irr[r,c]= sum(dummy2[[t]])*blue_map[r,c]
        
        
      }else{ #celle NULL (no irrigation cells)
        next
      }
      
    }#end of loop: columns (c) 
  }#end of loop: rows (s)
  
  if(from_B_to_I=="2" | from_B_to_I=="3"){
    
    units="m3"
    #from mm to m3
    mean_irr_m3=mean_irr*as.matrix(area_irr)*10 
    mean_irr_m3_r=raster_as(mean_irr_m3, ita)
    
    
    writeRaster(mean_irr_m3_r, paste0(saving_location,year_1,"-",year_2,"/", code,"_B",B_l,"_I_",units,"_",year_1,"-",year_2,"_lgp_irr_WCv2.tif"))
    print(paste0("printing: ", code,"_B",B_l,"_I_",units,"_",year_1,"-",year_2,"_lgp_irr_WCv2.tiff"))
    

  }
  if(from_B_to_I=="1" | from_B_to_I=="3"){
    
    units="mm"
    mean_irr_r=raster_as(mean_irr, ita)
    
    
    writeRaster(mean_irr_m3_r, paste0(saving_location,year_1,"-",year_2,"/", code,"_B",B_l,"_I_",units,"_",year_1,"-",year_2,"_lgp_irr_WCv2.tif"))
    print(paste0("printing: ", code,"_B",B_l,"_I_",units,"_",year_1,"-",year_2,"_lgp_irr_WCv2.tiff"))
    
  }
  
}#end of loop: years (y)








