
'''
title: "Effects of environmental and individual
variation on time patterns of extinction in
small experimental populations -  Branching process functions"
authors:
  - name: Souleyman Bakker (corresponding author)
affiliation: Institut d’Écologie et des Sciences de l’Environnement de
Paris (iEES-Paris), Sorbonne Université, CNRS, INRAe, IRD, Université
Paris Créteil, Université Paris cité, 75005, Paris, France
email: souleyman1@hotmail.fr
- name: Tom J.M. Van Dooren
affiliation: Institut d’Écologie et des Sciences de l’Environnement de
Paris (iEES-Paris), Sorbonne Université, CNRS, INRAe, IRD, Université
Paris Créteil, Université Paris cité, 75005, Paris, France
email: tvdooren@gmail.com
- name: Thomas Tully
affiliation: Institut d’Écologie et des Sciences de l’Environnement de
Paris (iEES-Paris), Sorbonne Université, CNRS, INRAe, IRD, Université
Paris Créteil, Université Paris cité, 75005, Paris, France
email: thomas.tully@sorbonne-universite.fr

This file repeats to a large extent code described in Van Dooren et al.
(2024) of which a CC BY-NC-SA copyright lies with these authors.

Van Dooren, T. J. M., Haccou, P., Hermus, G., Tully, F. T. (2024)
Extinction probabilities of small structured populations: adequate
short-term model predictions in Folsomia candida.
https://doi.org/10.1101/2024.08.26.609669

'''

# Establishment probabilities of small populations: 
# an age-structured branching process model and a test using Folsomia candida
##############################################################################

# the code allows for reproduction in any age class
# the code allows for populations founded by several individuals of the same age class. Not yet for founders in different age classes.


# Matrix Model
###############

# write the matrix representation of the demography

meanmatrix<-function(list){
  gammac<-list$gamma+list$delta*(1-list$gamma)
  matrix(list$gamma*c(rep(0,6),list$s1,rep(0,6),list$s2,rep(0,6),list$s3,rep(0,6),list$s4,rep(0,6),list$s5,list$s6),nrow=6,byrow=T)+
    matrix(gammac*c(list$f1,list$f2,list$f3,list$f4,list$f5,list$f6,
                    rep(0,30)),nrow=6,byrow=T)
}

# calculate matrix properties

# dominant eigenvalue - growth rate
#----------------------------------

growthrate<-function(matrix)Re(eigen(matrix)$values[1]) # real part of largest eigenvalue, function eigen gives the full spectrum

# stable age distribution
#------------------------

agedist<-function(matrix){vector<-Re(eigen(matrix)$vectors[,1]);vector/sum(vector)} 

# reproductive values
#-_------------------

repvals<-function(matrix){tmat<-t(matrix);vector<-Re(eigen(tmat)$vectors[,1]);vector/sum(agedist(matrix)*vector)} 

# sensitivities to matrix elements
#---------------------------------

growthratesensitivities<-function(matrix){rv<-repvals(matrix);ad<-agedist(matrix);return(rv%*%t(ad))}

# Branching Process Calculations
################################

# argument "list" is a list with parameter names and values

# ultimate extinction risk, function ultextinct
#----------------------------------------------


ultextinct<-function(list){ # argument: list with parameter names and values
  extinctvec<-numeric(6)
  
  # extra variables to reduce length of equations below
  ef<-function(x,y)exp(-y*(1-x))
  c1<-list$gamma*list$s1
  c2<-list$gamma*list$s2
  c3<-list$gamma*list$s3
  c4<-list$gamma*list$s4
  c5<-list$gamma*list$s5
  c6<-list$gamma*list$s6
  gammap<-list$gamma+list$delta*(1-list$gamma)
  f1<-list$f1/list$p1
  f2<-list$f2/list$p2  
  f3<-list$f3/list$p3  
  f4<-list$f4/list$p4  
  f5<-list$f5/list$p5  
  f6<-list$f6/list$p6    
  CQ6<-function(x1){gfact6<-(1-gammap*list$p6+gammap*list$p6*ef(x1,f6));(1-c6)*gfact6/(1-c6*gfact6)}
  CQ5<-function(x1){gfact5<-(1-gammap*list$p5+gammap*list$p5*ef(x1,f5));(1-c5+c5*CQ6(x1))*gfact5}
  CQ4<-function(x1){gfact4<-(1-gammap*list$p4+gammap*list$p4*ef(x1,f4));(1-c4+c4*CQ5(x1))*gfact4}
  CQ3<-function(x1){gfact3<-(1-gammap*list$p3+gammap*list$p3*ef(x1,f3));(1-c3+c3*CQ4(x1))*gfact3}
  CQ2<-function(x1){gfact2<-(1-gammap*list$p2+gammap*list$p2*ef(x1,f2));(1-c2+c2*CQ3(x1))*gfact2}
  CQ1<-function(x1){gfact1<-(1-gammap*list$p1+gammap*list$p1*ef(x1,f1));(1-c1+c1*CQ2(x1))*gfact1}
  
  
  
  # different root finding algorithms tried, optimize is used
  
  #ROOTER<-function(x1)CQ1(x1)-x1
  ROOTER2<-function(x1)(CQ1(x1)-x1)^2
  root<-optimize(ROOTER2,lower=0,upper=1)
  #root<-nlminb(0,ROOTER2)
  
  #root<-nlm(ROOTER2,0)
  
  #xx1<-root$par
  xx1<-root$minimum
  
  extinctvec[6]<-CQ6(xx1)
  extinctvec[5]<-CQ5(xx1)
  extinctvec[4]<-CQ4(xx1)
  extinctvec[3]<-CQ3(xx1)
  extinctvec[2]<-CQ2(xx1)
  extinctvec[1]<-CQ1(xx1)
  return(list(root=root,ultextinctvec=extinctvec))
  
}

# Calculation of Q(t)
#----------------------  

QTot<-function(list){
  Qarray<-array(0,c(list$T+1,6))
  
  ef<-function(x,y)exp(-y*(1-x))
  c1<-list$gamma*list$s1
  c2<-list$gamma*list$s2
  c3<-list$gamma*list$s3
  c4<-list$gamma*list$s4
  c5<-list$gamma*list$s5
  c6<-list$gamma*list$s6
  gammap<-list$gamma+list$delta*(1-list$gamma)
  f1<-list$f1/list$p1
  f2<-list$f2/list$p2  
  f3<-list$f3/list$p3  
  f4<-list$f4/list$p4  
  f5<-list$f5/list$p5  
  f6<-list$f6/list$p6    
  
  for(i in 2:(list$T+1)){
    (1-c1+c1*Qarray[i-1,2])*(1-list$gamma+list$gamma*ef(Qarray[i-1,1],list$f1))
    Qarray[i,1]<-(1-gammap*list$p1+gammap*list$p1*ef(Qarray[i-1,1],f1))*(c1*Qarray[i-1,2]-c1+1)
    Qarray[i,2]<-(1-gammap*list$p2+gammap*list$p2*ef(Qarray[i-1,1],f2))*(c2*Qarray[i-1,3]-c2+1)
    Qarray[i,3]<-(1-gammap*list$p3+gammap*list$p3*ef(Qarray[i-1,1],f3))*(c3*Qarray[i-1,4]-c3+1)
    Qarray[i,4]<-(1-gammap*list$p4+gammap*list$p4*ef(Qarray[i-1,1],f4))*(c4*Qarray[i-1,5]-c4+1)
    Qarray[i,5]<-(1-gammap*list$p5+gammap*list$p5*ef(Qarray[i-1,1],f5))*(c5*Qarray[i-1,6]-c5+1)
    Qarray[i,6]<-(1-gammap*list$p6+gammap*list$p6*ef(Qarray[i-1,1],f6))*(c6*Qarray[i-1,6]-c6+1)
  }
  return(Qarray)}



# Parrays() combines all results of the branching process approach
# allows for populations founded with several individuals of the same type
# ultextinct is removed and other unnecessary variables too.
#--------------------------------------------------------------------------

Parrays<-function(list){
  times<-c(0:list$T)
  ntimes<-length(times)
  pext<-array(0,c(ntimes,6)) # extinction probability per unit of time
  # probability of extinction within interval conditional on being at risk at end of previous interval
  psurv<-array(0,c(ntimes,6)) # survivorship functions
  
  
  QQ<-QTot(list) # cumulated extinction probability from day to day
  
  for(j in 1:6){
    psurv[1,j]<-1
    pext[1,j]<-0
    
    Z<-numeric(6)
    Z[j]<-list$initn    
    
    for(i in 2:ntimes){
      psurv[i,j]<-1-prod(QQ[i,]^Z) # survivorship functions
      pext[i,j]<-(QQ[i,j]-QQ[i-1,j])/(1-QQ[i-1,j]) # extinction probability per unit of time
      
    }
  }
  # the last is conditional time to extinction, given extinction before or at the n-th census
  
  return(list(parameters=list,times=times,Q=QQ,psurv=psurv,pext=pext))
}


parms<-list(s1 = 0.994,  s2 = 0.997,  s3 = 0.977, s4 = 0.943, s5 = 0.913, s6 = 0.887,  f1=0,  f2 = 0.174, f3 = 12.311, f4 = 52.505, f5 = 82.52, f6 = 96.619, gamma=1,  delta=0, p1 = 1, p2 = 1, p3 = 1, p4 = 1, p5 = 1, p6 = 1, T = 10,  initn = 1)


## Parameters value
parameters1<-list(
  s1 = 0.994,
  s2 = 0.997,
  s3 = 0.977,
  s4 = 0.943,
  s5 = 0.913,
  s6 = 0.887,
  f1 = 0.00,
  f2 = 0.,
  f3 = 12.31076,
  f4 = 52.50517,
  f5 = 82.5198,
  f6 = 96.61879,
  p1=1,
  p2=1,
  p3=1,
  p4=1,
  p5=1,
  p6=1,  
  gamma = 1,
  delta=0.0,
  T = 50,
  initn = 1)

parameters075<-list(
  s1 = 0.994,
  s2 = 0.997,
  s3 = 0.977,
  s4 = 0.943,
  s5 = 0.913,
  s6 = 0.887,
  f1 = 0.0,
  f2 = 0.,
  f3 = 12.31076,
  f4 = 52.50517,
  f5 = 82.5198,
  f6 = 96.61879,
  p1=1,
  p2=1,
  p3=1,
  p4=1,
  p5=1,
  p6=1, 
  gamma = .75,
  delta=0.0,
  T = 50,
  initn = 1)

parameters05<-list(
  s1 = 0.994,
  s2 = 0.997,
  s3 = 0.977,
  s4 = 0.943,
  s5 = 0.913,
  s6 = 0.887,
  f1 = 0.,
  f2 = 0.,
  f3 = 12.31076,
  f4 = 52.50517,
  f5 = 82.5198,
  f6 = 96.61879,
  p1=1,
  p2=1,
  p3=1,
  p4=1,
  p5=1,
  p6=1,
  gamma = .5,
  delta=0,
  T = 50,
  initn = 1)

parameters025<-list(
  s1 = 0.994,
  s2 = 0.997,
  s3 = 0.977,
  s4 = 0.943,
  s5 = 0.913,
  s6 = 0.887,
  f1 = 0.0,
  f2 = 0.0,
  f3 = 12.31076,
  f4 = 52.50517,
  f5 = 82.5198,
  f6 = 96.61879,
  p1=1,
  p2=1,
  p3=1,
  p4=1,
  p5=1,
  p6=1,
  gamma = .25,
  delta=0,
  T = 50,
  initn = 1)

# Sensitivities of extinction probabilities to fecundities and survival parameters and gamma
#############################################################################################

# the "scale" argument determines which parameter deviations will be simulated
sensitivitiesextinction<-function(list){
  parms<-unlist(list)[1:20]
  parmnames<-names(parms)
  np<-length(parms)
  # scales per parameter, of 1% of value
  scales<-parms/100
  scales<-ifelse(scales==0,0.001,scales)
  
  # prepare array 20 parameters times 5 points times 6 initial states
  valsplus<-array(0,c(np,5,6))
  valsmin<-array(0,c(np,5,6))
  # vary each parameter, add points to list
  for(i in 1:np){# make a change to one parameter, select which one
    parmname<-parmnames[i]
    for(j in 1:5){ # add the increment to this parameter
      parmtemp<-list
      parmtemp1<-parmtemp2<-parmtemp
      parmtemp1[[parmname]]<-parmtemp[[parmname]]+scales[i]*(j-1)
      parmtemp2[[parmname]]<-parmtemp[[parmname]]-scales[i]*(j-1)
      valsplus[i,j,]<-ultextinct(parmtemp1)$ultextinctvec
      valsmin[i,j,]<-ultextinct(parmtemp2)$ultextinctvec
    }
  }
  
  # regressions using lm
  slopevalsplus<-array(0,c(np,6),dimnames=list(parmnames,c(1:6)))
  slopevalsmin<-array(0,c(np,6),dimnames=list(parmnames,c(1:6)))
  for(i in 1:np){
    for(j in 1:6){
      slopevalsplus[i,j]<-coef(lm(valsplus[i,,j]~I(c(0:4)*scales[i])))[2]	
      slopevalsmin[i,j]<-coef(lm(valsmin[i,,j]~I(-c(0:4)*scales[i])))[2]	
    }
  }
  
  # convert array to data frame
  # add row and column names
  # return slopes as data frame
  senstableplus<-as.data.frame.table(slopevalsplus)
  senstablemin<-as.data.frame.table(slopevalsmin)
  names(senstableplus)<-c("Parameter","Stage","Sensitivity")
  names(senstablemin)<-c("Parameter","Stage","Sensitivity")
  return(list(senstableplus,senstablemin))
  
}

# Example

# extexample<-Parrays(parms)
# sensitivity1<-sensitivitiesextinction(parameters05)
# sensitivityplus<-sensitivity1[[1]]
# sensitivityminus<-sensitivity1[[2]]
# plot(Sensitivity~Parameter, data=sensitivityminus)
# plot(Sensitivity~Parameter, data=sensitivityplus)


