#' Shorten vector
#' Shorten a vector to a specified length adding 3 dots at the end if name is
#' longer than the specified length
shorten_vector <- function(x, length, suffix="...") {
  x <- as.character(x)
  sapply(x, function(y) {
  if (nchar(y) > length) {
    y <- substr(y, 1, length)
    y <- paste0(y, suffix)
  }
  y
  })
}


#'Refactor
refactor<- function(x){
  #keep the previous level order for x_cat 
  lvls <- levels(x)
  # Remove NA values from the factor and re-assign factor levels
  x <- x[!is.na(x)]
  
  #remove values from the levels that are not in x_cat but keep lvls order
  lvls <- lvls[lvls %in% unique(x)]
  factor(x, levels = lvls)
}

#' Filter out rare combinations of factors
filter_rare_combinations <- function(cdf, col, factors, min_obs = 3) {
  # filter out data that has less than min references
  len <- length(factors) 
  if(length(setdiff(factors, colnames(cdf)))!= 0) {
    missingCols <- setdiff(factors, colnames(cdf))
    stop(paste0("factors ", missingCols, " not found in cdf"))  
  }
  counts <- aggregate(cdf[[col]],
                          by=cdf[factors], 
                          FUN=function(x) length(unique(x)))
  counts <- counts[counts[,len+1] >= min_obs,]
  res <- merge(cdf, counts[factors], by=factors)
  
  res
}


# Get p-value stars
pval2Stars <- function(pval) {
  sapply(pval, function(p) {
    if(is.na(p)) return("")
    else if(p < 0.001) return("***")
    else if (p < 0.01) return("**")
    else if (p < 0.05) return("*")
    else return("")
  })
}



#' Format Tukey test results
print_tukey<- function(model, digits=2){
  cols2Keep <- c("contrast", "estimate", "conf.low", "conf.high", "adj.p.value", "stars")
  tukey <- broom::tidy(model)
  tukey$estimate <- round(tukey$estimate, digits)
  tukey$conf.low <- round(tukey$conf.low, digits)
  tukey$conf.high <- round(tukey$conf.high, digits)
  tukey$stars <- pval2Stars(tukey$adj.p.value)
  tukey$adj.p.value <- format.pval(tukey$adj.p.value, digits= digits, eps = 0.001)
  tukey[,cols2Keep]
}

#' Format lmerTest results
print_lmerTest <- function(model, digits=2, formula = TRUE){
  if(is.null(model)) return(NULL)
  
  df <- broom.mixed::tidy(as(model, "lmerMod"))
  ms <- summary(model)
  msc <- as.data.frame(ms$coefficients)
  msc$term <- rownames(msc)
  df <- merge(df, msc[c("term","Pr(>|t|)")], by=c("term"), all.x=TRUE)
  df$stars <- pval2Stars(df[["Pr(>|t|)"]])
  df[["Pr(>|t|)"]] <- format.pval(df[["Pr(>|t|)"]], digits= digits, eps = 0.001)  
  df$statistic <- round(df$statistic, digits)
  df$estimate <- round(df$estimate, digits)
  df$effect <- NULL
  df$std.error <- round(df$std.error, digits)
  
  #replace NA with empty string
  df[is.na(df)] <- ""
  df[df=="NA"] <- ""  
  
  cat(ms$methTitle, "\n")
  if(formula) cat("Formula: ", deparse(ms$call$formula), "\n")
  ngrps <- vapply(model@flist, nlevels, 0L)
  dims <- model@devcomp$dims
  cat(lme4::.prt.grps(ngrps, nobs = dims[["n"]]), "\n")
  df
  
}

#' Convert Formula to String
#'
#' This function converts a formula object into a string representation.
#' @param formula A formula object to be converted into a string.
#' @return A string representation of the input formula.
form2str <- function(formula){
  fstr <- paste0(as.character(formula)[2:3], collapse = "~")
  fstr <- gsub(" ", "", fstr)
  #remove characters not allowed in Windows file names
  fstr <- gsub("[<>:\"/\\\\|?*]", "_x_", fstr)
  fstr
}



#' Calculate mean, median, N, and 95% CI
summary_stats <- function(x, R=1000) {
  if(length(unique(x)) < 2 | is.null(R)) { 
    boot_ci <- list(percent = rep(NA, 5))
  } else {
    # Bootstrap the mean
    boot_obj <- boot::boot(x, function(u, i) mean(u[i]), R=R)
    boot_ci <- boot::boot.ci(boot_obj, type="perc")
  }
  
  n <- length(x)
  df <- n - 1
  
  mean_val <- mean(x, na.rm = TRUE)
  sd_val <- sd(x, na.rm = TRUE)
  se_val <- sd_val / sqrt(n)
  
  mean_lower <- ifelse(df > 0, mean_val - qt(0.975, df=df) * se_val, NA)
  mean_upper <- ifelse(df > 0, mean_val + qt(0.975, df=df) * se_val, NA)
  
  return(c(mean = mean_val,
           median = median(x, na.rm = TRUE),
           N = n,
           sd = sd_val,
           se = se_val,
           mean.lower = mean_lower,
           mean.upper = mean_upper,
           median.lower = ifelse(is.na(boot_ci$percent[4]), NA, boot_ci$percent[4]),
           median.upper = ifelse(is.na(boot_ci$percent[5]), NA, boot_ci$percent[5])))
}




#' Print Named Vector
#'
#' This function prints a named vector in a specific format. For each unique item in the vector, 
#' it lists the names of these items. If the vector is unnamed, it prints the sorted values.
#'
#' @param vec A named or unnamed vector.
#'
#' @return No return value. The function prints to the console.
#'
#' @examples
#' a <- c(x="a",y="a",z="b",n="c")
#' printNamed(a)
#' printNamed(unname(a))
printNamed <- function(vec) {
  if(is.null(names(vec))) {
    #paste sorted and numbered
    cat(paste0(1:length(vec), ") ", sort(vec), collapse = "\n"))
  } else {
    unique_values <- sort(unique(vec))
    for(val in unique_values) {
      #number the vals
      nr <- paste0(which(unique_values == val), ") ")
      cat(paste0(nr,val, "\n"))
      cat(paste0("\t", names(vec[vec == val]), "\n"))
    }
  }
}


