#########################################################################################
#
#  Script for the computations of the paper
#
#  "Automorphisms of Nikulin-type orbifolds" (2024)
#  by Simon Brandhorst, Gregoire Menet and Stevell Muller.
#
#  To load the data from the folders "symplectic" and "nonsymplectic" in Oscar, use the
#  function 
#
#  `julia> load("local/path/to/file.mrdi")'.
#
#  This will create an object of type `ZZLatWithIsom`. Refer to the documentation
#  of OSCAR, section "QuadFormAndIsom" for more details on how to work with such
#  objects.
#
#  To load this file, just use
#  
#  `julia> include("path/to/code.jl")`.
#
#  The code has been tested on version v1.2.0 of OSCAR.
#
#########################################################################################


@doc raw"""
    is_symplectic(Lf::ZZLatWithIsom) -> Bool

Given a pair ``(L, f)`` where ``L`` is isometric to the Beauville-Bogomolov
quadratic lattice of an orbifold of Nikulin-type and ``f`` is a finite order
isometry of ``L``, return whether the isometry ``f`` is symplectic.
"""
function is_symplectic(Lf::ZZLatWithIsom)
  L = lattice(Lf)
  C = lattice(coinvariant_lattice(Lf))

  rank(C) == 0 && return true
  !is_negative_definite(C) && return false

  # numerical prime exceptional divisors are of type (-2, 1) and (-4, 2)

  ### Case (-2, 1)

  sv2 = QQMatrix[matrix(QQ,1,rank(C), v)*basis_matrix(C) for (v, _) in short_vectors(C, 2, 2)]
  any(v -> divisibility(L, v) == 1, sv2) && return false
  
  ### Case (-4, 2)
  
  # In that case, all vectors in C have divisibility 1 in C
  is_odd(numerator(det(C))) && return true 
  
  F = lattice(invariant_lattice(Lf))
  _, _, HCinqC = glue_map(L, F, C)

  # In that case, all vectors in C have divisibility 1 in L
  is_bijective(HCinqC) && return true
  
  # Vectors of C with divisibility 2 in L lie in 2*L^v \cap C
  S2 = intersect(2*dual(L), C)  
  !isempty(short_vectors(S2, 4, 5)) && return false
  return true
end

@doc raw"""
    is_regular_symplectic(Lf::ZZLatWithIsom) -> Bool

Given a pair ``(L, f)`` where ``L`` is isometric to the Beauville-Bogomolov
quadratic lattice of an orbifold of Nikulin-type and ``f`` is a finite order
isometry of ``L``, return whether the isometry ``f`` is regular symplectic.
"""
function is_regular_symplectic(Lf::ZZLatWithIsom)
  !is_symplectic(Lf) && return false
  
  L = lattice(Lf)
  C = lattice(coinvariant_lattice(Lf))
  qL, _ = discriminant_group(Lf)
  
  # Needed for the wall divisors of numerical type (-12,2)
  N, qLtoN = normal_form(qL)

  # numerical wall divisors which are not prime exceptional are either
  # - vectors of type (-6, 2)
  # - vectors of type (-12, 2) such that the projection onto the U(2)^3-part of L is 2-divisible
  # Note that the latter is invariant under isometry of L and can be checked on D_L directly. For this,
  # we fix the normal form of D_L such that the first 6 generators span a copy of D_{U(2)^3}. Then
  # a vector v of type (-12, 2) is a numerical wall divisor if the 6 first coordinate of v/2+L in the fixed
  # normal form of D_L are zero.
  
  # Vectors of C with divisibility 2 in L lie in 2*L^v \cap C
  S2 = intersect(2*dual(L), C)
  
  ### Case (-6, 2)
  !isempty(short_vectors(S2, 6, 7)) && return false
  
  ### Case (-12, 2)
  sv = short_vectors(S2, 12, 13)
  is_empty(sv) && return true
  
  # Elements in sv are given in the coordinates of S2, we express in the coordinates of the ambient space
  # and we rescale then by 1//2 to get the vectors in the dual of L (since the have divibisility 2)
  sv_ambient = QQMatrix[1//2*matrix(QQ, 1, rank(S2), v[1])*basis_matrix(S2) for v in sv]
  
  # Now to verify the condition, we look at the class of these vectors in qL, and we map them in the
  # normal form N of qL: there, we can check the condition previoulsy stated
  svN = elem_type(N)[qLtoN(qL(vec(collect(v)))) for v in sv_ambient]
  any(v -> iszero(data(v).coeff[1, 1:6]), svN) && return false
  
  return true
end

### Symplectic case

@doc raw"""
    symplectic_isometry_nikulin_orb(path) -> Nothing

Let ``L`` be isometric to U(2)^3+E_8+A_1^2. This function computes, by
induction, representatives for the ``O(L)``-conjugacy classes of symplectic
isometries in ``O^+(L)`` of finite order.

Each output is saved in the folder `path`.
"""
function symplectic_isometry_nikulin_orb(path; D)
  # Construct the Nikulin-type lattice
  U2 = hyperbolic_plane_lattice(2)
  E8 = rescale(root_lattice(:E, 8), -1)
  A1 = rescale(root_lattice(:A, 1), -1)
  L = direct_sum(U2, U2, U2, E8, A1, A1)[1]
  
  # Currently we can only do systematically orders with at most two distinct prime divisors
  # We also remove order 1 since we add it by hand
  ords = Int.(filter(k -> length(prime_divisors(k)) in [1,2], 1:66))
  
  # We iterate on the order, so we save intermediate results in a dictionary
  D = Dict{Int, Vector{ZZLatWithIsom}}()
  
  # We initialize with the isometry of order 1
  D[1] = ZZLatWithIsom[integer_lattice_with_isometry(L)]
  for m in ords
    # Separate the case of prime powers and composite orders
    ok, j, p = is_prime_power_with_data(m)
    if ok
      Nm = _get_isometry_prime_power!(D, L, p, j)
    else
      Nm = _get_isometry_composite!(D, m)
    end
    
    # We save our partial results, i.e. conjugacy classes for the order m
    if !isempty(Nm)
      k = length(Nm)
      if m >= 10
        pp = path*"$m"
      else
        pp = path*"0$m"
      end
      l = ndigits(k)
      for i in 1:k
        count = "0"^(l-ndigits(i))*"$i"
        save(pp*"-"*count*".mrdi", Nm[i])
      end
    end
  end
  return D
end

# Iterate from p^(j-1) to p^j. Since we do in increasing order, p^(j-1) has already been computed.
function _get_isometry_prime_power!(D::Dict, L::ZZLat, p::Hecke.IntegerUnion, j::Hecke.IntegerUnion)
  @assert haskey(D, p^(j-1))
  # Recover the cases of order p^(j-1)
  Dp = D[p^(j-1)]
  
  # Where we store the results to be serialized later
  Dpj = ZZLatWithIsom[]
  
  # To avoid duplication, we need to filter and apply the next loop for only one representative of each type (conjugate => same type, but not the converse)
  rtype = []
  for N in Dp
    any(t -> is_of_type(N, t), rtype) && continue
    push!(rtype, type(N))
    
    # Compute representatives of conjugacy classes of isometry whose pth power has same type as N, and invariant lattice of positive signature p_inv=3
    Np = splitting_of_mixed_prime_power(N, p; p_inv=3) 
    
    # Sanity check
    filter!(NN -> valuation(order_of_isometry(NN), p) == j, Np)
    
    # Take care of effictivity
    filter!(NN -> is_symplectic(NN), Np)
    append!(Dpj, Np)
   end
   
   # Store the new list for later iterations
   D[p^j] = Dpj
   return Dpj
end

# Iterate from a divisor of n: here n = p^i*q^j, we choose as convention to iterate from p^i*q^(j-1)
# if p < q (one could choose to do from p^(i-1)*q^j instead, we would get the same results)
function _get_isometry_composite!(D::Dict, n::Hecke.IntegerUnion)
  p, q = sort(prime_divisors(n))
  i, j = valuation(n, p), valuation(n, q)
  @assert haskey(D, p^i*q^(j-1))
  Dp = D[p^i*q^(j-1)]
  r = q
  Dn = ZZLatWithIsom[]
  rtype = []
  for N in Dp
    any(t -> is_of_type(N, t), rtype) && continue
    push!(rtype, type(N))
    Nq = splitting_of_mixed_prime_power(N, r; p_inv=3)
    filter!(NN -> order_of_isometry(NN) == n, Nq)
    filter!(NN -> is_symplectic(NN), Nq)
    append!(Dn, Nq)
  end
  D[n] = Dn
  return Dn
end
