PROGRAM READ_WIND_ERA5
USE NETCDF


IMPLICIT NONE

! ---------------------------------------------------------------------------- !
!                                                                              !
!     LOCAL VARIABLES.                                                         !
!     ----------------                                                         !


INTEGER, PARAMETER :: KIND_D = 8

INTEGER, SAVE         :: ICODE = 3  !! WIND CODE: 1= USTAR; 2= USTRESS; 3= U10
INTEGER, SAVE         :: N_LON      !! NUMBER OF LONGITUDES IN GRID.
INTEGER, SAVE         :: N_LAT      !! NUMBER OF LATITUDES IN GRID.
INTEGER, SAVE         :: N_FRAME      !! NUMBER OF TIME FRAME IN EACH ERA5 INPUT.
REAL (KIND=KIND_D)    :: D_LAT      !! LATITUDE INCREMENT OF GRID [DEG].
REAL (KIND=KIND_D)    :: D_LON      !! LONGITUDE INCREMENT OF GRID [DEG].
REAL (KIND=KIND_D)    :: SOUTH      !! SOUTH LATITUDE OF GRID [DEG].
REAL (KIND=KIND_D)    :: NORTH      !! NORTH LATITUDE OF GRID [DEG].
REAL (KIND=KIND_D)    :: WEST       !! WEST LONGITUDE OF GRID [DEG].
REAL (KIND=KIND_D)    :: EAST       !! EAST LONGITUDE OF GRID [DEG].
REAL,    ALLOCATABLE  :: U_MAP(:,:) !! 1. COMPONENT OF WIND MAP [M/S].
REAL,    ALLOCATABLE  :: V_MAP(:,:) !! 2. COMPONENT OF WIND MAP [M/S].
REAL, ALLOCATABLE  :: U10(:,:) !! TEMPORARY: FLIPUD OF ERA5.
REAL, ALLOCATABLE  :: V10(:,:) 
CHARACTER (LEN=14)    :: CDTWIR     !! DATE/TIME OF WIND FIELD

LOGICAL, SAVE  :: FRSTIME = .TRUE.
INTEGER        :: IOS, I, J
! ERA5 READING
INTEGER, SAVE  :: FRAME_COUNT       !! TIME FRAME COUNTER IN ERA5 MONTHLY INPUT
INTEGER, SAVE  :: NCF_COUNT       !! WHICH ERA MONTHLY FILE IS BEING READ NOW? 
INTEGER, SAVE  :: FIRSTFRAMETIME    !! FIRST TIME FRAME FOR THE CURRENT RUN
INTEGER, PARAMETER  :: IU_NC_NML = 474 !! FILE UNIT FOR NAMELIST.NC
INTEGER, SAVE  :: L_UV_NC, NCFC     !! NCFC: HOW MANY INPUT ERA5 FILES?
                                    !! L_UV_NC: 1, U,V SEPERATE; 0,U,V TOGETHER
CHARACTER (LEN=70), DIMENSION(:), ALLOCATABLE :: UV_NCNAMES,U_NCNAMES,V_NCNAMES
REAL (KIND=KIND_D), DIMENSION(:), ALLOCATABLE :: WLON,WLAT
INTEGER, DIMENSION(:), ALLOCATABLE :: WTIME
SAVE UV_NCNAMES,U_NCNAMES,V_NCNAMES
SAVE WLON,WLAT,WTIME
NAMELIST /NCUSER/ NCFC,L_UV_NC
NAMELIST /NCFNAMES/  U_NCNAMES,V_NCNAMES,UV_NCNAMES

!DATETIME VARS
TYPE DATETIME_TYPE
  INTEGER :: YEAR 
  INTEGER :: MONTH
  INTEGER :: DAY
  INTEGER :: HOUR
  INTEGER :: MINUTE
  INTEGER :: SECOND
  !START DATE&TIME IS SET TO THE FIRST FRAME OF WIND FIELD 
  INTEGER :: YEAR0   = 2021
  INTEGER :: MONTH0  = 1
  INTEGER :: DAY0    = 1
  INTEGER :: HOUR0   = 0
  INTEGER :: MINUTE0 = 0
  INTEGER :: SECOND0 = 0
  !INTEGER :: YEAR0   = 1900
  !INTEGER :: MONTH0  = 1
  !INTEGER :: DAY0    = 1
  !INTEGER :: HOUR0   = 0
  !INTEGER :: MINUTE0 = 0
  !INTEGER :: SECOND0 = 0
  !REAL(8) :: TIMESTAMP ! SECONDS FROM START TIME
  REAL(8) :: TIMESTAMP_HOUR ! HOURS FROM START TIME
  CHARACTER(14) :: DATE_STR       = "00000000000000"
END TYPE
TYPE(DATETIME_TYPE) :: WIND_DATETIME

INTEGER :: II

II=1
DO WHILE (II<1440)

! ---------------------------------------------------------------------------- !
!                                                                              !
!     1. READ IN NETCDF FILE NAMES DEFINED IN NAMELIST.NC                      !
!        AND DERIVE INPUT WIND FIELD BOUNDS AND TIME FRAME
!        ------------------------------------------                            !
IF (FRSTIME) THEN
    !OPEN NAMELIST.NC TO READ NETCDF FILE NAMES. SEE NML FOR DETAILS 
    !NAMELIST.NC SHOULD IN THE SAME FOLDER OF EXECUTABLE
    OPEN(UNIT=IU_NC_NML,FILE="NAMELIST.NC",STATUS='OLD')
    READ(IU_NC_NML,NML=NCUSER,IOSTAT=IOS)
    IF (IOS.NE.0) THEN
        WRITE(*,*) 'NAMELIST.NC READ FAILED'
        CLOSE(IU_NC_NML)
        STOP
    ELSE
        IF (L_UV_NC==1) THEN
            IF (.NOT.ALLOCATED(U_NCNAMES)) ALLOCATE(U_NCNAMES(NCFC))
            IF (.NOT.ALLOCATED(V_NCNAMES)) ALLOCATE(V_NCNAMES(NCFC))
            READ(IU_NC_NML,NML=NCFNAMES, IOSTAT=IOS)
            CLOSE(IU_NC_NML)
            IF (IOS.NE.0) THEN
                WRITE(*,*) 'NAMELIST.NC READ FAILED'
                STOP
            END IF
        ELSE
            !NAMELIST.NC: CONTAIN UV_NCNAMES ONLY
            IF (.NOT.ALLOCATED(UV_NCNAMES)) ALLOCATE(UV_NCNAMES(NCFC))
            READ(IU_NC_NML,NML=NCFNAMES, IOSTAT=IOS)
            CLOSE(IU_NC_NML)
            IF (IOS.NE.0) THEN
                WRITE(*,*) 'NAMELIST.NC READ FAILED'
                STOP
            END IF
        END IF
    END IF
    !NOW WIND NETCDF FILENAME LOADED
    !1.2 DERIVE WIND FIELD BOUNDS AND DELTA-X-Y ET AL.
    IF (L_UV_NC==1) THEN
        CALL READ_ERA5_DIM( U_NCNAMES(1) )
        !
        ALLOCATE( WLON(N_LON), WLAT(N_LAT) )
        ALLOCATE(  WTIME(N_FRAME) )
        !
        CALL READ_ERA5_GRID( U_NCNAMES(1) )
        SOUTH = MINVAL(WLAT)
        NORTH = MAXVAL(WLAT)
        WEST  = MINVAL(WLON)
        EAST  = MAXVAL(WLON)
        D_LAT = ABS(WLAT(1)-WLAT(2))
        D_LON = ABS(WLON(2)-WLON(1))

        PRINT *, "Basic info: ", SOUTH, NORTH, WEST,EAST,D_LAT, D_LON

        FRSTIME = .FALSE.

    ELSE  ! ELSE FOR L_UV_NC \= 1
        WRITE(*, *) 'WINDFIELD U, V COMPONENTS ARE IN SAME FILE'
        WRITE(*, *) 'TO DO LATER....'
        STOP
    END IF

    FRAME_COUNT = 1
    NCF_COUNT = 1    !FIRST NC FILE (FIRST MONTH)
    FIRSTFRAMETIME = WTIME(1)
    DEALLOCATE(WLON,WLAT,WTIME)

END IF  ! ENDIF FOR FRSTIME

! ---------------------------------------------------------------------------- !
!                                                                              !
!     2. ALLOCATE WIND INPUT ARRAYS.                                           !
!        ---------------------------                                           !

IF (.NOT.ALLOCATED(U_MAP) ) ALLOCATE(U_MAP(N_LON,N_LAT))
IF (.NOT.ALLOCATED(V_MAP) ) ALLOCATE(V_MAP(N_LON,N_LAT))
IF (.NOT.ALLOCATED(U10) ) ALLOCATE(U10(N_LON,N_LAT))
IF (.NOT.ALLOCATED(V10) ) ALLOCATE(V10(N_LON,N_LAT))

! ---------------------------------------------------------------------------- !
!                                                                              !
!    3. READ WIND FIELD.                                                       !
!       -------------------                                                    !

IF (L_UV_NC==1) THEN
    
    IF(FRAME_COUNT==1) THEN
        CALL READ_ERA5_TIME( U_NCNAMES(NCF_COUNT) )
    END IF

    !!! NOTE: START DATE&TIME SHOULD BE SET TO THE FIRST FRAME OF WINDFIELD
    !!! AT DECLARATION OF TYPE()
    WIND_DATETIME%TIMESTAMP_HOUR = REAL(WTIME(FRAME_COUNT)-FIRSTFRAMETIME,8)
    CALL UPDATE_DATETIME_STRING(WIND_DATETIME)
    CDTWIR=WIND_DATETIME%DATE_STR

    PRINT * , "WIND FIELD TIME STRING: ", CDTWIR

    CALL READ_ERA5_FIELD(U_NCNAMES(NCF_COUNT),V_NCNAMES(NCF_COUNT))

    IF (FRAME_COUNT==N_FRAME) THEN
        FRAME_COUNT = 1
        NCF_COUNT = NCF_COUNT+1
    ELSE
        FRAME_COUNT = FRAME_COUNT + 1
    END IF



ELSE  ! ELSE FOR L_UV_NC \= 1
    WRITE(*, *) 'WINDFIELD U, V COMPONENTS ARE IN SAME FILE'
    WRITE(*, *) 'TO DO LATER....'
    STOP
END IF

! ---------------------------------------------------------------------------- !
!                                                                              !
!    4. WRITE TEST OUTPUT AND DEALLOCATE ARRAYS.                               !
!       ----------------------------------------                               !


DEALLOCATE(U_MAP)
DEALLOCATE(V_MAP)
DEALLOCATE(U10)
DEALLOCATE(V10)

PRINT *, "READ FRAME: ", II, FRAME_COUNT, NCF_COUNT
II = II+1
END DO

RETURN

CONTAINS

    SUBROUTINE READ_ERA5_DIM( NCFNAME )
        IMPLICIT NONE
        INTEGER :: NCID
        INTEGER :: LON_DIMID, LAT_DIMID, TIME_DIMID
        CHARACTER(30) :: DIMNAME
        CHARACTER(LEN=70), INTENT(IN) ::  NCFNAME

        CALL CHECK_NC( NF90_OPEN(TRIM(ADJUSTL(NCFNAME)),NF90_NoWRITE,NCID),"OPEN NC" )
        ! GET DIM ID
        CALL CHECK_NC(NF90_INQ_DIMID(NCID, 'longitude' ,LON_DIMID  ),"LON DIMID")
        CALL CHECK_NC(NF90_INQ_DIMID(NCID, 'latitude'  ,LAT_DIMID  ),"LAT DIMID")
        CALL CHECK_NC(NF90_INQ_DIMID(NCID, 'time'      ,TIME_DIMID ),"TIME DIMID")
        ! GET DIMENSION NUM
        CALL CHECK_NC(NF90_INQUIRE_DIMENSION(NCID,LON_DIMID,DIMNAME,N_LON),"LON DIM")
        CALL CHECK_NC(NF90_INQUIRE_DIMENSION(NCID,LAT_DIMID,DIMNAME,N_LAT),"LAT DIM")
        CALL CHECK_NC(NF90_INQUIRE_DIMENSION(NCID,TIME_DIMID,DIMNAME,N_FRAME),"TIME DIM")
        !
        CALL CHECK_NC(NF90_CLOSE(NCID))

    END SUBROUTINE READ_ERA5_DIM

    SUBROUTINE READ_ERA5_GRID( NCFNAME )
        IMPLICIT NONE
        INTEGER :: NCID
        INTEGER :: LON_ID, LAT_ID, TIME_ID
        CHARACTER(LEN=70), INTENT(IN) ::  NCFNAME

        CALL CHECK_NC(NF90_OPEN(TRIM(ADJUSTL(NCFNAME)),NF90_NOWRITE,NCID))
        ! GET VAR ID
        CALL CHECK_NC(NF90_INQ_VARID(NCID, 'longitude', LON_ID),"LON ID")
        CALL CHECK_NC(NF90_INQ_VARID(NCID, 'latitude' , LAT_ID),"LAT ID")
        CALL CHECK_NC(NF90_INQ_VARID(NCID, 'time'     ,TIME_ID),"TIME ID")
        CALL CHECK_NC(NF90_GET_VAR(NCID, LON_ID,  WLON),"LON")
        CALL CHECK_NC(NF90_GET_VAR(NCID, LAT_ID,  WLAT), "LAT")
        CALL CHECK_NC(NF90_GET_VAR(NCID, TIME_ID,WTIME), "TIME")
        CALL CHECK_NC(NF90_CLOSE(NCID))

    END SUBROUTINE READ_ERA5_GRID

    SUBROUTINE READ_ERA5_TIME( NCFNAME )
        IMPLICIT NONE
        INTEGER :: NCID
        INTEGER :: TIME_DIMID, TIME_ID
        CHARACTER(30) :: DIMNAME
        CHARACTER(LEN=70), INTENT(IN) ::  NCFNAME

        CALL CHECK_NC( NF90_OPEN(TRIM(ADJUSTL(NCFNAME)),NF90_NoWRITE,NCID) )
        ! GET DIM ID
        CALL CHECK_NC(NF90_INQ_DIMID(NCID, 'time' , TIME_DIMID ),"TIME DIMID")
        CALL CHECK_NC(NF90_INQ_VARID(NCID, 'time' , TIME_ID),"TIM ID")
        ! GET DIMENSION NUM
        CALL CHECK_NC(NF90_INQUIRE_DIMENSION(NCID,TIME_DIMID,DIMNAME,N_FRAME),"TIME DIM")
        IF (ALLOCATED(WTIME)) DEALLOCATE(WTIME)
        ALLOCATE(WTIME(N_FRAME))
        CALL CHECK_NC(NF90_GET_VAR(NCID, TIME_ID,WTIME),"TIME")
        !
        CALL CHECK_NC(NF90_CLOSE(NCID))

    END SUBROUTINE READ_ERA5_TIME


    SUBROUTINE READ_ERA5_FIELD(U_NCFNAME,V_NCFNAME)
        IMPLICIT NONE
        INTEGER :: NCID
        INTEGER :: U10_ID, V10_ID, TIME_ID
        CHARACTER(LEN=70), INTENT(IN) ::  U_NCFNAME, V_NCFNAME
        REAL(8) :: SCALE_FACTOR, ADD_OFFSET, FILLVALUE
        


       CALL CHECK_NC(NF90_OPEN(TRIM(ADJUSTL(U_NCFNAME)),NF90_NOWRITE,NCID) )
       ! GET VAR ID
       CALL CHECK_NC(NF90_INQ_VARID(NCID, 'u10n', U10_ID),"U10n ID")
       CALL CHECK_NC(NF90_GET_VAR(NCID, U10_ID, U10  ,                  &
                     START=(/1,1,FRAME_COUNT/),COUNT=(/N_LON,N_LAT,1/)),"U10n")
       CALL CHECK_NC(NF90_GET_ATT(NCID  ,U10_ID  ,'scale_factor',       &
                                  SCALE_FACTOR),"SCALE FACTOR")
       CALL CHECK_NC(NF90_GET_ATT(NCID  ,U10_ID  ,'add_offset',         &
                                  ADD_OFFSET),"addoffset")
       CALL CHECK_NC(NF90_GET_ATT(NCID, U10_ID, "_FillValue",           &
                                  FILLVALUE),'fillvalue')
       WHERE (U10 .NE. FILLVALUE) U10 = U10*SCALE_FACTOR + ADD_OFFSET
       U_MAP(:,:) = U10(:, N_LAT:1:-1)
       CALL CHECK_NC(NF90_CLOSE(NCID))

      WRITE(*,'(1X,48F6.1)') U_MAP(1:MIN(48,N_LON),1:MIN(10,N_LAT))

       !
       CALL CHECK_NC(NF90_OPEN(TRIM(ADJUSTL(V_NCFNAME)),NF90_NOWRITE,NCID) )
       ! GET VAR ID
       CALL CHECK_NC(NF90_INQ_VARID(NCID, 'v10n', V10_ID))
       CALL CHECK_NC(NF90_GET_VAR(NCID, V10_ID, V10  ,                  &
                     START=(/1,1,FRAME_COUNT/),COUNT=(/N_LON,N_LAT,1/)))
       CALL CHECK_NC(NF90_GET_ATT(NCID  ,V10_ID  ,'scale_factor',       &
                                  SCALE_FACTOR))
       CALL CHECK_NC(NF90_GET_ATT(NCID  ,V10_ID  ,'add_offset',         &
                                  ADD_OFFSET))
       CALL CHECK_NC(NF90_GET_ATT(NCID, V10_ID, "_FillValue",           &
                                  FILLVALUE))
       WHERE (V10 .NE. FILLVALUE) V10 = V10*SCALE_FACTOR + ADD_OFFSET
       V_MAP(:,:) = V10(:, N_LAT:1:-1)
       CALL CHECK_NC(NF90_CLOSE(NCID))

    END SUBROUTINE READ_ERA5_FIELD

    SUBROUTINE CHECK_NC(STATUS,OPERATION)
       IMPLICIT NONE
       INTEGER,INTENT(IN) :: STATUS
       CHARACTER(LEN=*), INTENT(IN),OPTIONAL :: OPERATION
       IF (STATUS /= NF90_NOERR) THEN
           PRINT*, TRIM( NF90_STRERROR(STATUS) ), OPERATION
           STOP
       END IF
    END SUBROUTINE CHECK_NC

    SUBROUTINE UPDATE_DATETIME_STRING(THIS)

        TYPE(DATETIME_TYPE), INTENT(INOUT) :: THIS
        INTEGER :: DAYS, HOURS, MINUTES, SECONDS, MONTH_DAYS, I
        REAL(8) :: DAYS_R
        
        !THIS%TIMESTAMP = THIS%TIMESTAMP_HOUR*3600.0
    
        DAYS_R  = THIS%TIMESTAMP_HOUR/24.0
        DAYS  = INT(DAYS_R)
        !HOURS  = MOD(THIS%TIMESTAMP_HOUR, 24.0)
        HOURS  = MOD(INT(THIS%TIMESTAMP_HOUR), 24)
        MINUTES = 0
        SECONDS = 0
    
        THIS%YEAR   = THIS%YEAR0
        THIS%MONTH  = THIS%MONTH0
        THIS%DAY    = THIS%DAY0
        THIS%HOUR   = HOURS
        THIS%MINUTE = MINUTES
        THIS%SECOND = SECONDS
    
        IF (DAYS > 0) THEN
           DO I = 1, DAYS
              MONTH_DAYS = DAYS_OF_MONTH(THIS%YEAR, THIS%MONTH)
              THIS%DAY = THIS%DAY + 1
              IF (THIS%DAY > MONTH_DAYS) THEN
                 THIS%DAY =  1
                 THIS%MONTH = THIS%MONTH + 1
              END IF
              
              IF (THIS%MONTH > 12) THEN
                 THIS%MONTH =  1
                 THIS%YEAR = THIS%YEAR + 1
              END IF
           END DO
        END IF
     
        !! DATE STRING YYYY-MM-DD_HH:MM:SS 
        WRITE(THIS%DATE_STR(1:4),"(I4)") THIS%YEAR
        IF (THIS%MONTH .LT. 10) THEN
           WRITE(THIS%DATE_STR(5:5),"(I1)") 0
           WRITE(THIS%DATE_STR(6:6),"(I1)") THIS%MONTH
        ELSE
           WRITE(THIS%DATE_STR(5:6),"(I2)") THIS%MONTH
        END IF
        IF (THIS%DAY .LT. 10) THEN
           WRITE(THIS%DATE_STR(7:7),"(I1)")   0
           WRITE(THIS%DATE_STR(8:8),"(I1)") THIS%DAY
        ELSE
           WRITE(THIS%DATE_STR(7:8),"(I2)") THIS%DAY
        END IF
        IF (THIS%HOUR .LT. 10) THEN
           WRITE(THIS%DATE_STR(9:9),"(I1)") 0
           WRITE(THIS%DATE_STR(10:10),"(I1)") THIS%HOUR
        ELSE
           WRITE(THIS%DATE_STR(9:10),"(I2)") THIS%HOUR
        END IF
        IF (THIS%MINUTE .LT. 10) THEN
           WRITE(THIS%DATE_STR(11:11),"(I1)") 0
           WRITE(THIS%DATE_STR(12:12),"(I1)") THIS%MINUTE
        ELSE
           WRITE(THIS%DATE_STR(11:12),"(I2)") THIS%MINUTE
        END IF
        IF (THIS%SECOND .LT. 10) THEN
           WRITE(THIS%DATE_STR(13:13),"(I1)") 0
           WRITE(THIS%DATE_STR(14:14),"(I1)") THIS%SECOND
        ELSE
           WRITE(THIS%DATE_STR(13:14),"(I2)") THIS%SECOND
        END IF
    
    END SUBROUTINE UPDATE_DATETIME_STRING
    
    INTEGER FUNCTION DAYS_OF_MONTH(YEAR, MONTH) RESULT(RES)
        INTEGER, INTENT(IN) :: YEAR
        INTEGER, INTENT(IN) :: MONTH
        INTEGER, PARAMETER :: DAYS(12) = [31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31]
    
        IF (MONTH == 2 .AND. IS_LEAP_YEAR(YEAR)) THEN
            RES = 29
        ELSE
            RES = DAYS(MONTH)
        END IF
     END FUNCTION DAYS_OF_MONTH
    
    LOGICAL FUNCTION IS_LEAP_YEAR(YEAR) RESULT(RES)
        INTEGER, INTENT(IN) :: YEAR
        RES = (MOD(YEAR, 4) == 0 .AND. .NOT. MOD(YEAR, 100) ==0) .OR. (MOD(YEAR, 400) == 0)
    END FUNCTION IS_LEAP_YEAR


END PROGRAM READ_WIND_ERA5
