#!/usr/bin/env python3
"""
Make a manifest of all experiment configurations to generate the pipeline.

Rerun this script after adding, removing, or modifying configurations in 
the config directory.
"""

from pathlib import Path
import sys
from argparse import ArgumentParser, Namespace
import logging
import json

from yaml import safe_load

from gbsim.script_util import init_logging

_log = logging.getLogger('update-manifest')
cfg_dir = Path('config')


def main(args: Namespace):
    manifest = {}
    for file in cfg_dir.glob('*.yaml'):
        _log.info('reading file %s', file)
        with file.open('rt') as f:
            data = safe_load(f)
        manifest[data['experiment_name']] = {
            'config': file.name,
            'name': data['experiment_name'],
            'algo': data['algo']['name'],
        }
    
    _log.info('saving %d experiments to manifest', len(manifest))
    with (cfg_dir / 'manifest.json').open('wt') as f:
        json.dump({'simulations': manifest}, f)


def parse_arguments() -> Namespace:
    parser = ArgumentParser(description='Update experiment manifest')
    parser.add_argument('-v', '--verbose', action='store_true', help='verbose logging')
    return parser.parse_args()


if __name__ == '__main__':
    args = parse_arguments()
    init_logging(args)
    main(args)