/********************************************************
 *     _____________________
 *    / ____/  _/ ___/_  __/
 *   / / __ / / \__ \ / /   
 *  / /_/ // / ___/ // /    
 *  \____/___//____//_/ 
 * Geophysical Inversions using Spherical Tetrahedral meshes (GIST)
 *
 * Copyright (c) 2022  Yi Zhang (yizhang-geo@zju.edu.cn)
 *
 * GIST is distributed under a dual licensing scheme. You can redistribute 
 * it and/or modify it under the terms of the GNU Affero General Public 
 * License (AGPL) as published by the Free Software Foundation, either version 
 * 3 of the License, or (at your option) any later version. You should have 
 * received a copy of the GNU Affero General Public License along with this 
 * program. If not, see <http://www.gnu.org/licenses/>.
 * 
 * If the terms and conditions of the AGPL v.3. would prevent you from using 
 * the GIST, please consider the option to obtain a commercial license for a 
 * fee. These licenses are offered by the original author, Yi Zhang. As a rule, 
 * licenses are provided "as-is", unlimited in time for a one time fee. Please 
 * send corresponding requests to: yizhang-geo@zju.edu.cn. Please do not forget 
 * to include some description of your company and the realm of its activities. 
 * Also add information on how to contact you by electronic and paper mail.
 ******************************************************/

#ifndef _GIST_THERMOPHYSIC_H
#define _GIST_THERMOPHYSIC_H

#include "gctl/core.h"
#include "gctl/io.h"
#include "gctl/mesh.h"

#include "../utility/enum.h"

using namespace gctl;

namespace GIST
{
    class Thermophysics
    {
    public:
        Thermophysics();
        virtual ~Thermophysics();

        /**
         * @brief Clear the reference grid.
         * 
         */
        void Clear();

        /**
         * @brief Initiate the reference grid from a data array.
         * 
         * @param name Name of the reference grid.
         * @param unit Unit of the physical quantity,
         * @param tmin Minimal temperature.
         * @param tmax Maximal temperature.
         * @param dt   Temperature spacing.
         * @param pmin Minimal pressure.
         * @param pmax Maximal pressure.
         * @param dp   Pressure spacing.
         * @param cmin Minimal composition.
         * @param cmax Maximal composition.
         * @param dc   Compositional spacing
         * @param data Data array.
         */
        void ImportArray(std::string name, std::string unit, double tmin, double tmax, double dt, 
            double pmin, double pmax, double dp, double cmin, double cmax, double dc, const array<double> &data); // 从数组初始化
        
        /**
         * @brief Initiate the reference grid from a text file.
         * 
         * File format:
         * <Name> <Unit>
         * <tmin> <tmax> <dt>
         * <pmin> <pmax> <dp>
         * <cmin> <cmax> <dc>
         * <data>
         * <data>
         * ...
         * 
         * @param filename Input file name.
         */
        void ImportText(std::string filename); // 载入专用的text文件

        /**
         * @brief Load a reference grid from a binary file.
         * 
         * @param filename Input file name.
         */
        void LoadGrid(std::string filename); // 载入专用的rg文件

        /**
         * @brief Save a reference grid to a binary file.
         * 
         * @param filename Output file name.
         */
        void SaveGrid(std::string filename); // 保存专用的rg文件

        /**
         * @brief Save the reference grid to Gmsh mesh file.
         * 
         * @note  Names of the axis are temperature (x), pressure (y) and composition (z). Axial ranges of the output mesh are normalized to [0, 1].
         * 
         * @param filename Output file name.
         */
        void SaveMsh(std::string filename); // 保存用于可视化的Gmsh文件

        /**
         * @brief Show grid information.
         * 
         * @param ss Output stream.
         */
        void ShowInfo(std::ostream &ss = std::clog); // 显示参数网格的信息

        /**
         * @brief Preform gauss smooth to the reference grid
         * 
         * @param times Times of the smoothness preocess.
         */
        void GaussSmooth(int times = 1);
        
        /**
         * @brief Convert parameter values to the range [0, 1].
         * 
         * @param in_model Input parameter array.
         * @param ret_model Output parameter array.
         * @param p_type Parameter type.
         */
        void Convert2RelativeModel(const array<double> &in_model, array<double> &ret_model, thermophysics_type_e p_type);

        /**
         * @brief Convert the normalized parameter to original range.
         * 
         * @param in_model Input parameter array.
         * @param ret_model Output parameter array.
         * @param p_type Parameter type.
         */
        void Convert2AbsoluteModel(const array<double> &in_model, array<double> &ret_model, thermophysics_type_e p_type);

        /**
         * @brief Export 1D profile at given coordinates.
         * 
         * @param out_name Output file name.
         * @param p_val1 Value of the first coordinate.
         * @param p_type1 Type fo the first coordinate.
         * @param p_val2 Value of the second coordinate.
         * @param p_type2 Value of the second coordinate.
         */
        void Export1DProfile(std::string out_name, double p_val1, thermophysics_type_e p_type1, double p_val2, thermophysics_type_e p_type2);

        /**
         * @brief Export 2D profile at given coordinate.
         * 
         * @param out_name Output file name.
         * @param p_val Value of the cutting coordinate.
         * @param p_type Type fo the cutting coordinate.
         */
        void Export2DProfile(std::string out_name, double p_val, thermophysics_type_e p_type);

        /**
         * @brief Export a customized profile at given coordinate.
         * 
         * @param out_name Output file name.
         * @param t Inquire temperature coordinates.
         * @param p Inquire pressure coordinates.
         * @param c Inquire composition coordinates.
         */
        void ExportCustomProfile(std::string out_name, const std::vector<double> &t, const std::vector<double> &p, const std::vector<double> &c);

        /**
         * @brief Get the data value from the reference grid.
         * 
         * @param rel_t Normalized temperature value.
         * @param rel_p Normalized pressure value.
         * @param rel_c Normalized compositional value.
         * @param val data value.
         */
        void GetValue(const double &rel_t, const double &rel_p, const double &rel_c, double &val);

        /**
         * @brief Get gradient of the data value respected to a parameter type. 
         * 
         * @param rel_t Normalized temperature value.
         * @param rel_p Normalized pressure value.
         * @param rel_c Normalized compositional value.
         * @param grad gradient value.
         * @param p_type Parameter type.
         */
        void GetGradient(const double &rel_t, const double &rel_p, const double &rel_c, double &grad, thermophysics_type_e p_type);  // 获取参数网格中任意位置的参数值相对于坐标轴的梯度值

        /**
         * @brief Calculate data array using normalized parameter arrays.
         * 
         * @param rel_t Normalized temperature array.
         * @param rel_p Normalized pressure array.
         * @param rel_c Normalized compositional array.
         * @param ret_val Calculated data array.
         */
        //void ForwardArrayRel(const array<double> &rel_t, const array<double> &rel_p, const array<double> &rel_c, array<double> &ret_val);

        /**
         * @brief Calculate model array using parameter arrays.
         * 
         * @param rel_t Temperature model array.
         * @param rel_p Pressure model array.
         * @param rel_c Compositional model array.
         * @param ret_val Calculated data array.
         * @param ret_idx Index of a subset of the termperature, pressure and compositional model arrays. 
         * This is used to exract a subset data array of the input model arrays.
         */
        void ForwardModel(const array<double> &t, const array<double> &p, const array<double> &c, array<double> &ret_val, array<size_t> *ret_idx = nullptr);

        /**
         * @brief Calculate model array using parameter arrays.
         * 
         * @param tpc Assembled array of the temperature, pressure and compositional models. Size of tpc must be three times of that of ret_val's.
         * @param ret_val Calculated data array.
         */
        void ForwardModel(const array<double> &tpc, array<double> &ret_val);

        /**
         * @brief Calculate normalized gradient array of the reference grid with respect to a parameter type.
         * 
         * @param t Temperature array.
         * @param p Pressure array.
         * @param c Compositional array.
         * @param ret_grad Calculated gradient array.
         * @param p_type Parameter type.
         */
        void ForwardModelGradientRel(const array<double> &t, const array<double> &p, const array<double> &c, array<double> &ret_grad, thermophysics_type_e p_type);

        /**
         * @brief Calculate normalized gradient array of the reference grid with respect to a parameter type.
         * 
         * @param tpc Assembled array of the temperature, pressure and compositional models. Size of tpc must be three times of that of ret_val's.
         * @param ret_grad Calculated gradient array.
         * @param p_type Parameter type.
         */
        void ForwardModelGradientRel(const array<double> &tpc, array<double> &ret_grad, thermophysics_type_e p_type);

        /**
         * @brief Calculate gradient array of the reference grid with respect to a parameter type using parameter arrays.
         * 
         * @param t Temperature array.
         * @param p Pressure array.
         * @param c Compositional array.
         * @param ret_grad Calculated gradient array.
         * @param p_type Parameter type.
         */
        void ForwardModelGradientAbs(const array<double> &t, const array<double> &p, const array<double> &c, array<double> &ret_grad, thermophysics_type_e p_type);

        /**
         * @brief Calculate gradient array of the reference grid with respect to a parameter type using parameter arrays.
         * 
         * @param tpc Assembled array of the temperature, pressure and compositional models. Size of tpc must be three times of that of ret_val's.
         * @param ret_grad Calculated gradient array.
         * @param p_type Parameter type.
         */
        void ForwardModelGradientAbs(const array<double> &tpc, array<double> &ret_grad, thermophysics_type_e p_type);

        double get_mini_temperature() const; ///< Get the minimal temperature value.
        double get_maxi_temperature() const; ///< Get the maximal temperature value.
        double get_mini_pressure() const; ///< Get the minimal pressure value.
        double get_maxi_pressure() const; ///< Get the maximal pressure value.
        double get_mini_mg() const; ///< Get the minimal Mg value.
        double get_maxi_mg() const; ///< Get the maximal Mg value.

    private:
        void BuildGrid(); // 内部函数 构建参考网络

        double linear_cube_interpolate(double x0,double y0,double z0,double dx,double dy,double dz,double x,double y,double z,
	        double d0,double d1,double d2,double d3,double d4,double d5,double d6,double d7);

        bool initialized_;
        std::string name_, unit_; // 参数网格的名称和网格数据的单位
        size_t t_size_, p_size_, c_size_, d_size_; // 温度坐标轴 压力坐标轴 镁数坐标轴 网格节点 的数量
        double t_min_, t_max_, dt_, rel_dt_; // 网格中温度坐标轴的最小值，最大值和间隔
        double p_min_, p_max_, dp_, rel_dp_; // 网格中压力坐标轴的最小值，最大值和间隔
        double c_min_, c_max_, dc_, rel_dc_; // 网格中镁数坐标轴的最小值，最大值和间隔
        double d_min_, d_max_; // 节点数据最小值 最大值
        array<double> data_, data_gt_, data_gp_, data_gc_; // 节点数据集 温度 压力 镁数 梯度数据集
    };
}

#endif // _GIST_THERMOPHYSIC_H