/********************************************************
 *     _____________________
 *    / ____/  _/ ___/_  __/
 *   / / __ / / \__ \ / /   
 *  / /_/ // / ___/ // /    
 *  \____/___//____//_/ 
 * Geophysical Inversions using Spherical Tetrahedral meshes (GIST)
 *
 * Copyright (c) 2022  Yi Zhang (yizhang-geo@zju.edu.cn)
 *
 * GIST is distributed under a dual licensing scheme. You can redistribute 
 * it and/or modify it under the terms of the GNU Affero General Public 
 * License (AGPL) as published by the Free Software Foundation, either version 
 * 3 of the License, or (at your option) any later version. You should have 
 * received a copy of the GNU Affero General Public License along with this 
 * program. If not, see <http://www.gnu.org/licenses/>.
 * 
 * If the terms and conditions of the AGPL v.3. would prevent you from using 
 * the GIST, please consider the option to obtain a commercial license for a 
 * fee. These licenses are offered by the original author, Yi Zhang. As a rule, 
 * licenses are provided "as-is", unlimited in time for a one time fee. Please 
 * send corresponding requests to: yizhang-geo@zju.edu.cn. Please do not forget 
 * to include some description of your company and the realm of its activities. 
 * Also add information on how to contact you by electronic and paper mail.
 ******************************************************/

#include "thermophysics.h"

void GIST::Thermophysics::SaveMsh(std::string filename)
{
    size_t tb_size = t_size_-1;
	size_t pb_size = p_size_-1;
	size_t cb_size = c_size_-1;

	// initiate regular mesh
	gctl::regular_mesh_3d out_mesh("RockPhys", "Physical Properties of Mantle Rocks",
		tb_size, pb_size, cb_size, 0.5/tb_size, 0.5/pb_size, 0.5/cb_size, 1.0/tb_size, 1.0/pb_size, 1.0/cb_size);

	gctl::meshdata *d_val = out_mesh.add_data(name_ + " " + unit_, gctl::NodeData, true, gctl::Scalar);
	gctl::meshdata *gt_val = out_mesh.add_data("rel-grad-t", gctl::NodeData, true, gctl::Scalar);
	gctl::meshdata *gp_val = out_mesh.add_data("rel-grad-p", gctl::NodeData, true, gctl::Scalar);
	gctl::meshdata *gc_val = out_mesh.add_data("rel-grad-c", gctl::NodeData, true, gctl::Scalar);
	gctl::meshdata *gt_val2 = out_mesh.add_data("abs-grad-t", gctl::NodeData, true, gctl::Scalar);
	gctl::meshdata *gp_val2 = out_mesh.add_data("abs-grad-p", gctl::NodeData, true, gctl::Scalar);
	gctl::meshdata *gc_val2 = out_mesh.add_data("abs-grad-c", gctl::NodeData, true, gctl::Scalar);

	gctl::array<double> *data_ptr = (gctl::array<double>*) d_val->get_datval_ptr();
	for (size_t i = 0; i < d_size_; i++)
	{
		data_ptr->at(i) = data_[i];
	}

	data_ptr = (gctl::array<double>*) gt_val->get_datval_ptr();
	for (size_t i = 0; i < d_size_; i++)
	{
		data_ptr->at(i) = data_gt_[i];
	}

	data_ptr = (gctl::array<double>*) gt_val2->get_datval_ptr();
	for (size_t i = 0; i < d_size_; i++)
	{
		data_ptr->at(i) = data_gt_[i]/(t_max_ - t_min_);
	}

	data_ptr = (gctl::array<double>*) gp_val->get_datval_ptr();
	for (size_t i = 0; i < d_size_; i++)
	{
		data_ptr->at(i) = data_gp_[i];
	}

	data_ptr = (gctl::array<double>*) gp_val2->get_datval_ptr();
	for (size_t i = 0; i < d_size_; i++)
	{
		data_ptr->at(i) = data_gp_[i]/(p_max_ - p_min_);
	}

	data_ptr = (gctl::array<double>*) gc_val->get_datval_ptr();
	for (size_t i = 0; i < d_size_; i++)
	{
		data_ptr->at(i) = data_gc_[i];
	}

	data_ptr = (gctl::array<double>*) gc_val2->get_datval_ptr();
	for (size_t i = 0; i < d_size_; i++)
	{
		data_ptr->at(i) = data_gc_[i]/(c_max_ - c_min_);
	}

	out_mesh.save_gmsh(filename, gctl::NodeData, gctl::OverWrite, gctl::NotPacked);
	return;
}