/********************************************************
 *     _____________________
 *    / ____/  _/ ___/_  __/
 *   / / __ / / \__ \ / /   
 *  / /_/ // / ___/ // /    
 *  \____/___//____//_/ 
 * Geophysical Inversions using Spherical Tetrahedral meshes (GIST)
 *
 * Copyright (c) 2022  Yi Zhang (yizhang-geo@zju.edu.cn)
 *
 * GIST is distributed under a dual licensing scheme. You can redistribute 
 * it and/or modify it under the terms of the GNU Affero General Public 
 * License (AGPL) as published by the Free Software Foundation, either version 
 * 3 of the License, or (at your option) any later version. You should have 
 * received a copy of the GNU Affero General Public License along with this 
 * program. If not, see <http://www.gnu.org/licenses/>.
 * 
 * If the terms and conditions of the AGPL v.3. would prevent you from using 
 * the GIST, please consider the option to obtain a commercial license for a 
 * fee. These licenses are offered by the original author, Yi Zhang. As a rule, 
 * licenses are provided "as-is", unlimited in time for a one time fee. Please 
 * send corresponding requests to: yizhang-geo@zju.edu.cn. Please do not forget 
 * to include some description of your company and the realm of its activities. 
 * Also add information on how to contact you by electronic and paper mail.
 ******************************************************/

#include "thermophysics.h"

void GIST::Thermophysics::ImportText(std::string filename)
{
    std::ifstream infile;
	gctl::open_infile(infile, filename, ".txt");
	infile >> name_ >> unit_;

	int num;
	double stval, edval;
	double t_st, t_ed, p_st, p_ed, c_st, c_ed;
	std::string type_str;
	std::string order_str[3];
	for (size_t i = 0; i < 3; i++)
	{
		infile >> type_str >> stval >> edval >> num;

		if (type_str == "T") {t_st = stval; t_ed = edval; t_size_ = num; order_str[i] = type_str;}
		else if (type_str == "P") {p_st = stval; p_ed = edval; p_size_ = num; order_str[i] = type_str;}
		else if (type_str == "C") {c_st = stval; c_ed = edval; c_size_ = num; order_str[i] = type_str;}
		else throw std::runtime_error("Invalid axis type. From GIST::Thermophysics::ImportText(...)");
	}

	t_min_ = std::min(t_st, t_ed); t_max_ = std::max(t_st, t_ed);
	p_min_ = std::min(p_st, p_ed); p_max_ = std::max(p_st, p_ed);
	c_min_ = std::min(c_st, c_ed); c_max_ = std::max(c_st, c_ed);

	dt_ = (t_max_ - t_min_)/(t_size_-1);
	dp_ = (p_max_ - p_min_)/(p_size_-1);
	dc_ = (c_max_ - c_min_)/(c_size_-1);

	rel_dt_ = 1.0/(t_size_-1);
	rel_dp_ = 1.0/(p_size_-1);
	rel_dc_ = 1.0/(c_size_-1);

	array<int> t_idx(t_size_), p_idx(p_size_), c_idx(c_size_);
	if (t_st < t_ed) sequence(t_idx, 0, 1);
	else sequence(t_idx, (int) t_size_ - 1, -1);

	if (p_st < p_ed) sequence(p_idx, 0, 1);
	else sequence(p_idx, (int) p_size_ - 1, -1);

	if (c_st < c_ed) sequence(c_idx, 0, 1);
	else sequence(c_idx, (int) c_size_ - 1, -1);

	d_size_ = t_size_ * p_size_ * c_size_;
	data_.resize(d_size_);
	d_min_ = 1e+30; d_max_ = -1e+30;

	// 注意这个我们将参考网络进行了归一化，同时记录原参考网络中的极值以便获取原始值
	int id;
	double tmp_d;
	if (order_str[0] == "T" && order_str[1] == "P" && order_str[2] == "C")
	{
		for (size_t k = 0; k < c_size_; k++)
		{
			for (size_t j = 0; j < p_size_; j++)
			{
				for (size_t i = 0; i < t_size_; i++)
				{
					id = c_idx[k]*p_size_*t_size_ + p_idx[j]*t_size_ + t_idx[i];	
					
					infile >> tmp_d;
					data_[id] = tmp_d;
					d_min_ = std::min(d_min_, data_[id]);
					d_max_ = std::max(d_max_, data_[id]);
				}
			}
		}
	}
	else if (order_str[0] == "P" && order_str[1] == "T" && order_str[2] == "C")
	{
		for (size_t k = 0; k < c_size_; k++)
		{
			for (size_t i = 0; i < t_size_; i++)
			{
				for (size_t j = 0; j < p_size_; j++)
				{
					id = c_idx[k]*p_size_*t_size_ + p_idx[j]*t_size_ + t_idx[i];
					
					infile >> tmp_d;
					data_[id] = tmp_d;
					d_min_ = std::min(d_min_, data_[id]);
					d_max_ = std::max(d_max_, data_[id]);
				}
			}
		}
	}
	else if (order_str[0] == "T" && order_str[1] == "C" && order_str[2] == "P")
	{
		for (size_t j = 0; j < p_size_; j++)
		{
			for (size_t k = 0; k < c_size_; k++)
			{
				for (size_t i = 0; i < t_size_; i++)
				{
					id = c_idx[k]*p_size_*t_size_ + p_idx[j]*t_size_ + t_idx[i];
					
					infile >> tmp_d;
					data_[id] = tmp_d;
					d_min_ = std::min(d_min_, data_[id]);
					d_max_ = std::max(d_max_, data_[id]);
				}
			}
		}
	}
	else if (order_str[0] == "C" && order_str[1] == "T" && order_str[2] == "P")
	{
		for (size_t j = 0; j < p_size_; j++)
		{
			for (size_t i = 0; i < t_size_; i++)
			{
				for (size_t k = 0; k < c_size_; k++)
				{
					id = c_idx[k]*p_size_*t_size_ + p_idx[j]*t_size_ + t_idx[i];
					
					infile >> tmp_d;
					data_[id] = tmp_d;
					d_min_ = std::min(d_min_, data_[id]);
					d_max_ = std::max(d_max_, data_[id]);
				}
			}
		}
	}
	else if (order_str[0] == "C" && order_str[1] == "P" && order_str[2] == "T")
	{
		for (size_t i = 0; i < t_size_; i++)
		{
			for (size_t j = 0; j < p_size_; j++)
			{
				for (size_t k = 0; k < c_size_; k++)
				{
					id = c_idx[k]*p_size_*t_size_ + p_idx[j]*t_size_ + t_idx[i];
					
					infile >> tmp_d;
					data_[id] = tmp_d;
					d_min_ = std::min(d_min_, data_[id]);
					d_max_ = std::max(d_max_, data_[id]);
				}
			}
		}
	}
	else if (order_str[0] == "P" && order_str[1] == "C" && order_str[2] == "T")
	{
		for (size_t i = 0; i < t_size_; i++)
		{
			for (size_t k = 0; k < c_size_; k++)
			{
				for (size_t j = 0; j < p_size_; j++)
				{
					id = c_idx[k]*p_size_*t_size_ + p_idx[j]*t_size_ + t_idx[i];
					
					infile >> tmp_d;
					data_[id] = tmp_d;
					d_min_ = std::min(d_min_, data_[id]);
					d_max_ = std::max(d_max_, data_[id]);
				}
			}
		}
	}
	else throw std::runtime_error("Invalid axis order. From GIST::Thermophysics::ImportText(...)");
	infile.close();

	// Build the reference grid.
	BuildGrid();
	return;
}