/********************************************************
 *     _____________________
 *    / ____/  _/ ___/_  __/
 *   / / __ / / \__ \ / /   
 *  / /_/ // / ___/ // /    
 *  \____/___//____//_/ 
 * Geophysical Inversions using Spherical Tetrahedral meshes (GIST)
 *
 * Copyright (c) 2022  Yi Zhang (yizhang-geo@zju.edu.cn)
 *
 * GIST is distributed under a dual licensing scheme. You can redistribute 
 * it and/or modify it under the terms of the GNU Affero General Public 
 * License (AGPL) as published by the Free Software Foundation, either version 
 * 3 of the License, or (at your option) any later version. You should have 
 * received a copy of the GNU Affero General Public License along with this 
 * program. If not, see <http://www.gnu.org/licenses/>.
 * 
 * If the terms and conditions of the AGPL v.3. would prevent you from using 
 * the GIST, please consider the option to obtain a commercial license for a 
 * fee. These licenses are offered by the original author, Yi Zhang. As a rule, 
 * licenses are provided "as-is", unlimited in time for a one time fee. Please 
 * send corresponding requests to: yizhang-geo@zju.edu.cn. Please do not forget 
 * to include some description of your company and the realm of its activities. 
 * Also add information on how to contact you by electronic and paper mail.
 ******************************************************/

#include "thermophysics.h"

void GIST::Thermophysics::GetGradient(const double &rel_t, const double &rel_p, const double &rel_c, double &grad, thermophysics_type_e p_type)
{
	if (rel_t < 0.0 || rel_t > 1.0 || rel_p < 0.0 || rel_p > 1.0 || rel_c < 0.0 || rel_c > 1.0)
	{
		throw gctl::runtime_error("Invalid inquire coordinates. From GIST::Thermophysics::GetGradient(...)");
	}

	size_t t_id = floor(rel_t/rel_dt_);
	size_t p_id = floor(rel_p/rel_dp_);
	size_t c_id = floor(rel_c/rel_dc_);

	// 如果索引到达边界则后撤一格
	if (t_id == t_size_-1) t_id -= 1;
	if (p_id == p_size_-1) p_id -= 1;
	if (c_id == c_size_-1) c_id -= 1;

    if (p_type == TempVal)
    {
        grad = linear_cube_interpolate(t_id*rel_dt_, p_id*rel_dp_, c_id*rel_dc_, rel_dt_, rel_dp_, rel_dc_, rel_t, rel_p, rel_c, 
            data_gt_[t_id+p_id*t_size_+c_id*t_size_*p_size_], 
            data_gt_[t_id+1+p_id*t_size_+c_id*t_size_*p_size_], 
            data_gt_[t_id+1+(p_id+1)*t_size_+c_id*t_size_*p_size_], 
            data_gt_[t_id+(p_id+1)*t_size_+c_id*t_size_*p_size_], 
            data_gt_[t_id+p_id*t_size_+(c_id+1)*t_size_*p_size_], 
            data_gt_[t_id+1+p_id*t_size_+(c_id+1)*t_size_*p_size_], 
            data_gt_[t_id+1+(p_id+1)*t_size_+(c_id+1)*t_size_*p_size_], 
            data_gt_[t_id+(p_id+1)*t_size_+(c_id+1)*t_size_*p_size_]);
    }
    else if (p_type == PressVal)
    {
        grad = linear_cube_interpolate(t_id*rel_dt_, p_id*rel_dp_, c_id*rel_dc_, rel_dt_, rel_dp_, rel_dc_, rel_t, rel_p, rel_c, 
            data_gp_[t_id+p_id*t_size_+c_id*t_size_*p_size_], 
            data_gp_[t_id+1+p_id*t_size_+c_id*t_size_*p_size_], 
            data_gp_[t_id+1+(p_id+1)*t_size_+c_id*t_size_*p_size_], 
            data_gp_[t_id+(p_id+1)*t_size_+c_id*t_size_*p_size_], 
            data_gp_[t_id+p_id*t_size_+(c_id+1)*t_size_*p_size_], 
            data_gp_[t_id+1+p_id*t_size_+(c_id+1)*t_size_*p_size_], 
            data_gp_[t_id+1+(p_id+1)*t_size_+(c_id+1)*t_size_*p_size_], 
            data_gp_[t_id+(p_id+1)*t_size_+(c_id+1)*t_size_*p_size_]);
    }
    else if (p_type == CompVal)
    {
        grad = linear_cube_interpolate(t_id*rel_dt_, p_id*rel_dp_, c_id*rel_dc_, rel_dt_, rel_dp_, rel_dc_, rel_t, rel_p, rel_c, 
            data_gc_[t_id+p_id*t_size_+c_id*t_size_*p_size_], 
            data_gc_[t_id+1+p_id*t_size_+c_id*t_size_*p_size_], 
            data_gc_[t_id+1+(p_id+1)*t_size_+c_id*t_size_*p_size_], 
            data_gc_[t_id+(p_id+1)*t_size_+c_id*t_size_*p_size_], 
            data_gc_[t_id+p_id*t_size_+(c_id+1)*t_size_*p_size_], 
            data_gc_[t_id+1+p_id*t_size_+(c_id+1)*t_size_*p_size_], 
            data_gc_[t_id+1+(p_id+1)*t_size_+(c_id+1)*t_size_*p_size_], 
            data_gc_[t_id+(p_id+1)*t_size_+(c_id+1)*t_size_*p_size_]);
    }
    else throw std::runtime_error("Invalid thermophysical properity type. From GIST::Thermophysics::GetGradient(...).");
    return;
}