/********************************************************
 *     _____________________
 *    / ____/  _/ ___/_  __/
 *   / / __ / / \__ \ / /   
 *  / /_/ // / ___/ // /    
 *  \____/___//____//_/ 
 * Geophysical Inversions using Spherical Tetrahedral meshes (GIST)
 *
 * Copyright (c) 2022  Yi Zhang (yizhang-geo@zju.edu.cn)
 *
 * GIST is distributed under a dual licensing scheme. You can redistribute 
 * it and/or modify it under the terms of the GNU Affero General Public 
 * License (AGPL) as published by the Free Software Foundation, either version 
 * 3 of the License, or (at your option) any later version. You should have 
 * received a copy of the GNU Affero General Public License along with this 
 * program. If not, see <http://www.gnu.org/licenses/>.
 * 
 * If the terms and conditions of the AGPL v.3. would prevent you from using 
 * the GIST, please consider the option to obtain a commercial license for a 
 * fee. These licenses are offered by the original author, Yi Zhang. As a rule, 
 * licenses are provided "as-is", unlimited in time for a one time fee. Please 
 * send corresponding requests to: yizhang-geo@zju.edu.cn. Please do not forget 
 * to include some description of your company and the realm of its activities. 
 * Also add information on how to contact you by electronic and paper mail.
 ******************************************************/

#include "thermophysics.h"

void GIST::Thermophysics::ForwardModelGradientRel(const array<double> &t, const array<double> &p, const array<double> &c, 
    array<double> &ret_grad, thermophysics_type_e p_type)
{
    if (t.size() != p.size() || p.size() != c.size())
    {
        throw std::runtime_error("Incompatible array sizes. From GIST::Thermophysics::ForwardModelGradientRel(...).");
    }

    size_t phys_num = t.size();
    ret_grad.resize(phys_num);

    double rel_t, rel_p, rel_c;
    for (size_t i = 0; i < phys_num; i++)
    {
        if (t[i] >= t_max_) rel_t = 1.0;
        else if (t[i] <= t_min_) rel_t = 0.0;
        else rel_t = (t[i] - t_min_)/(t_max_ - t_min_);

        if (p[i] >= p_max_) rel_p = 1.0;
        else if (p[i] <= p_min_) rel_p = 0.0;
        else rel_p = (p[i] - p_min_)/(p_max_ - p_min_);

        if (c[i] >= c_max_) rel_c = 1.0;
        else if (c[i] <= c_min_) rel_c = 0.0;
        else rel_c = (c[i] - c_min_)/(c_max_ - c_min_);

        GetGradient(rel_t, rel_p, rel_c, ret_grad[i], p_type);
    }

    //normalize(ret_grad, 1.0, L2);
    return;
}

void GIST::Thermophysics::ForwardModelGradientRel(const array<double> &tpc, array<double> &ret_grad, thermophysics_type_e p_type)
{
    size_t phys_num = round(tpc.size()/3.0);
    ret_grad.resize(phys_num);

    double t, p, c, rel_t, rel_p, rel_c;
    for (size_t i = 0; i < phys_num; i++)
    {
        t = tpc[i];
        p = tpc[i + phys_num];
        c = tpc[i + 2*phys_num];

        if (t >= t_max_) rel_t = 1.0;
        else if (t <= t_min_) rel_t = 0.0;
        else rel_t = (t - t_min_)/(t_max_ - t_min_);

        if (p >= p_max_) rel_p = 1.0;
        else if (p <= p_min_) rel_p = 0.0;
        else rel_p = (p - p_min_)/(p_max_ - p_min_);

        if (c >= c_max_) rel_c = 1.0;
        else if (c <= c_min_) rel_c = 0.0;
        else rel_c = (c - c_min_)/(c_max_ - c_min_);

        GetGradient(rel_t, rel_p, rel_c, ret_grad[i], p_type);
    }
    return;
}

void GIST::Thermophysics::ForwardModelGradientAbs(const array<double> &t, const array<double> &p, const array<double> &c, 
    array<double> &ret_grad, thermophysics_type_e p_type)
{
    ForwardModelGradientRel(t, p, c, ret_grad, p_type);

    size_t phys_num = t.size();
    for (size_t i = 0; i < phys_num; i++)
    {
        // Convert to absoulte gradient value
        if (p_type == GIST::TempVal)
        {
            ret_grad[i] /= (t_max_ - t_min_);
        }
        else if (p_type == GIST::PressVal)
        {
            ret_grad[i] /= (p_max_ - p_min_);
        }
        else if (p_type == GIST::CompVal)
        {
            ret_grad[i] /= (c_max_ - c_min_);
        }
        else throw std::runtime_error("Invalid inquire type. From GIST::Thermophysics::ForwardModelGradientAbs(...).");
    }
    return;
}

void GIST::Thermophysics::ForwardModelGradientAbs(const array<double> &tpc, array<double> &ret_grad, thermophysics_type_e p_type)
{
    ForwardModelGradientRel(tpc, ret_grad, p_type);

    size_t phys_num = ret_grad.size();
    for (size_t i = 0; i < phys_num; i++)
    {
        // Convert to absoulte gradient value
        if (p_type == GIST::TempVal)
        {
            ret_grad[i] /= (t_max_ - t_min_);
        }
        else if (p_type == GIST::PressVal)
        {
            ret_grad[i] /= (p_max_ - p_min_);
        }
        else if (p_type == GIST::CompVal)
        {
            ret_grad[i] /= (c_max_ - c_min_);
        }
        else throw std::runtime_error("Invalid inquire type. From GIST::Thermophysics::ForwardModelGradientAbs(...).");
    }
    return;
}