/********************************************************
 *     _____________________
 *    / ____/  _/ ___/_  __/
 *   / / __ / / \__ \ / /   
 *  / /_/ // / ___/ // /    
 *  \____/___//____//_/ 
 * Geophysical Inversions using Spherical Tetrahedral meshes (GIST)
 *
 * Copyright (c) 2022  Yi Zhang (yizhang-geo@zju.edu.cn)
 *
 * GIST is distributed under a dual licensing scheme. You can redistribute 
 * it and/or modify it under the terms of the GNU Affero General Public 
 * License (AGPL) as published by the Free Software Foundation, either version 
 * 3 of the License, or (at your option) any later version. You should have 
 * received a copy of the GNU Affero General Public License along with this 
 * program. If not, see <http://www.gnu.org/licenses/>.
 * 
 * If the terms and conditions of the AGPL v.3. would prevent you from using 
 * the GIST, please consider the option to obtain a commercial license for a 
 * fee. These licenses are offered by the original author, Yi Zhang. As a rule, 
 * licenses are provided "as-is", unlimited in time for a one time fee. Please 
 * send corresponding requests to: yizhang-geo@zju.edu.cn. Please do not forget 
 * to include some description of your company and the realm of its activities. 
 * Also add information on how to contact you by electronic and paper mail.
 ******************************************************/

#include "thermophysics.h"

void GIST::Thermophysics::ForwardModel(const array<double> &t, const array<double> &p, const array<double> &c, array<double> &ret_val, array<size_t> *ret_idx)
{
    if (t.size() != p.size() || p.size() != c.size())
    {
        throw std::runtime_error("Incompatible array sizes for forward modeling thermophysics models.");
    }

    size_t phys_num;
    double rel_t, rel_p, rel_c;
    if (ret_idx == nullptr)
    {
        phys_num = t.size();
        ret_val.resize(phys_num);

        for (size_t i = 0; i < phys_num; i++)
        {
            if (t[i] >= t_max_) rel_t = 1.0;
            else if (t[i] <= t_min_) rel_t = 0.0;
            else rel_t = (t[i] - t_min_)/(t_max_ - t_min_);

            if (p[i] >= p_max_) rel_p = 1.0;
            else if (p[i] <= p_min_) rel_p = 0.0;
            else rel_p = (p[i] - p_min_)/(p_max_ - p_min_);

            if (c[i] >= c_max_) rel_c = 1.0;
            else if (c[i] <= c_min_) rel_c = 0.0;
            else rel_c = (c[i] - c_min_)/(c_max_ - c_min_);

            GetValue(rel_t, rel_p, rel_c, ret_val[i]);
        }
    }
    else
    {
        phys_num = ret_idx->size();
        ret_val.resize(phys_num);

        size_t id;
        for (size_t i = 0; i < phys_num; i++)
        {
            id = ret_idx->at(i);

            if (t[id] >= t_max_) rel_t = 1.0;
            else if (t[id] <= t_min_) rel_t = 0.0;
            else rel_t = (t[id] - t_min_)/(t_max_ - t_min_);

            if (p[id] >= p_max_) rel_p = 1.0;
            else if (p[id] <= p_min_) rel_p = 0.0;
            else rel_p = (p[id] - p_min_)/(p_max_ - p_min_);

            if (c[id] >= c_max_) rel_c = 1.0;
            else if (c[id] <= c_min_) rel_c = 0.0;
            else rel_c = (c[id] - c_min_)/(c_max_ - c_min_);

            GetValue(rel_t, rel_p, rel_c, ret_val[i]);
        }
    }
    return;
}

void GIST::Thermophysics::ForwardModel(const array<double> &tpc, array<double> &ret_val)
{
    size_t phys_num = round(tpc.size()/3.0);
    ret_val.resize(phys_num);

    double t, p, c, rel_t, rel_p, rel_c;
    for (size_t i = 0; i < phys_num; i++)
    {
        t = tpc[i];
        p = tpc[i + phys_num];
        c = tpc[i + 2*phys_num];

        if (t >= t_max_) rel_t = 1.0;
        else if (t <= t_min_) rel_t = 0.0;
        else rel_t = (t - t_min_)/(t_max_ - t_min_);

        if (p >= p_max_) rel_p = 1.0;
        else if (p <= p_min_) rel_p = 0.0;
        else rel_p = (p - p_min_)/(p_max_ - p_min_);

        if (c >= c_max_) rel_c = 1.0;
        else if (c <= c_min_) rel_c = 0.0;
        else rel_c = (c - c_min_)/(c_max_ - c_min_);

        GetValue(rel_t, rel_p, rel_c, ret_val[i]);
    }
    return;
}