/********************************************************
 *     _____________________
 *    / ____/  _/ ___/_  __/
 *   / / __ / / \__ \ / /   
 *  / /_/ // / ___/ // /    
 *  \____/___//____//_/ 
 * Geophysical Inversions using Spherical Tetrahedral meshes (GIST)
 *
 * Copyright (c) 2022  Yi Zhang (yizhang-geo@zju.edu.cn)
 *
 * GIST is distributed under a dual licensing scheme. You can redistribute 
 * it and/or modify it under the terms of the GNU Affero General Public 
 * License (AGPL) as published by the Free Software Foundation, either version 
 * 3 of the License, or (at your option) any later version. You should have 
 * received a copy of the GNU Affero General Public License along with this 
 * program. If not, see <http://www.gnu.org/licenses/>.
 * 
 * If the terms and conditions of the AGPL v.3. would prevent you from using 
 * the GIST, please consider the option to obtain a commercial license for a 
 * fee. These licenses are offered by the original author, Yi Zhang. As a rule, 
 * licenses are provided "as-is", unlimited in time for a one time fee. Please 
 * send corresponding requests to: yizhang-geo@zju.edu.cn. Please do not forget 
 * to include some description of your company and the realm of its activities. 
 * Also add information on how to contact you by electronic and paper mail.
 ******************************************************/

#include "thermophysics.h"

void GIST::Thermophysics::Export1DProfile(std::string out_name, double p_val1, thermophysics_type_e p_type1, double p_val2, thermophysics_type_e p_type2)
{
    double temp_val, press_val, comp_val;
    bool temp_set = false, press_set = false, comp_set = false;

    if (p_type1 == GIST::TempVal) {temp_val = p_val1; temp_set = true;}
    else if (p_type1 == GIST::PressVal) {press_val = p_val1; press_set = true;}
    else if (p_type1 == GIST::CompVal) {comp_val = p_val1; comp_set = true;}
    else throw std::runtime_error("Invalid inquire type. From GIST::Thermophysics::Export1DProfile(...)");

    if (p_type2 == GIST::TempVal) {temp_val = p_val2; temp_set = true;}
    else if (p_type2 == GIST::PressVal) {press_val = p_val2; press_set = true;}
    else if (p_type2 == GIST::CompVal) {comp_val = p_val2; comp_set = true;}
    else throw std::runtime_error("Invalid inquire type. From GIST::Thermophysics::Export1DProfile(...)");

    std::ofstream ofile;
    open_outfile(ofile, out_name, ".txt");

    double rel_t, rel_p, rel_c, out_val;

    if (!temp_set && press_set && comp_set)
    {
        if (press_val >= p_max_) rel_p = 1.0;
        else if (press_val <= p_min_) rel_p = 0.0;
        else rel_p = (press_val - p_min_)/(p_max_ - p_min_);

        if (comp_val >= c_max_) rel_c = 1.0;
        else if (comp_val <= c_min_) rel_c = 0.0;
        else rel_c = (comp_val - c_min_)/(c_max_ - c_min_);

        ofile << "# Pressure(GPa) = " << press_val << ", Mg# = " << comp_val << "\n";
        ofile << "# Temperature(K) " << name_ << "(" << unit_ << ")\n";
        for (size_t i = 0; i < t_size_; i++)
        {
            GetValue(i*dt_/(t_max_ - t_min_), rel_p, rel_c, out_val);
            ofile << t_min_ + i*dt_ << " " << out_val << "\n";
        }

        ofile.close();
        return;
    }
    
    if (!press_set && temp_set && comp_set)
    {
        if (temp_val >= t_max_) rel_t = 1.0;
        else if (temp_val <= t_min_) rel_t = 0.0;
        else rel_t = (temp_val - t_min_)/(t_max_ - t_min_);

        if (comp_val >= c_max_) rel_c = 1.0;
        else if (comp_val <= c_min_) rel_c = 0.0;
        else rel_c = (comp_val - c_min_)/(c_max_ - c_min_);

        ofile << "# Temperature(K) = " << temp_val << ", Mg# = " << comp_val << "\n";
        ofile << "# Pressure(GPa) " << name_ << "(" << unit_ << ")\n";
        for (size_t i = 0; i < p_size_; i++)
        {
            GetValue(rel_t, i*dp_/(p_max_ - p_min_), rel_c, out_val);
            ofile << p_min_ + i*dp_ << " " << out_val << "\n";
        }

        ofile.close();
        return;
    }
    
    if (!comp_set && temp_set && press_set)
    {
        if (temp_val >= t_max_) rel_t = 1.0;
        else if (temp_val <= t_min_) rel_t = 0.0;
        else rel_t = (temp_val - t_min_)/(t_max_ - t_min_);

        if (press_val >= p_max_) rel_p = 1.0;
        else if (press_val <= p_min_) rel_p = 0.0;
        else rel_p = (press_val - p_min_)/(p_max_ - p_min_);

        ofile << "# Temperature(K) = " << temp_val << ", Pressure(GPa) = " << press_val << "\n";
        ofile << "# Mg# " << name_ << "(" << unit_ << ")\n";
        for (size_t i = 0; i < c_size_; i++)
        {
            GetValue(rel_t, rel_p, i*dc_/(c_max_ - c_min_), out_val);
            ofile << c_min_ + i*dc_ << " " << out_val << "\n";
        }

        ofile.close();
        return;
    }

    ofile.close();
    throw std::runtime_error("Invalid coordinates to export 1D profile. From GIST::Thermophysics::Export1DProfile(...)");
    return;
}