/********************************************************
 *     _____________________
 *    / ____/  _/ ___/_  __/
 *   / / __ / / \__ \ / /   
 *  / /_/ // / ___/ // /    
 *  \____/___//____//_/ 
 * Geophysical Inversions using Spherical Tetrahedral meshes (GIST)
 *
 * Copyright (c) 2022  Yi Zhang (yizhang-geo@zju.edu.cn)
 *
 * GIST is distributed under a dual licensing scheme. You can redistribute 
 * it and/or modify it under the terms of the GNU Affero General Public 
 * License (AGPL) as published by the Free Software Foundation, either version 
 * 3 of the License, or (at your option) any later version. You should have 
 * received a copy of the GNU Affero General Public License along with this 
 * program. If not, see <http://www.gnu.org/licenses/>.
 * 
 * If the terms and conditions of the AGPL v.3. would prevent you from using 
 * the GIST, please consider the option to obtain a commercial license for a 
 * fee. These licenses are offered by the original author, Yi Zhang. As a rule, 
 * licenses are provided "as-is", unlimited in time for a one time fee. Please 
 * send corresponding requests to: yizhang-geo@zju.edu.cn. Please do not forget 
 * to include some description of your company and the realm of its activities. 
 * Also add information on how to contact you by electronic and paper mail.
 ******************************************************/

#include "thermophysics.h"

void GIST::Thermophysics::BuildGrid()
{
	// Check parameters
	// We use the central differential equation to calculate the spatial gradients. The grid size must be at least three. 
	if (t_size_ <= 3 || p_size_ <= 3 || c_size_ <= 3 || rel_dt_ <= 0 || rel_dp_ <= 0 || rel_dc_ <= 0)
	{
		throw gctl::invalid_argument("Invalid grid sizes. From GIST::Thermophysics::BuildGrid(...).");
	}

	data_gt_.resize(d_size_);
	data_gp_.resize(d_size_);
	data_gc_.resize(d_size_);

	// use forward and backward difference for boundary nodes
	for (size_t j = 0; j < p_size_; j++)
	{
		for (size_t i = 0; i < t_size_; i++)
		{
			data_gc_[i+j*t_size_] = (-3.0*data_[i+j*t_size_] 
				+ 4.0*data_[i+j*t_size_+t_size_*p_size_] 
				- data_[i+j*t_size_+2*t_size_*p_size_])/(2.0*rel_dc_);
			data_gc_[i+j*t_size_+(c_size_-1)*t_size_*p_size_] = (3.0*data_[i+j*t_size_+(c_size_-1)*t_size_*p_size_] 
				- 4.0*data_[i+j*t_size_+(c_size_-2)*t_size_*p_size_] 
				+ data_[i+j*t_size_+(c_size_-3)*t_size_*p_size_])/(2.0*rel_dc_);
		}
	}

	for (size_t k = 0; k < c_size_; k++)
	{
		for (size_t j = 0; j < p_size_; j++)
		{
			data_gt_[j*t_size_+k*t_size_*p_size_] = (-3.0*data_[j*t_size_+k*t_size_*p_size_] + 
				+ 4.0*data_[1+j*t_size_+k*t_size_*p_size_] 
				- data_[2+j*t_size_+k*t_size_*p_size_])/(2.0*rel_dt_);
			data_gt_[t_size_-1+j*t_size_+k*t_size_*p_size_] = (3.0*data_[t_size_-1+j*t_size_+k*t_size_*p_size_] 
				- 4.0*data_[t_size_-2+j*t_size_+k*t_size_*p_size_] 
				+ data_[t_size_-3+j*t_size_+k*t_size_*p_size_])/(2.0*rel_dt_);
		}
	}

	for (size_t k = 0; k < c_size_; k++)
	{
		for (size_t i = 0; i < t_size_; i++)
		{
			data_gp_[i+k*t_size_*p_size_] = (-3.0*data_[i+k*t_size_*p_size_] 
				+ 4.0*data_[i+1*t_size_+k*t_size_*p_size_] 
				- data_[i+2*t_size_+k*t_size_*p_size_])/(2.0*rel_dp_);
			data_gp_[i+(p_size_-1)*t_size_+k*t_size_*p_size_] = (3.0*data_[i+(p_size_-1)*t_size_+k*t_size_*p_size_] 
				- 4.0*data_[i+(p_size_-2)*t_size_+k*t_size_*p_size_] 
				+ data_[i+(p_size_-3)*t_size_+k*t_size_*p_size_])/(2.0*rel_dp_);
		}
	}

	// use central difference for interior nodes
	for (size_t k = 1; k < c_size_-1; k++)
	{
		for (size_t j = 0; j < p_size_; j++)
		{
			for (size_t i = 0; i < t_size_; i++)
			{
				data_gc_[i+j*t_size_+k*t_size_*p_size_] = (data_[i+j*t_size_+(k+1)*t_size_*p_size_] 
					- data_[i+j*t_size_+(k-1)*t_size_*p_size_])/(2.0*rel_dc_);
			}
		}
	}

	for (size_t k = 0; k < c_size_; k++)
	{
		for (size_t j = 0; j < p_size_; j++)
		{
			for (size_t i = 1; i < t_size_-1; i++)
			{
				data_gt_[i+j*t_size_+k*t_size_*p_size_] = (data_[i+1+j*t_size_+k*t_size_*p_size_] 
					- data_[i-1+j*t_size_+k*t_size_*p_size_])/(2.0*rel_dt_);
			}
		}
	}

	for (size_t k = 0; k < c_size_; k++)
	{
		for (size_t j = 1; j < p_size_-1; j++)
		{
			for (size_t i = 0; i < t_size_; i++)
			{
				data_gp_[i+j*t_size_+k*t_size_*p_size_] = (data_[i+(j+1)*t_size_+k*t_size_*p_size_] 
					- data_[i+(j-1)*t_size_+k*t_size_*p_size_])/(2.0*rel_dp_);
			}
		}
	}

	return;
}