/********************************************************
 *     _____________________
 *    / ____/  _/ ___/_  __/
 *   / / __ / / \__ \ / /   
 *  / /_/ // / ___/ // /    
 *  \____/___//____//_/ 
 * Geophysical Inversions using Spherical Tetrahedral meshes (GIST)
 *
 * Copyright (c) 2022  Yi Zhang (yizhang-geo@zju.edu.cn)
 *
 * GIST is distributed under a dual licensing scheme. You can redistribute 
 * it and/or modify it under the terms of the GNU Affero General Public 
 * License (AGPL) as published by the Free Software Foundation, either version 
 * 3 of the License, or (at your option) any later version. You should have 
 * received a copy of the GNU Affero General Public License along with this 
 * program. If not, see <http://www.gnu.org/licenses/>.
 * 
 * If the terms and conditions of the AGPL v.3. would prevent you from using 
 * the GIST, please consider the option to obtain a commercial license for a 
 * fee. These licenses are offered by the original author, Yi Zhang. As a rule, 
 * licenses are provided "as-is", unlimited in time for a one time fee. Please 
 * send corresponding requests to: yizhang-geo@zju.edu.cn. Please do not forget 
 * to include some description of your company and the realm of its activities. 
 * Also add information on how to contact you by electronic and paper mail.
 ******************************************************/

#include "thermal.h"

double GIST::Thermal::ThermalObjectiveFunc(const array<double> &pphys_thermal, const array<double> &fem_thermal, 
    const array<double> &pphys_grad, array<double> &out_grad)
{
    if (!err_ready_)
    {
        throw std::runtime_error("[GIST::Thermal::ThermalObjectiveFunc] Thermal uncertainty not set.");
    }

    elem_thermal_.resize(elem_num_); // resize the array for element-wise thermal model.
    m_space_->Node2Element(fem_thermal, elem_thermal_);  // get the element-wise thermal model.

    // Calculate the model gradient of elem_thermal_ w.r.t. the pphys_themal_ using the joint-matrix algorithm.
    elem_diff_.resize(elem_num_);
    for (size_t i = 0; i < elem_num_; i++)
    {
        elem_diff_[i] = 2.0*(pphys_thermal[i] - elem_thermal_[i])/(thermal_err_*thermal_err_*elem_num_);
    }

    // Calculate the model gradient of FEM temperature w.r.t. the pertrophysical ones using the joint-matrix algorithm.
    obs_grad_.resize(node_num_);
    m_space_->Element2Node(elem_diff_, obs_grad_);

    // Since boundary values are fixed in the FEM calculation. There are no gradients at these nodes.
    for (size_t i = 0; i < node_num_; i++)
    {
        if (has_val_cndt_[i])
        {
            obs_grad_[i] = 0.0;
        }
    }

    node_grad_.resize(node_num_, 0.0);
    // 预优矩阵已在solve函数中初始化
    LCG_Minimize(node_grad_, obs_grad_, gctl::LCG_PCG);

    // 计算有限元系统相对于模型热导率的梯度 注意在有限元上下顶面均施加第一类边界条件时有限元系统右端项无相对于模型热导率的梯度
    out_grad.resize(elem_num_, 0.0);

    size_t r_id, c_id;
    double node_spt;
    for (size_t i = 0; i < elem_num_; i++)
	{
		for (size_t j = 0; j < 4; j++)
		{
            r_id = kernel_list_[16*i + 4*j].r_id;

            node_spt = 0.0; // 右端项无相对于模型热导率的梯度为0
			for (size_t f = 0; f < 4; f++)
			{
                // 减去核矩阵相对于热导率的梯度 即kernel_list_元素值本身
                c_id = kernel_list_[16*i + 4*j + f].c_id;
                node_spt -= fem_thermal[c_id] * kernel_list_[16*i + 4*j + f].val;
			}

            out_grad[i] += node_spt*node_grad_[r_id];
		}
	}

    for (size_t i = 0; i < elem_num_; i++)
    {
        out_grad[i] = 2.0*(pphys_thermal[i] - elem_thermal_[i])/(thermal_err_*thermal_err_*elem_num_) - out_grad[i]*pphys_grad[i];
    }

    double msf = 0.0;
    for (size_t i = 0; i < elem_num_; i++)
    {
        msf += pow((pphys_thermal[i] - elem_thermal_[i])/thermal_err_, 2);
    }
    return msf/elem_num_;
}