/********************************************************
 *     _____________________
 *    / ____/  _/ ___/_  __/
 *   / / __ / / \__ \ / /   
 *  / /_/ // / ___/ // /    
 *  \____/___//____//_/ 
 * Geophysical Inversions using Spherical Tetrahedral meshes (GIST)
 *
 * Copyright (c) 2022  Yi Zhang (yizhang-geo@zju.edu.cn)
 *
 * GIST is distributed under a dual licensing scheme. You can redistribute 
 * it and/or modify it under the terms of the GNU Affero General Public 
 * License (AGPL) as published by the Free Software Foundation, either version 
 * 3 of the License, or (at your option) any later version. You should have 
 * received a copy of the GNU Affero General Public License along with this 
 * program. If not, see <http://www.gnu.org/licenses/>.
 * 
 * If the terms and conditions of the AGPL v.3. would prevent you from using 
 * the GIST, please consider the option to obtain a commercial license for a 
 * fee. These licenses are offered by the original author, Yi Zhang. As a rule, 
 * licenses are provided "as-is", unlimited in time for a one time fee. Please 
 * send corresponding requests to: yizhang-geo@zju.edu.cn. Please do not forget 
 * to include some description of your company and the realm of its activities. 
 * Also add information on how to contact you by electronic and paper mail.
 ******************************************************/

#ifndef _GIST_THERMAL_H
#define _GIST_THERMAL_H

#include "gctl/core.h"
#include "gctl/maths.h"
#include "gctl/algorithm.h"
#include "gctl/geometry.h"
#include "gctl/io.h"
#include "gctl/optimization.h"

#include "../utility/enum.h"
#include "../utility/data_type.h"
#include "../earth_1d/earth_1d.h"
#include "../model_space/model_space.h"

#include "thermal_conductivity_JGR2022.h"

using namespace gctl;

namespace GIST
{
    enum thermal_FEM_type
    {
        ThermalKernel,
        HeatProduct,
        HeatFlow,
        //TransferKernel,
        //TransferForce,
    };

    enum thermal_solve_type
    {
        SolveFEM,
        SolveCFEM,
    };

    struct thermal_FEM_setup
    {
        size_t uv_id[2];
        thermal_FEM_type i_type;
    };
    
    class Thermal : public gctl::lcg_solver, public gctl::glni::tetrahedron, public gctl::glni::triangle
    {
    public:
        Thermal();
        virtual ~Thermal();
        virtual void LCG_Mx(const array<double> &x, array<double> &ax); // LCG预优函数
        virtual void LCG_Ax(const array<double> &x, array<double> &ax); // LCG核函数
        virtual double tetrahedron_func(double ksi, double eta, double zta, 
            double x1, double x2, double x3, double x4, 
            double y1, double y2, double y3, double y4,
            double z1, double z2, double z3, double z4, void *att);
        virtual double triangle_func(double x, double y, 
            double x1, double x2, double x3, 
            double y1, double y2, double y3, void *att);

        /**
         * @brief 初始化有限元热系统
         * 
         * @param m_space 模型空间的指针
         * @param earth_1d 一维参考模型的指针
         */
        void ImportModelSetup(ModelSpace *m_space, Earth1D *earth_1d);

        /**
         * @brief 组合有限元线性系统
         * 
         * @param cndt 热导率模型
         */
        void AssembleLinearSystem(const array<double> &cndt);

        /**
         * @brief 求解有限元的系统
         * 
         * @param t 求解所得温度模型
         */
        void Solve(array<double> &t);

        /**
         * @brief 求解正则化的有限元系统
         * 
         * @param ref_t 参考温度模型
         * @param t 求解所得温度模型
         */
        void Solve(const array<double> &ref_t, array<double> &t);
        
        /**
         * @brief 初始化（再初始化）边界条件数组
         * 
         */
        void ResetBoundaryValue();
        
        /**
         * @brief 由顶点索引数组设置第一类边界（边界上的值）
         * 
         * @param node_idx 边界上的顶点索引
         * @param bnd_val 边界值
         */
        void SetBoundaryValue(const array<size_t> &node_idx, double node_val);
        
        /**
         * @brief 由顶点索引数组设置第一类边界（边界上的值）
         * 
         * @param node_idx 边界上的顶点索引
         * @param bnd_vals 边界值
         */
        void SetBoundaryValue(const array<size_t> &node_idx, const array<double> &bnd_vals);
        
        /**
         * @brief 初始化（再初始化）热梯度边界条件数组
         * 
         */
        void ResetBoundaryHeatFlow();
        
        /**
         * @brief 由面索引数组设置第二类边界（边界上的热流值）
         * 
         * @param fac_idx 边界上的面索引
         * @param hf_val 边界上的热流值
         */
        void SetBoundaryHeatFlow(const array<size_t> &fac_idx, double hf_val);
        
        /**
         * @brief 由面索引数组设置第二类边界（边界上的热流值）
         * 
         * @param fac_idx 边界上的面索引
         * @param bnd_vals 定义在顶点上的整个模型的热流值
         */
        void SetBoundaryHeatFlow(const array<size_t> &fac_idx, const array<double> &bnd_vals);
        
        /**
         * @brief 初始化（再初始化）热传导边界条件数组
         * 
         */
        //void ResetBoundaryTransfer();
        
        /**
         * @brief 由面索引数组设置第三类边界（边界上的值和梯度值）
         * 
         * @param fac_idx 边界上的面索引
         * @param hf_val 边界上的热流值
         * @param temper_val 边界上的温度值
         */
        //void SetBoundaryTransfer(const array<size_t> &fac_idx, double hf_val, double temper_val);
        
        /**
         * @brief 初始化（再初始化）热产率条件数组
         * 
         */
        void ResetHeatProductivity();
        
        /**
         * @brief 由元素索引数组设置热产率
         * 
         * @param ele_idx 产热元素的索引
         * @param hp_val 产热率
         */
        void SetHeatProductivity(const array<size_t> &ele_idx, double hp_val); 
        
        /**
         * @brief 插值温度的热导率
         * 
         * @param cndt 温度模型的热导率 (大小为元素个数)
         * @param t 温度模型（大小为顶点个数）
         * @param hf 插值后的热导率
         */
        void BodyHeatFlow(const array<double> &cndt, const array<double> &t, array<double> &hf);
        
        /**
         * @brief 初始化大地热流的观测点位
         * 
         * @note 需要在模型文件内标注观测点位
         * 
         * @param obs_name 观测点位的标签名称
         */
        void InitObsSites(std::string obs_name);
        
        /**
         * @brief 插值观测点位的大地热流值
         * 
         * @param cndt 温度模型的热导率
         * @param t 温度模型
         */
        void ObsHeatFlow(const array<double> &cndt, const array<double> &t);
        
        /**
         * @brief 读入大地热流观测文件
         * 
         * @param filename 数据文件名
         * @param off_set  定位观测点的允许误差
         */
        void ReadObs(std::string filename, double off_set = 1e-6);
        
        /**
         * @brief 保存正演的大地热流数据
         * 
         * @param filename 数据文件名
         */
        void SaveObs(std::string filename);

        /**
         * @brief Objective function of the squared L-2 differences between the thermophysical temperature and FEM calculated one.
         * 
         * @param pphys_thermal Thermophysical temperature.
         * @param fem_thermal FEM calculated temperature.
         * @param pphys_grad Gradient of the density value w.r.t. the temperature condition calculated using the thermophysical model.
         * @param out_grad Gradient of the thermophysical temperature w.r.t. the FEM calculated temperature.
         * @return Value of the objective function.
         */
        double ThermalObjectiveFunc(const array<double> &pphys_thermal, const array<double> &fem_thermal, 
            const array<double> &pphys_grad, array<double> &out_grad);
        
        /**
         * @brief 设置高斯积分的阶数
         * 
         * @param order 阶数
         */
        void set_gauss_order(int order);
        
        /**
         * @brief 设置温度模型的不确定度
         * 
         * @param err 不确定度
         */
        void set_thermal_error(double err);
        
        /**
         * @brief 返回温度模型的不确定度
         * 
         * @return 不确定度
         */
        double get_thermal_error();
        
        /**
         * @brief 设置温度模型约束求解中的正则化强度
         * 
         * @param val 正则化参数
         */
        void set_regularization_strength(double val);
        
        /**
         * @brief 返回观测点位数组
         * 
         * @return 观测点位数组 
         */
        array<point3ds> &get_obsp();

    protected:

        bool system_ready_, err_ready_, has_obs_;
        size_t node_num_, face_num_, elem_num_;

        std::vector<mat_node<double>> triplts_; // vector形式的三元组列表（要用到排序算法）
        array<mat_node<double>> kernel_list_; // 以三元组储存的核矩阵元素
        //array<mat_node<double>> trans_list_; // 热对流核函数三元组
        spmat<double> thermal_kernel_; // 有限元核矩阵（稀疏矩阵）

        size_t obs_num_;
        double thermal_err_;
        array<size_t> obs_idx_;
        array<point3dc*> obs_loc_;
        array<point3ds> obs_locS_;
        array<double> pre_hf_, obs_hf_;

        std::vector<std::vector<gctl::tetrahedron*> > node_hosts_, obs_hosts_;
        spmat<double> hf_kernel_;
        std::vector<double> hf_coeff_;

        array<bool> has_val_cndt_; // 是否存在第一类（边界值）边界条件
        //array<bool> has_trn_cndt_; // 是否存在热传导边界条件
        array<double> pdt_cndt_; // 产热边界条件数组
        array<double> val_cndt_; // 边界值数组
        array<double> grd_cndt_; // 热梯度边界条件数组
        //array<double> trn_cndt_; // 热传导边界条件数组
        array<double> tar_, cnst_tar_; // 有限元的目标向量 包含了边界温度 热流 及产热等信息
        array<double> elem_temper_, node_temper_;
        array<double> precndt_; // 预优矩阵

        gctl::linear_sf lsf_;

        array<double> elem_thermal_, elem_diff_, obs_grad_, node_grad_;

        ModelSpace *m_space_;
        Earth1D *earth_1d_;

        thermal_solve_type ts_type_;
        double reg_strength_;
    };
}

#endif // _GIST_THERMAL_H