/********************************************************
 *     _____________________
 *    / ____/  _/ ___/_  __/
 *   / / __ / / \__ \ / /   
 *  / /_/ // / ___/ // /    
 *  \____/___//____//_/ 
 * Geophysical Inversions using Spherical Tetrahedral meshes (GIST)
 *
 * Copyright (c) 2022  Yi Zhang (yizhang-geo@zju.edu.cn)
 *
 * GIST is distributed under a dual licensing scheme. You can redistribute 
 * it and/or modify it under the terms of the GNU Affero General Public 
 * License (AGPL) as published by the Free Software Foundation, either version 
 * 3 of the License, or (at your option) any later version. You should have 
 * received a copy of the GNU Affero General Public License along with this 
 * program. If not, see <http://www.gnu.org/licenses/>.
 * 
 * If the terms and conditions of the AGPL v.3. would prevent you from using 
 * the GIST, please consider the option to obtain a commercial license for a 
 * fee. These licenses are offered by the original author, Yi Zhang. As a rule, 
 * licenses are provided "as-is", unlimited in time for a one time fee. Please 
 * send corresponding requests to: yizhang-geo@zju.edu.cn. Please do not forget 
 * to include some description of your company and the realm of its activities. 
 * Also add information on how to contact you by electronic and paper mail.
 ******************************************************/

#include "thermal.h"

GIST::Thermal::Thermal()
{
    triangle::initiate(2);
    tetrahedron::initiate(2);

    system_ready_ = err_ready_ = has_obs_ = false;
    ts_type_ = SolveFEM;
    reg_strength_ = 0.0;
}

GIST::Thermal::~Thermal(){}

void GIST::Thermal::LCG_Mx(const array<double> &x, array<double> &ax)
{
    vecdiv(ax, x, precndt_);
    return;
}

void GIST::Thermal::LCG_Ax(const array<double> &x, array<double> &ax)
{
    thermal_kernel_.multiply_vector(x, ax);

    if (ts_type_ == SolveCFEM)
    {
        m_space_->Node2Element(x, elem_temper_);
		m_space_->Node2Element(elem_temper_, node_temper_, Trans);
        scale(node_temper_, reg_strength_);

		ax += node_temper_;
    }
    return;
}

double GIST::Thermal::tetrahedron_func(double ksi, double eta, double zta, 
    double x1, double x2, double x3, double x4, 
    double y1, double y2, double y3, double y4,
    double z1, double z2, double z3, double z4, void *att)
{
    thermal_FEM_setup *tt = static_cast<thermal_FEM_setup*>(att);
    if (tt->i_type == ThermalKernel)
    {
        return lsf_.tetrahedron(ksi, eta, zta, x1, x2, x3, x4, y1, y2, y3, y4, z1, z2, z3, z4, tt->uv_id[0], gctl::Dx)*lsf_.tetrahedron(ksi, eta, zta, x1, x2, x3, x4, y1, y2, y3, y4, z1, z2, z3, z4, tt->uv_id[1], gctl::Dx) 
            + lsf_.tetrahedron(ksi, eta, zta, x1, x2, x3, x4, y1, y2, y3, y4, z1, z2, z3, z4, tt->uv_id[0], gctl::Dy)*lsf_.tetrahedron(ksi, eta, zta, x1, x2, x3, x4, y1, y2, y3, y4, z1, z2, z3, z4, tt->uv_id[1], gctl::Dy) 
            + lsf_.tetrahedron(ksi, eta, zta, x1, x2, x3, x4, y1, y2, y3, y4, z1, z2, z3, z4, tt->uv_id[0], gctl::Dz)*lsf_.tetrahedron(ksi, eta, zta, x1, x2, x3, x4, y1, y2, y3, y4, z1, z2, z3, z4, tt->uv_id[1], gctl::Dz);
    }
    else if (tt->i_type == HeatProduct)
    {
        return lsf_.tetrahedron(ksi, eta, zta, x1, x2, x3, x4, y1, y2, y3, y4, z1, z2, z3, z4, tt->uv_id[0], gctl::Value);
    }
    else throw std::runtime_error("Invalid integral type. From GIST::Thermal::glni_tetrahedron_func(...)");
    return 0.0;
}

double GIST::Thermal::triangle_func(double x, double y, 
    double x1, double x2, double x3, 
    double y1, double y2, double y3, void *att)
{
    thermal_FEM_setup *tt = static_cast<thermal_FEM_setup*> (att);
    //if (tt->i_type == TransferKernel)
    //{
    //    return lsf_.triangle(x, y, x1, x2, x3, y1, y2, y3, tt->uv_id[0], gctl::Value) * lsf_.triangle(x, y, x1, x2, x3, y1, y2, y3, tt->uv_id[1], gctl::Value);
    //}
    //else if (tt->i_type == TransferForce || tt->i_type == HeatFlow)
    //{
    //    return lsf_.triangle(x, y, x1, x2, x3, y1, y2, y3, tt->uv_id[0], gctl::Value);
    //}
    //else throw std::runtime_error("Invalid integral type. From GIST::Thermal::glni_triangle_func(...)");
    if (tt->i_type == HeatFlow)
    {
        return lsf_.triangle(x, y, x1, x2, x3, y1, y2, y3, tt->uv_id[0], gctl::Value);
    }
    else throw std::runtime_error("Invalid integral type. From GIST::Thermal::glni_triangle_func(...)");
    return 0.0;
}

void GIST::Thermal::set_gauss_order(int order)
{
    if (order <= 0)
    {
        throw gctl::runtime_error("Invalid gauss order. From GIST::Thermal::set_gauss_order(...)");
    }

    triangle::clear();
    tetrahedron::clear();
    triangle::initiate(order);
    tetrahedron::initiate(order);
    return;
}

void GIST::Thermal::set_thermal_error(double err)
{
    thermal_err_ = err;
    err_ready_ = true;
    return;
}

double GIST::Thermal::get_thermal_error()
{
    if (!err_ready_)
    {
        throw std::runtime_error("The data uncertainty is not set. From GIST::Thermal::get_thermal_error(...)");
    }

    return thermal_err_;
}

void GIST::Thermal::set_regularization_strength(double val)
{
    if (val < 0.0)
    {
        throw std::invalid_argument("Invalid regularization factor. From GIST::Thermal::set_regularization_strength(...)");
    }
    
    reg_strength_ = val;
    return;
}

array<point3ds> &GIST::Thermal::get_obsp()
{
    obs_locS_.resize(obs_num_);

    for (size_t i = 0; i < obs_num_; i++)
    {
        obs_locS_[i] =  obs_loc_[i]->c2s();
    }
    
    return obs_locS_;
}
