/********************************************************
 *     _____________________
 *    / ____/  _/ ___/_  __/
 *   / / __ / / \__ \ / /   
 *  / /_/ // / ___/ // /    
 *  \____/___//____//_/ 
 * Geophysical Inversions using Spherical Tetrahedral meshes (GIST)
 *
 * Copyright (c) 2022  Yi Zhang (yizhang-geo@zju.edu.cn)
 *
 * GIST is distributed under a dual licensing scheme. You can redistribute 
 * it and/or modify it under the terms of the GNU Affero General Public 
 * License (AGPL) as published by the Free Software Foundation, either version 
 * 3 of the License, or (at your option) any later version. You should have 
 * received a copy of the GNU Affero General Public License along with this 
 * program. If not, see <http://www.gnu.org/licenses/>.
 * 
 * If the terms and conditions of the AGPL v.3. would prevent you from using 
 * the GIST, please consider the option to obtain a commercial license for a 
 * fee. These licenses are offered by the original author, Yi Zhang. As a rule, 
 * licenses are provided "as-is", unlimited in time for a one time fee. Please 
 * send corresponding requests to: yizhang-geo@zju.edu.cn. Please do not forget 
 * to include some description of your company and the realm of its activities. 
 * Also add information on how to contact you by electronic and paper mail.
 ******************************************************/

#include "thermal.h"

void GIST::Thermal::ResetBoundaryHeatFlow()
{
    if (!system_ready_)
    {
        throw gctl::runtime_error("[GIST::Thermal::ResetBoundaryHeatFlow] System not initialized.");
    }

    grd_cndt_.resize(node_num_, 0.0); // 热梯度边界条件数组
    return;
}

void GIST::Thermal::SetBoundaryHeatFlow(const array<size_t> &fac_idx, double hf_val)
{
    array<gctl::triangle> &faces = m_space_->get_face();

    size_t c_id;
    double x1 = 0.0, x2, x3, y1 = 0.0, y2 = 0.0, y3;

    thermal_FEM_setup tt;
    tt.i_type = HeatFlow;

    int v_id;
    for (size_t i = 0; i < fac_idx.size(); i++)
    {
        faces[fac_idx[i]].local_coordinates(x2, x3, y3, c_id);

        for (size_t k = 0; k < 3; k++)
        {
            tt.uv_id[0] = k;
            v_id = faces[fac_idx[i]].vert[(c_id+k)%3]->id;
            // set unit to W/m.K. The input unit is asummed in mW/m.K
            grd_cndt_[v_id] += -1e-3*hf_val*triangle::integral(x1, x2, x3, y1, y2, y3, &tt);
        }
    }
    return;
}

void GIST::Thermal::SetBoundaryHeatFlow(const array<size_t> &fac_idx, const array<double> &node_vals)
{
    array<gctl::triangle> &faces = m_space_->get_face();

    size_t c_id;
    thermal_FEM_setup tt;
    double x1 = 0.0, x2, x3, y1 = 0.0, y2 = 0.0, y3;

    int v_id;
    tt.i_type = HeatFlow;
    for (size_t i = 0; i < fac_idx.size(); i++)
    {
        faces[fac_idx[i]].local_coordinates(x2, x3, y3, c_id);

        for (size_t k = 0; k < 3; k++)
        {
            tt.uv_id[0] = k;
            v_id = faces[fac_idx[i]].vert[(c_id+k)%3]->id;
            grd_cndt_[v_id] += -1e-3*node_vals[v_id]*triangle::integral(x1, x2, x3, y1, y2, y3, &tt);
        }
    }
    return;
}