/********************************************************
 *     _____________________
 *    / ____/  _/ ___/_  __/
 *   / / __ / / \__ \ / /   
 *  / /_/ // / ___/ // /    
 *  \____/___//____//_/ 
 * Geophysical Inversions using Spherical Tetrahedral meshes (GIST)
 *
 * Copyright (c) 2022  Yi Zhang (yizhang-geo@zju.edu.cn)
 *
 * GIST is distributed under a dual licensing scheme. You can redistribute 
 * it and/or modify it under the terms of the GNU Affero General Public 
 * License (AGPL) as published by the Free Software Foundation, either version 
 * 3 of the License, or (at your option) any later version. You should have 
 * received a copy of the GNU Affero General Public License along with this 
 * program. If not, see <http://www.gnu.org/licenses/>.
 * 
 * If the terms and conditions of the AGPL v.3. would prevent you from using 
 * the GIST, please consider the option to obtain a commercial license for a 
 * fee. These licenses are offered by the original author, Yi Zhang. As a rule, 
 * licenses are provided "as-is", unlimited in time for a one time fee. Please 
 * send corresponding requests to: yizhang-geo@zju.edu.cn. Please do not forget 
 * to include some description of your company and the realm of its activities. 
 * Also add information on how to contact you by electronic and paper mail.
 ******************************************************/

#include "thermal.h"

void GIST::Thermal::SaveObs(std::string filename)
{
    std::ofstream outfile;
    gctl::open_outfile(outfile, filename, ".txt");

    double ele;
    point3ds ps;

    outfile << "# reference system: " << std::to_string(earth_1d_->get_minor_radius()) << "/" << std::to_string(earth_1d_->get_major_radius()) << "\n";
    outfile << "# observation number\n" << obs_num_ << "\n";

    array<double> noises;
    if (err_ready_)
    {
        noises.resize(obs_num_);
        random(noises, 0.0, thermal_err_);
    }
    
    outfile << "# lon(deg) lat(deg) elev(m) heatflow_for(mW/m^2)";
    if (err_ready_) outfile << " STD_for(mW/m^2)";
    if (has_obs_) outfile << " heatflow_obs(mW/m^2) diff(mW/m^2)";
    outfile << "\n";

    for (size_t i = 0; i < obs_num_; i++)
    {
        ps = obs_loc_[i]->c2s();
        ele = ps.rad - earth_1d_->get_radius(ps.lat);

        outfile << std::fixed << std::setprecision(6) << ps.lon << " " << ps.lat << " " << ele;

        if (err_ready_) outfile << std::setprecision(12) << " " << pre_hf_[i] + noises[i] << " " << thermal_err_;
        else outfile << std::setprecision(12) << " " << pre_hf_[i];

        if (has_obs_) outfile << std::setprecision(12) << " " << obs_hf_[i] << " " << pre_hf_[i] - obs_hf_[i];

        outfile << "\n";
    }

    outfile.close();
    return;
}