/********************************************************
 *     _____________________
 *    / ____/  _/ ___/_  __/
 *   / / __ / / \__ \ / /   
 *  / /_/ // / ___/ // /    
 *  \____/___//____//_/ 
 * Geophysical Inversions using Spherical Tetrahedral meshes (GIST)
 *
 * Copyright (c) 2022  Yi Zhang (yizhang-geo@zju.edu.cn)
 *
 * GIST is distributed under a dual licensing scheme. You can redistribute 
 * it and/or modify it under the terms of the GNU Affero General Public 
 * License (AGPL) as published by the Free Software Foundation, either version 
 * 3 of the License, or (at your option) any later version. You should have 
 * received a copy of the GNU Affero General Public License along with this 
 * program. If not, see <http://www.gnu.org/licenses/>.
 * 
 * If the terms and conditions of the AGPL v.3. would prevent you from using 
 * the GIST, please consider the option to obtain a commercial license for a 
 * fee. These licenses are offered by the original author, Yi Zhang. As a rule, 
 * licenses are provided "as-is", unlimited in time for a one time fee. Please 
 * send corresponding requests to: yizhang-geo@zju.edu.cn. Please do not forget 
 * to include some description of your company and the realm of its activities. 
 * Also add information on how to contact you by electronic and paper mail.
 ******************************************************/

#include "thermal.h"

void GIST::Thermal::ReadObs(std::string filename, double off_set)
{
    _1s_vector str_vec;
    text_descriptor t_desc(filename);
    read_text_lines(t_desc, str_vec);

    if (str_vec.empty())
    {
        throw std::runtime_error("Empty input observation file.");
    }

    size_t o_num;
    str2type(str_vec[0], o_num);
    if (o_num != obs_num_ || str_vec.size() != obs_num_ + 1)
    {
        throw std::runtime_error("Number of the observation sites in file does not match with the current model file.");
    }

    obs_hf_.resize(obs_num_);

    bool site_found;
    point3ds ps;
    double lon, lat, rad, val, ref_rad;
    for (size_t i = 1; i < str_vec.size(); i++)
    {
        parse_string_to_value(str_vec[i], ' ', true, lon, lat, rad, val);

        site_found = false;
        for (size_t i = 0; i < obs_num_; i++)
        {
            ps = obs_loc_[i]->c2s();
            ref_rad = earth_1d_->get_radius(ps.lat);

            if (fabs(ps.lon - lon) < off_set &&
                fabs(ps.lat - lat) < off_set && 
                fabs(ps.rad - rad - ref_rad) < off_set)
            {
                obs_hf_[i] = val;
                site_found = true;
                break;
            }
        }
        
        if (!site_found)
        {
            throw gctl::runtime_error("No observation site is found for: " + str_vec[i]);
        }
    }

    has_obs_ = true;
    return;
}