/********************************************************
 *     _____________________
 *    / ____/  _/ ___/_  __/
 *   / / __ / / \__ \ / /   
 *  / /_/ // / ___/ // /    
 *  \____/___//____//_/ 
 * Geophysical Inversions using Spherical Tetrahedral meshes (GIST)
 *
 * Copyright (c) 2022  Yi Zhang (yizhang-geo@zju.edu.cn)
 *
 * GIST is distributed under a dual licensing scheme. You can redistribute 
 * it and/or modify it under the terms of the GNU Affero General Public 
 * License (AGPL) as published by the Free Software Foundation, either version 
 * 3 of the License, or (at your option) any later version. You should have 
 * received a copy of the GNU Affero General Public License along with this 
 * program. If not, see <http://www.gnu.org/licenses/>.
 * 
 * If the terms and conditions of the AGPL v.3. would prevent you from using 
 * the GIST, please consider the option to obtain a commercial license for a 
 * fee. These licenses are offered by the original author, Yi Zhang. As a rule, 
 * licenses are provided "as-is", unlimited in time for a one time fee. Please 
 * send corresponding requests to: yizhang-geo@zju.edu.cn. Please do not forget 
 * to include some description of your company and the realm of its activities. 
 * Also add information on how to contact you by electronic and paper mail.
 ******************************************************/

#include "thermal.h"

void GIST::Thermal::InitObsSites(std::string obs_name)
{
    array<gmsh_physical_group> &phys_groups = m_space_->get_physics();
    array<int> &nde_tags = m_space_->get_tag(Node); // element node
    array<enode> &nde = m_space_->get_enode(); // 获取元素点数组

    // Find tag number for the observation stations
    int tar_tag;
    bool ok_tag = false;
    for (size_t i = 0; i < phys_groups.size(); i++)
    {
        if (phys_groups[i].name == obs_name)
        {
            tar_tag = phys_groups[i].phys_tag;
            ok_tag = true;
        }
    }
    
    if (!ok_tag)
    {
        throw gctl::runtime_error("[GIST::Thermal::InitObsSites] Fail to find the targeting tag: " + obs_name);
    }

    size_t ct = 0;
    for (size_t i = 0; i < nde.size(); i++)
    {
        if (nde_tags[i] == tar_tag) ct++;
    }

    // Initiate indexes of the observation sites
    obs_num_ = ct;
    obs_idx_.resize(obs_num_);
    obs_loc_.resize(obs_num_);
    pre_hf_.resize(obs_num_);

    ct = 0;
    for (size_t i = 0; i < nde.size(); i++)
    {
        if (nde_tags[i] == tar_tag)
        {
            obs_idx_[ct] = nde[i].vert[0]->id;
            obs_loc_[ct] = nde[i].vert[0];
            ct++;
        }
    }

    size_t all_ct = 0;
    obs_hosts_.resize(obs_num_);
    for (size_t i = 0; i < obs_num_; i++)
    {
        ct = obs_idx_[i];
        obs_hosts_[i].resize(node_hosts_[ct].size());
        for (size_t j = 0; j < node_hosts_[ct].size(); j++)
        {
            obs_hosts_[i][j] = node_hosts_[ct][j];
            all_ct++;
        }
    }

    // 初始化梯度矩阵
    hf_kernel_.malloc(obs_num_, node_num_, 0.0);
    hf_coeff_.reserve(all_ct);

    double total_vol, xcf, ycf, zcf;
    double ksi, eta, zta, wgtx, wgty, wgtz;
    vertex3dc *vt[4];
    point3ds ps;
    for (size_t i = 0; i < obs_num_; i++)
    {
        ps = obs_loc_[i]->c2s();
        // Set heat flow to mW/m^2 and rotate the gradient to radial direction
        xcf = 1e+3*sin((0.5-ps.lat/180.0)*GCTL_Pi)*cos((2.0+ps.lon/180.0)*GCTL_Pi);
        ycf = 1e+3*sin((0.5-ps.lat/180.0)*GCTL_Pi)*sin((2.0+ps.lon/180.0)*GCTL_Pi);
        zcf = 1e+3*cos((0.5-ps.lat/180.0)*GCTL_Pi);

        total_vol = 0.0;
        for (size_t j = 0; j < obs_hosts_[i].size(); j++)
        {
            //total_vol += 1.0/cbrt(3.0*obs_hosts_[i][j]->volume()); // 以等效块体的边长的倒数作为插值的权重
            total_vol += 1.0/cbrt(node_hosts_[i][j]->volume()); // 以体积三次根的倒数作为插值的权重
        }

        for (size_t j = 0; j < obs_hosts_[i].size(); j++)
        {
            for (size_t k = 0; k < 4; k++)
            {
                vt[k] = obs_hosts_[i][j]->vert[k];
            }

            lsf_.global2local_tetrahedron(ksi, eta, zta, obs_loc_[i]->x, obs_loc_[i]->y, obs_loc_[i]->z, 
                vt[0]->x, vt[1]->x, vt[2]->x, vt[3]->x, vt[0]->y, vt[1]->y, vt[2]->y, vt[3]->y, 
                vt[0]->z, vt[1]->z, vt[2]->z, vt[3]->z);

            for (size_t k = 0; k < 4; k++)
            {
                wgtx = 1.0/(cbrt(obs_hosts_[i][j]->volume())*total_vol)*lsf_.tetrahedron(ksi, eta, zta, vt[0]->x, vt[1]->x, vt[2]->x, vt[3]->x, 
                    vt[0]->y, vt[1]->y, vt[2]->y, vt[3]->y, vt[0]->z, vt[1]->z, vt[2]->z, vt[3]->z, k, gctl::Dx);

                wgty = 1.0/(cbrt(obs_hosts_[i][j]->volume())*total_vol)*lsf_.tetrahedron(ksi, eta, zta, vt[0]->x, vt[1]->x, vt[2]->x, vt[3]->x, 
                    vt[0]->y, vt[1]->y, vt[2]->y, vt[3]->y, vt[0]->z, vt[1]->z, vt[2]->z, vt[3]->z, k, gctl::Dy);

                wgtz = 1.0/(cbrt(obs_hosts_[i][j]->volume())*total_vol)*lsf_.tetrahedron(ksi, eta, zta, vt[0]->x, vt[1]->x, vt[2]->x, vt[3]->x, 
                    vt[0]->y, vt[1]->y, vt[2]->y, vt[3]->y, vt[0]->z, vt[1]->z, vt[2]->z, vt[3]->z, k, gctl::Dz);

                hf_coeff_.push_back(xcf*wgtx + ycf*wgty + zcf*wgtz);
            }
        }
    }

    return;
}