/********************************************************
 *     _____________________
 *    / ____/  _/ ___/_  __/
 *   / / __ / / \__ \ / /   
 *  / /_/ // / ___/ // /    
 *  \____/___//____//_/ 
 * Geophysical Inversions using Spherical Tetrahedral meshes (GIST)
 *
 * Copyright (c) 2022  Yi Zhang (yizhang-geo@zju.edu.cn)
 *
 * GIST is distributed under a dual licensing scheme. You can redistribute 
 * it and/or modify it under the terms of the GNU Affero General Public 
 * License (AGPL) as published by the Free Software Foundation, either version 
 * 3 of the License, or (at your option) any later version. You should have 
 * received a copy of the GNU Affero General Public License along with this 
 * program. If not, see <http://www.gnu.org/licenses/>.
 * 
 * If the terms and conditions of the AGPL v.3. would prevent you from using 
 * the GIST, please consider the option to obtain a commercial license for a 
 * fee. These licenses are offered by the original author, Yi Zhang. As a rule, 
 * licenses are provided "as-is", unlimited in time for a one time fee. Please 
 * send corresponding requests to: yizhang-geo@zju.edu.cn. Please do not forget 
 * to include some description of your company and the realm of its activities. 
 * Also add information on how to contact you by electronic and paper mail.
 ******************************************************/

#include "thermal.h"

void GIST::Thermal::ImportModelSetup(ModelSpace *m_space, Earth1D *earth_1d)
{
    // Initiate variables
    earth_1d_ = earth_1d;
    m_space_ = m_space;
    node_num_ = m_space_->get_node_number();
    elem_num_ = m_space_->get_element_number();
    face_num_ = m_space_->get_face_number();

    kernel_list_.resize(elem_num_ * 16);
    thermal_kernel_.malloc(node_num_, node_num_, 0.0);
    tar_.resize(node_num_, 0.0); // 有限元目标数组

    // create aliases 
    array<gctl::tetrahedron> &elem = m_space->get_element();
    array<gctl::vertex3dc> &node = m_space->get_node();

    vertex3dc *vt[4];
    mat_node<double> tmp_entry;
    thermal_FEM_setup tt;

    size_t i, j, f;
#pragma omp parallel for private (i, j, f, vt, tmp_entry, tt) schedule(guided)
    for (i = 0; i < elem_num_; i++)
	{
        //初始化形函数计算类型
        tt.i_type = ThermalKernel;

        // 拷贝到局部变量
        for (j = 0; j < 4; j++)
        {
            vt[j] = elem[i].vert[j];
        }

		for (j = 0; j < 4; j++)
		{
            // 记录顶点在核矩阵中的位置和形函数局部索引
            tmp_entry.r_id = vt[j]->id; tt.uv_id[0] = j;

			for (f = 0; f < 4; f++)
			{                
                tmp_entry.c_id = vt[f]->id; tt.uv_id[1] = f;
                tmp_entry.val = tetrahedron::integral(vt[0]->x, vt[1]->x, vt[2]->x, vt[3]->x, 
                    vt[0]->y, vt[1]->y, vt[2]->y, vt[3]->y, vt[0]->z, vt[1]->z, vt[2]->z, vt[3]->z, &tt);

                // 保存核矩阵三元组 注意还没有乘热导率
                kernel_list_[16*i + 4*j + f] = tmp_entry;
			}
		}
	}

    // Initiate the host elements of all nodes
    node_hosts_.resize(node_num_);
    for (i = 0; i < elem_num_; i++)
    {
        for (j = 0; j < 4; j++)
        {
            node_hosts_[elem[i].vert[j]->id].push_back(elem.get(i));   
        }
    }

    system_ready_ = true;
    return;
}