/********************************************************
 *     _____________________
 *    / ____/  _/ ___/_  __/
 *   / / __ / / \__ \ / /   
 *  / /_/ // / ___/ // /    
 *  \____/___//____//_/ 
 * Geophysical Inversions using Spherical Tetrahedral meshes (GIST)
 *
 * Copyright (c) 2022  Yi Zhang (yizhang-geo@zju.edu.cn)
 *
 * GIST is distributed under a dual licensing scheme. You can redistribute 
 * it and/or modify it under the terms of the GNU Affero General Public 
 * License (AGPL) as published by the Free Software Foundation, either version 
 * 3 of the License, or (at your option) any later version. You should have 
 * received a copy of the GNU Affero General Public License along with this 
 * program. If not, see <http://www.gnu.org/licenses/>.
 * 
 * If the terms and conditions of the AGPL v.3. would prevent you from using 
 * the GIST, please consider the option to obtain a commercial license for a 
 * fee. These licenses are offered by the original author, Yi Zhang. As a rule, 
 * licenses are provided "as-is", unlimited in time for a one time fee. Please 
 * send corresponding requests to: yizhang-geo@zju.edu.cn. Please do not forget 
 * to include some description of your company and the realm of its activities. 
 * Also add information on how to contact you by electronic and paper mail.
 ******************************************************/

#include "thermal.h"

void GIST::Thermal::BodyHeatFlow(const array<double> &cndt, const array<double> &t, array<double> &hf)
{
    if (!system_ready_)
    {
        throw gctl::runtime_error("[GIST::Thermal::BodyHeatFlow] System not initialized.");
    }

    array<gctl::vertex3dc> &nodes = m_space_->get_node();

    // 初始化梯度矩阵
    spmat<double> hf_kernel_all(node_num_, node_num_, 0.0);

    double xcf, ycf, zcf;
    double ksi, eta, zta, wgtx, wgty, wgtz;
    vertex3dc *vt[4] = {nullptr, nullptr, nullptr, nullptr};
    point3ds ps;

    double total_vol, vol_cndt;
    for (size_t i = 0; i < node_num_; i++)
    {
        ps = nodes[i].c2s();
        // Set heat flow to mW/m^2 and rotate the gradient to radial direction
        xcf = 1e+3*sin((0.5-ps.lat/180.0)*GCTL_Pi)*cos((2.0+ps.lon/180.0)*GCTL_Pi);
        ycf = 1e+3*sin((0.5-ps.lat/180.0)*GCTL_Pi)*sin((2.0+ps.lon/180.0)*GCTL_Pi);
        zcf = 1e+3*cos((0.5-ps.lat/180.0)*GCTL_Pi);

        total_vol = 0.0;
        for (size_t j = 0; j < node_hosts_[i].size(); j++)
        {
            //total_vol += 1.0/cbrt(3.0*node_hosts_[i][j]->volume()); // 以等效块体的边长的倒数作为插值的权重
            total_vol += 1.0/cbrt(node_hosts_[i][j]->volume()); // 以体积三次根的倒数作为插值的权重
        }

        for (size_t j = 0; j < node_hosts_[i].size(); j++)
        {
            for (size_t k = 0; k < 4; k++)
            {
                vt[k] = node_hosts_[i][j]->vert[k];
            }

            vol_cndt = cndt[node_hosts_[i][j]->id];

            lsf_.global2local_tetrahedron(ksi, eta, zta, nodes[i].x, nodes[i].y, nodes[i].z, 
                vt[0]->x, vt[1]->x, vt[2]->x, vt[3]->x, vt[0]->y, vt[1]->y, vt[2]->y, vt[3]->y, 
                vt[0]->z, vt[1]->z, vt[2]->z, vt[3]->z);

            for (size_t k = 0; k < 4; k++)
            {
                wgtx = 1.0/(cbrt(node_hosts_[i][j]->volume())*total_vol)*lsf_.tetrahedron(ksi, eta, zta, vt[0]->x, vt[1]->x, vt[2]->x, vt[3]->x, 
                    vt[0]->y, vt[1]->y, vt[2]->y, vt[3]->y, vt[0]->z, vt[1]->z, vt[2]->z, vt[3]->z, k, gctl::Dx);

                wgty = 1.0/(cbrt(node_hosts_[i][j]->volume())*total_vol)*lsf_.tetrahedron(ksi, eta, zta, vt[0]->x, vt[1]->x, vt[2]->x, vt[3]->x, 
                    vt[0]->y, vt[1]->y, vt[2]->y, vt[3]->y, vt[0]->z, vt[1]->z, vt[2]->z, vt[3]->z, k, gctl::Dy);

                wgtz = 1.0/(cbrt(node_hosts_[i][j]->volume())*total_vol)*lsf_.tetrahedron(ksi, eta, zta, vt[0]->x, vt[1]->x, vt[2]->x, vt[3]->x, 
                    vt[0]->y, vt[1]->y, vt[2]->y, vt[3]->y, vt[0]->z, vt[1]->z, vt[2]->z, vt[3]->z, k, gctl::Dz);

                // 注意我们设置朝向地心的方向为正 所以要乘-1
                hf_kernel_all.add_scalar(i, node_hosts_[i][j]->vert[k]->id, -1.0*vol_cndt*(wgtx*xcf + wgty*ycf + wgtz*zcf));
            }
        }
    }

    hf_kernel_all.multiply_vector(t, hf);
    return;
}