/********************************************************
 *     _____________________
 *    / ____/  _/ ___/_  __/
 *   / / __ / / \__ \ / /   
 *  / /_/ // / ___/ // /    
 *  \____/___//____//_/ 
 * Geophysical Inversions using Spherical Tetrahedral meshes (GIST)
 *
 * Copyright (c) 2022  Yi Zhang (yizhang-geo@zju.edu.cn)
 *
 * GIST is distributed under a dual licensing scheme. You can redistribute 
 * it and/or modify it under the terms of the GNU Affero General Public 
 * License (AGPL) as published by the Free Software Foundation, either version 
 * 3 of the License, or (at your option) any later version. You should have 
 * received a copy of the GNU Affero General Public License along with this 
 * program. If not, see <http://www.gnu.org/licenses/>.
 * 
 * If the terms and conditions of the AGPL v.3. would prevent you from using 
 * the GIST, please consider the option to obtain a commercial license for a 
 * fee. These licenses are offered by the original author, Yi Zhang. As a rule, 
 * licenses are provided "as-is", unlimited in time for a one time fee. Please 
 * send corresponding requests to: yizhang-geo@zju.edu.cn. Please do not forget 
 * to include some description of your company and the realm of its activities. 
 * Also add information on how to contact you by electronic and paper mail.
 ******************************************************/

#include "thermal.h"

void GIST::Thermal::AssembleLinearSystem(const array<double> &cndt)
{
    if (!system_ready_)
    {
        throw std::runtime_error("[GIST::Thermal::AssembleLinearSystem] System not initialized.");
    }

    if (cndt.size() != elem_num_)
    {
        throw std::runtime_error("[GIST::Thermal::AssembleLinearSystem] Invalid size of the thermal conductivity model.");
    }

    if (val_cndt_.empty() && grd_cndt_.empty())
    {
        throw std::runtime_error("[GIST::Thermal::AssembleLinearSystem] No boundary conditions found.");
    }

    std::clog << "Assembling the thermal system...\n";

    if (triplts_.size() != kernel_list_.size())
    {
        triplts_.resize(kernel_list_.size());
    }

    // Initiate the thermal kernel according ot the elements' conductivity
    for (size_t i = 0; i < elem_num_; i++)
	{
		for (size_t j = 0; j < 4; j++)
		{
			for (size_t f = 0; f < 4; f++)
			{
                triplts_[16*i + 4*j + f] = kernel_list_[16*i + 4*j + f];
                triplts_[16*i + 4*j + f].val *= cndt[i]; // 乘上热导率
			}
		}
	}

    // Modify the thermal kernel according to the heat transfer
    //if (!trn_cndt_.empty())
    //{
    //    for (size_t i = 0; i < face_num_; i++)
    //    {
    //        if (has_trn_cndt_[i])
    //        {
    //            for (size_t j = 0; j < 3; j++)
    //            {
    //                for (size_t f = 0; f < 3; f++)
    //                {
    //                    triplts_.push_back(trans_list_[9*i + 3*j + f]);
    //                }
    //            }
    //        }
    //    }
    //}

    // Sort triplts to increase in both row and column indexes
    std::sort(triplts_.begin(), triplts_.end(), [](mat_node<double> a, mat_node<double> b)->bool{
        if (a.r_id < b.r_id) return true;
        if (a.r_id == b.r_id && a.c_id < b.c_id) return true;
        return false;
    });

    thermal_kernel_.set_triplts(triplts_);

    tar_.assign_all(0.0);
    // Set first-class boundary conditions
    if (!val_cndt_.empty())
    {
        for (size_t i = 0; i < node_num_; i++)
        {
            tar_[i] = val_cndt_[i];
        }

        for (size_t i = 0; i < node_num_; i++)
        {
            if (!has_val_cndt_[i])
            {
                for (size_t j = 0; j < node_num_; j++)
                {
                    if (has_val_cndt_[j])
                    {
                        tar_[i] -= thermal_kernel_.at(i, j) * tar_[j];
                    }
                }
            }
        }

        for (size_t i = 0; i < node_num_; i++)
        {
            if (has_val_cndt_[i])
            {
                thermal_kernel_.clear(i, gctl::RowMajor);
                thermal_kernel_.clear(i, gctl::ColMajor);
                thermal_kernel_.insert(i, i, 1.0);
            }
        }
    }

    // Set second-class boundary conditions and heat productivity
    if (!pdt_cndt_.empty())
    {
        for (size_t i = 0; i < node_num_; i++)
        {
            tar_[i] += pdt_cndt_[i];
        }
    }

    if (!grd_cndt_.empty())
    {
        for (size_t i = 0; i < node_num_; i++)
        {
            tar_[i] += grd_cndt_[i];
        }
    }

    //if (!trn_cndt_.empty())
    //{
    //    for (size_t i = 0; i < node_num_; i++)
    //    {
    //        tar_[i] += trn_cndt_[i];
    //    }
    //}
    return;
}