/********************************************************
 *     _____________________
 *    / ____/  _/ ___/_  __/
 *   / / __ / / \__ \ / /   
 *  / /_/ // / ___/ // /    
 *  \____/___//____//_/ 
 * Geophysical Inversions using Spherical Tetrahedral meshes (GIST)
 *
 * Copyright (c) 2022  Yi Zhang (yizhang-geo@zju.edu.cn)
 *
 * GIST is distributed under a dual licensing scheme. You can redistribute 
 * it and/or modify it under the terms of the GNU Affero General Public 
 * License (AGPL) as published by the Free Software Foundation, either version 
 * 3 of the License, or (at your option) any later version. You should have 
 * received a copy of the GNU Affero General Public License along with this 
 * program. If not, see <http://www.gnu.org/licenses/>.
 * 
 * If the terms and conditions of the AGPL v.3. would prevent you from using 
 * the GIST, please consider the option to obtain a commercial license for a 
 * fee. These licenses are offered by the original author, Yi Zhang. As a rule, 
 * licenses are provided "as-is", unlimited in time for a one time fee. Please 
 * send corresponding requests to: yizhang-geo@zju.edu.cn. Please do not forget 
 * to include some description of your company and the realm of its activities. 
 * Also add information on how to contact you by electronic and paper mail.
 ******************************************************/

#include "pressure.h"

void GIST::Pressure::Solve(const array<double> &den, array<double> &p)
{
    if (!kernel_ready_ || !bnd_set_)
    {
        throw gctl::runtime_error("The pressure kernel or boundary condition is not initialized. From GIST::Pressure::Solve(...)");
    }

    if (den.size() != elem_num_)
    {
        throw std::runtime_error("The density model is of the wrong size. From GIST::Pressure::Solve(...).");
    }
    
    // Start the preparation of the kernal matrix

    if (bnd_changed_) // Reassemble the kernal matrix if boundary values change
    {
        pressure_kernel_.set_triplts(k_list_);

        bnd_correct_.assign_all(0.0);
        for (size_t i = 0; i < node_num_; i++)
        {
            if (!has_val_cndt_[i])
            {
                for (size_t j = 0; j < node_num_; j++)
                {
                    if (has_val_cndt_[j])
                    {
                        bnd_correct_[i] += pressure_kernel_.at(i, j) * val_cndt_[j];
                    }
                }
            }
        }

        for (size_t i = 0; i < node_num_; i++)
        {
            if (has_val_cndt_[i])
            {
                pressure_kernel_.clear(i, gctl::RowMajor);
                pressure_kernel_.clear(i, gctl::ColMajor);
                pressure_kernel_.insert(i, i, 1.0);
            }
        }

        // Get the preconditioning matrix as the diagonal elements
        pressure_kernel_.get_diagonal(precndt_);

        bnd_changed_ = false;
    }

    // Set first-class boundary conditions
    tar_.assign_all(0.0);
    for (size_t i = 0; i < 4*elem_num_; i++)
    {
        tar_[ik_list_[i].r_id] += ik_list_[i].val * den[ik_list_[i].c_id];
    }
    
    for (size_t i = 0; i < node_num_; i++)
    {
        if (has_val_cndt_[i])
        {
            tar_[i] = val_cndt_[i];
        }
    }

    for (size_t i = 0; i < node_num_; i++)
    {
        tar_[i] -= bnd_correct_[i];
    }

    // End the preparation of the kernal matrix

	// 调用共轭梯度基类的求解函数
    if (p.size() != node_num_) p.resize(node_num_, 0.0);
    else p.assign_all(0.0);

    LCG_Minimize(p, tar_, gctl::LCG_PCG);
    return;
}