/********************************************************
 *     _____________________
 *    / ____/  _/ ___/_  __/
 *   / / __ / / \__ \ / /   
 *  / /_/ // / ___/ // /    
 *  \____/___//____//_/ 
 * Geophysical Inversions using Spherical Tetrahedral meshes (GIST)
 *
 * Copyright (c) 2022  Yi Zhang (yizhang-geo@zju.edu.cn)
 *
 * GIST is distributed under a dual licensing scheme. You can redistribute 
 * it and/or modify it under the terms of the GNU Affero General Public 
 * License (AGPL) as published by the Free Software Foundation, either version 
 * 3 of the License, or (at your option) any later version. You should have 
 * received a copy of the GNU Affero General Public License along with this 
 * program. If not, see <http://www.gnu.org/licenses/>.
 * 
 * If the terms and conditions of the AGPL v.3. would prevent you from using 
 * the GIST, please consider the option to obtain a commercial license for a 
 * fee. These licenses are offered by the original author, Yi Zhang. As a rule, 
 * licenses are provided "as-is", unlimited in time for a one time fee. Please 
 * send corresponding requests to: yizhang-geo@zju.edu.cn. Please do not forget 
 * to include some description of your company and the realm of its activities. 
 * Also add information on how to contact you by electronic and paper mail.
 ******************************************************/

#include "pressure.h"

void GIST::Pressure::SetElevationPressure(const array<size_t> &elev_idx, const array<double> &elev, double topo_den)
{
    if (!kernel_ready_)
    {
        throw gctl::runtime_error("The pressure kernel is not initialized. From GIST::Pressure::SetElevationPressure(...)");
    }

    elev_size_ = elev.size();
    elev_index_.resize(elev_size_);
    obs_elev_press_.resize(elev_size_);
    bkg_elev_press_.resize(elev_size_, 0.0);
    pre_elev_press_.resize(elev_size_);

    norm_vol_wgts_.resize(elem_num_, 1.0);

    for (size_t i = 0; i < elev_size_; i++)
    {
        elev_index_[i] = elev_idx[i];
        obs_elev_press_[i] = 1e-6*topo_den*GIST_NORMAL_GRAVITY*elev[i];
    }

    tar_ready_ = true;
    return;
}

void GIST::Pressure::SetElevationPressureLevel(double bkg_press_val)
{
    if (!tar_ready_)
    {
        throw std::runtime_error("Targeting pressure is not set. From GIST::Pressure::SetElevationPressureLevel(...)");
    }

    bkg_elev_press_.assign_all(bkg_press_val);
    return;
}

void GIST::Pressure::SetElevationPressureLevel(const array<size_t> &elev_idx, const array<double> &bkg_press)
{
    if (!tar_ready_)
    {
        throw std::runtime_error("Targeting pressure is not set. From GIST::Pressure::SetElevationPressureLevel(...)");
    }

    for (size_t i = 0; i < elev_size_; i++)
    {
        bkg_elev_press_[i] = bkg_press[elev_idx[i]];
    }
    return;
}

void GIST::Pressure::SetNormalizedVolumeWeights(const array<double> &in_wgts)
{
    for (size_t i = 0; i < elem_num_; i++)
    {
        norm_vol_wgts_[i] = in_wgts[i];
    }
    return;
}