/********************************************************
 *     _____________________
 *    / ____/  _/ ___/_  __/
 *   / / __ / / \__ \ / /   
 *  / /_/ // / ___/ // /    
 *  \____/___//____//_/ 
 * Geophysical Inversions using Spherical Tetrahedral meshes (GIST)
 *
 * Copyright (c) 2022  Yi Zhang (yizhang-geo@zju.edu.cn)
 *
 * GIST is distributed under a dual licensing scheme. You can redistribute 
 * it and/or modify it under the terms of the GNU Affero General Public 
 * License (AGPL) as published by the Free Software Foundation, either version 
 * 3 of the License, or (at your option) any later version. You should have 
 * received a copy of the GNU Affero General Public License along with this 
 * program. If not, see <http://www.gnu.org/licenses/>.
 * 
 * If the terms and conditions of the AGPL v.3. would prevent you from using 
 * the GIST, please consider the option to obtain a commercial license for a 
 * fee. These licenses are offered by the original author, Yi Zhang. As a rule, 
 * licenses are provided "as-is", unlimited in time for a one time fee. Please 
 * send corresponding requests to: yizhang-geo@zju.edu.cn. Please do not forget 
 * to include some description of your company and the realm of its activities. 
 * Also add information on how to contact you by electronic and paper mail.
 ******************************************************/

#include "pressure.h"

void GIST::Pressure::ResetBoundaryValue()
{
    if (!kernel_ready_)
    {
        throw std::runtime_error("The pressure system is not initialized.");
    }

    has_val_cndt_.resize(node_num_, false);
    val_cndt_.resize(node_num_, 0.0);
    return;
}

void GIST::Pressure::SetBoundaryValue(const array<size_t> &node_idx)
{
    // create an alias
    gctl::array<gctl::vertex3dc> &nodes = m_space_->get_node();

    point3ds ps;
    for (size_t i = 0; i < node_idx.size(); i++)
    {
        ps = nodes[node_idx[i]].c2s();
        val_cndt_[node_idx[i]] = earth_1d_->get_data(ps.rad, ps.lat, GIST::Lithostatic_Pressure);
        has_val_cndt_[node_idx[i]] = true;   
    }

    bnd_set_ = true;
    bnd_changed_ = true;
    return;
}

void GIST::Pressure::SetBoundaryValue(const array<size_t> &node_idx, double node_val)
{
    for (size_t i = 0; i < node_idx.size(); i++)
    {
        val_cndt_[node_idx[i]] = node_val;
        has_val_cndt_[node_idx[i]] = true;   
    }

    bnd_set_ = true;
    bnd_changed_ = true;
    return;
}

void GIST::Pressure::SetBoundaryValue(const array<size_t> &node_idx, const array<double> &node_vals)
{
    if (node_idx.size() != node_vals.size())
    {
        throw gctl::runtime_error("Unequal input arrays' size. From GIST::Pressure::SetBoundaryValue(...)");
    }

    for (size_t i = 0; i < node_idx.size(); i++)
    {
        val_cndt_[node_idx[i]] = node_vals[i];
        has_val_cndt_[node_idx[i]] = true;   
    }

    bnd_set_ = true;
    bnd_changed_ = true;
    return;
}
