/********************************************************
 *     _____________________
 *    / ____/  _/ ___/_  __/
 *   / / __ / / \__ \ / /   
 *  / /_/ // / ___/ // /    
 *  \____/___//____//_/ 
 * Geophysical Inversions using Spherical Tetrahedral meshes (GIST)
 *
 * Copyright (c) 2022  Yi Zhang (yizhang-geo@zju.edu.cn)
 *
 * GIST is distributed under a dual licensing scheme. You can redistribute 
 * it and/or modify it under the terms of the GNU Affero General Public 
 * License (AGPL) as published by the Free Software Foundation, either version 
 * 3 of the License, or (at your option) any later version. You should have 
 * received a copy of the GNU Affero General Public License along with this 
 * program. If not, see <http://www.gnu.org/licenses/>.
 * 
 * If the terms and conditions of the AGPL v.3. would prevent you from using 
 * the GIST, please consider the option to obtain a commercial license for a 
 * fee. These licenses are offered by the original author, Yi Zhang. As a rule, 
 * licenses are provided "as-is", unlimited in time for a one time fee. Please 
 * send corresponding requests to: yizhang-geo@zju.edu.cn. Please do not forget 
 * to include some description of your company and the realm of its activities. 
 * Also add information on how to contact you by electronic and paper mail.
 ******************************************************/

#include "pressure.h"

double GIST::Pressure::PressureObjectiveFunc(const array<double> &pphys_press, const array<double> &fem_press, 
    const array<double> &press_grad, array<double> &out_grad)
{
    if (!err_ready_)
    {
        throw std::runtime_error("Data uncertainty of the pressure model is not set.");
    }

    elem_press_.resize(elem_num_); // resize the array for element-wise pressure model.
    m_space_->Node2Element(fem_press, elem_press_); // get the element-wise pressure model.

    // Calculate the weighted differences between two pressure models.
    elem_diff_.resize(elem_num_);
    for (size_t i = 0; i < elem_num_; i++)
    {
        elem_diff_[i] = 2.0*(pphys_press[i] - elem_press_[i])/(press_err_*press_err_*elem_num_);
    }

    // Calculate the model gradient using the joint-matrix algorithm.
    obs_grad_.resize(node_num_);
    m_space_->Element2Node(elem_diff_, obs_grad_);

    // Since boundary values are fixed in the FEM calculation. There are no gradients at these nodes.
    for (size_t i = 0; i < node_num_; i++)
    {
        if (has_val_cndt_[i])
        {
            obs_grad_[i] = 0.0;
        }
    }

    node_grad_.resize(node_num_, 0.0);
    LCG_Minimize(node_grad_, obs_grad_, gctl::LCG_PCG);

    out_grad.resize(elem_num_, 0.0);
    // 计算有限元系统相对于模型密度的梯度 此处只有右端项相对于模型密度的梯度
    for (size_t i = 0; i < 4*elem_num_; i++)
    {
        out_grad[ik_list_[i].c_id] += ik_list_[i].val * node_grad_[ik_list_[i].r_id];
    }

    for (size_t i = 0; i < elem_num_; i++)
    {
        out_grad[i] = 2.0*(pphys_press[i] - elem_press_[i])/(press_err_*press_err_*elem_num_) - out_grad[i]*press_grad[i];
    }

    // Calculate the objective function's value.
    double msf = 0.0;
    for (size_t i = 0; i < elem_num_; i++)
    {
        msf += pow((pphys_press[i] - elem_press_[i])/press_err_, 2);
    }
    return msf/elem_num_;
}
/*
double GIST::Pressure::PressureObjectiveFunc(const array<double> &pphys_press, const array<double> &fem_press, 
    array<double> &out_grad, array<double> *temp_grad, array<double> *press_grad, array<double> *comp_grad)
{
    if (!err_ready_)
    {
        throw std::runtime_error("Data uncertainty of the pressure model is not set.");
    }

    elem_press_.resize(elem_num_); // resize the array for element-wise pressure model.
    m_space_->Node2Element(fem_press, elem_press_); // get the element-wise pressure model.

    // Calculate the weighted differences between two pressure models.
    elem_diff_.resize(elem_num_);
    for (size_t i = 0; i < elem_num_; i++)
    {
        elem_diff_[i] = 2.0*(pphys_press[i] - elem_press_[i])/(press_err_*press_err_*elem_num_);
    }

    // Calculate the model gradient using the joint-matrix algorithm.
    obs_grad_.resize(node_num_);
    m_space_->Node2Element(elem_diff_, obs_grad_, gctl::Trans);

    // Since boundary values are fixed in the FEM calculation. There are no gradients at these nodes.
    for (size_t i = 0; i < node_num_; i++)
    {
        if (has_val_cndt_[i])
        {
            obs_grad_[i] = 0.0;
        }
    }

    node_grad_.resize(node_num_, 0.0);
    LCG_Minimize(node_grad_, obs_grad_, gctl::LCG_PCG);

    // 计算有限元系统相对于模型热导率的梯度 此处只有右端项相对于模型密度的梯度
    elem_diff_.assign_all(0.0);
    for (size_t i = 0; i < 4*elem_num_; i++)
    {
        elem_diff_[ik_list_[i].c_id] += ik_list_[i].val * node_grad_[ik_list_[i].r_id];
    }

    out_grad.resize(3*elem_num_);
    if (temp_grad != nullptr)
    {
        // Phi = sum(pphys_press - elem_press_)^2
        for (size_t i = 0; i < elem_num_; i++)
        {
            out_grad[i] = -1.0*elem_diff_[i]*temp_grad->at(i);
        }
    }
    else out_grad.assign(0.0, 0, elem_num_);

    if (press_grad != nullptr)
    {
        for (size_t i = 0; i < elem_num_; i++)
        {
            out_grad[i + elem_num_] = 2.0*(pphys_press[i] - elem_press_[i])/(press_err_*press_err_*elem_num_) - elem_diff_[i]*press_grad->at(i);
        }
    }
    else
    {
        for (size_t i = 0; i < elem_num_; i++)
        {
            out_grad[i + elem_num_] = 2.0*(pphys_press[i] - elem_press_[i])/(press_err_*press_err_*elem_num_);
        }
    }

    if (comp_grad != nullptr)
    {
        for (size_t i = 0; i < elem_num_; i++)
        {
            out_grad[i + 2*elem_num_] = -1.0*elem_diff_[i]*comp_grad->at(i);
        }
    }
    else out_grad.assign(0.0, 2*elem_num_, elem_num_);

    // Calculate the objective function's value.
    double msf = 0.0;
    for (size_t i = 0; i < elem_num_; i++)
    {
        msf += pow((pphys_press[i] - elem_press_[i])/press_err_, 2);
    }
    return msf/elem_num_;
}
*/