/********************************************************
 *     _____________________
 *    / ____/  _/ ___/_  __/
 *   / / __ / / \__ \ / /   
 *  / /_/ // / ___/ // /    
 *  \____/___//____//_/ 
 * Geophysical Inversions using Spherical Tetrahedral meshes (GIST)
 *
 * Copyright (c) 2022  Yi Zhang (yizhang-geo@zju.edu.cn)
 *
 * GIST is distributed under a dual licensing scheme. You can redistribute 
 * it and/or modify it under the terms of the GNU Affero General Public 
 * License (AGPL) as published by the Free Software Foundation, either version 
 * 3 of the License, or (at your option) any later version. You should have 
 * received a copy of the GNU Affero General Public License along with this 
 * program. If not, see <http://www.gnu.org/licenses/>.
 * 
 * If the terms and conditions of the AGPL v.3. would prevent you from using 
 * the GIST, please consider the option to obtain a commercial license for a 
 * fee. These licenses are offered by the original author, Yi Zhang. As a rule, 
 * licenses are provided "as-is", unlimited in time for a one time fee. Please 
 * send corresponding requests to: yizhang-geo@zju.edu.cn. Please do not forget 
 * to include some description of your company and the realm of its activities. 
 * Also add information on how to contact you by electronic and paper mail.
 ******************************************************/

#ifndef _GIST_PRESSURE_H
#define _GIST_PRESSURE_H

#include "gctl/core.h"
#include "gctl/maths.h"
#include "gctl/algorithm.h"
#include "gctl/io.h"
#include "gctl/optimization.h"

#include "../utility/enum.h"
#include "../utility/data_type.h"
#include "../model_space/model_space.h"

using namespace gctl;

namespace GIST
{
    enum pressure_FEM_type
    {
        PressureKernel,
        PressureForce,
    };

    struct pressure_FEM_setup
    {
        size_t uv_id[2];
        pressure_FEM_type i_type;
    };

    class Pressure : public gctl::lcg_solver, public gctl::glni::tetrahedron
    {
    public:
        Pressure();
        virtual ~Pressure();
        virtual void LCG_Mx(const array<double> &x, array<double> &ax);
        virtual void LCG_Ax(const array<double> &x, array<double> &ax);
        virtual double tetrahedron_func(double ksi, double eta, double zta, 
            double x1, double x2, double x3, double x4, 
            double y1, double y2, double y3, double y4,
            double z1, double z2, double z3, double z4, void *att);

        /**
         * @brief Import objects of the model space and the 1D reference earth. Then initiate the pressure system.
         * 
         * @param m_space Object of the model space.
         * @param earth_1d Object of the 1D reference earth.
         */
        void ImportModelSetup(ModelSpace *m_space, Earth1D *earth_1d);
        void Solve(const array<double> &den, array<double> &p);

        void ResetBoundaryValue();
        void SetBoundaryValue(const array<size_t> &node_idx);
        void SetBoundaryValue(const array<size_t> &node_idx, double node_val);
        void SetBoundaryValue(const array<size_t> &node_idx, const array<double> &node_vals);

        void SetElevationPressure(const array<size_t> &elev_idx, const array<double> &elev, double topo_den);
        void SetElevationPressureLevel(double bkg_press_val);
        void SetElevationPressureLevel(const array<size_t> &elev_idx, const array<double> &bkg_press);
        void SetNormalizedVolumeWeights(const array<double> &in_wgts);

        void ExportElevation(double topo_den, double topo_press, const array<double> &press, const array<size_t> &node_idx, array<double> &out_elev);
        void ExportElevation(double topo_den, const array<double> &press, const array<double> &bkg_press, const array<size_t> &node_idx, array<double> &out_elev);
        void ExportElevationLevelOff(double topo_den, const array<double> &press, const array<size_t> &node_idx, array<double> &out_elev);

        void BodyPressureGradient(const array<double> &den, const array<double> &press, array<double> &gradient);

        double ElevationObjectiveFunc(const array<double> &press, array<double> &out_grad);
        double ElevationObjectiveFunc(const array<double> &press, array<double> &out_grad, bool cal_pphys, 
            array<double> *temp_grad = nullptr, array<double> *press_grad = nullptr, array<double> *comp_grad = nullptr);

        double PressureObjectiveFunc(const array<double> &pphys_press, const array<double> &fem_press, 
            const array<double> &press_grad, array<double> &out_grad);
        //double PressureObjectiveFunc(const array<double> &pphys_press, const array<double> &fem_press, array<double> &out_grad, 
        //    array<double> *temp_grad = nullptr, array<double> *press_grad = nullptr, array<double> *comp_grad = nullptr);

        void set_gauss_order(size_t order);
        void set_pressure_error(double err);
        void set_elevation_error(double err);
        double get_pressure_error();

    protected:
        bool kernel_ready_, tar_ready_, err_ready_, elev_err_ready_, bnd_set_, bnd_changed_;
        size_t node_num_, elem_num_;

        std::vector<mat_node<double>> k_list_; // 以三元组储存的核矩阵元素
        array<mat_node<double>> ik_list_; // 有限元右端积分矩阵
        spmat<double> pressure_kernel_; // 有限元核矩阵（稀疏矩阵）

        array<bool> has_val_cndt_;
        array<double> val_cndt_;
        array<double> tar_;
        array<double> precndt_;
        array<double> bnd_correct_;

        gctl::linear_sf lsf_;

        double press_err_, elev_err_;

        size_t elev_size_;
        array<double> pre_elev_press_, bkg_elev_press_, obs_elev_press_;
        array<size_t> elev_index_;
        array<double> norm_vol_wgts_; // 归一化后的元素体积加权矩阵 用来平衡高程反演中由元素体积引起的模型梯度不平衡的现象

        array<double> elem_press_, elem_diff_, obs_grad_, node_grad_;

        ModelSpace *m_space_;
        Earth1D *earth_1d_;
    };
}

#endif // _GIST_PRESSURE_H