/********************************************************
 *     _____________________
 *    / ____/  _/ ___/_  __/
 *   / / __ / / \__ \ / /   
 *  / /_/ // / ___/ // /    
 *  \____/___//____//_/ 
 * Geophysical Inversions using Spherical Tetrahedral meshes (GIST)
 *
 * Copyright (c) 2022  Yi Zhang (yizhang-geo@zju.edu.cn)
 *
 * GIST is distributed under a dual licensing scheme. You can redistribute 
 * it and/or modify it under the terms of the GNU Affero General Public 
 * License (AGPL) as published by the Free Software Foundation, either version 
 * 3 of the License, or (at your option) any later version. You should have 
 * received a copy of the GNU Affero General Public License along with this 
 * program. If not, see <http://www.gnu.org/licenses/>.
 * 
 * If the terms and conditions of the AGPL v.3. would prevent you from using 
 * the GIST, please consider the option to obtain a commercial license for a 
 * fee. These licenses are offered by the original author, Yi Zhang. As a rule, 
 * licenses are provided "as-is", unlimited in time for a one time fee. Please 
 * send corresponding requests to: yizhang-geo@zju.edu.cn. Please do not forget 
 * to include some description of your company and the realm of its activities. 
 * Also add information on how to contact you by electronic and paper mail.
 ******************************************************/

#include "pressure.h"

GIST::Pressure::Pressure()
{
    tetrahedron::initiate(2);
    kernel_ready_ = tar_ready_ = err_ready_ = elev_err_ready_ = false;
    bnd_changed_ = bnd_set_ = false; // gwgt_ready_ = false;
}

GIST::Pressure::~Pressure(){}

void GIST::Pressure::LCG_Mx(const array<double> &x, array<double> &ax)
{
    vecdiv(ax, x, precndt_);
    return;
}

void GIST::Pressure::LCG_Ax(const array<double> &x, array<double> &ax)
{
    pressure_kernel_.multiply_vector(x, ax);
    return;
}

double GIST::Pressure::tetrahedron_func(double ksi, double eta, double zta, 
    double x1, double x2, double x3, double x4, 
    double y1, double y2, double y3, double y4,
    double z1, double z2, double z3, double z4, void *att)
{
    pressure_FEM_setup *pt = static_cast<pressure_FEM_setup*>(att);

    double x, y, z;
    lsf_.local2global_tetrahedron(x, y, z, ksi, eta, zta, x1, x2, x3, x4, y1, y2, y3, y4, z1, z2, z3, z4);
    
    // 注意这里实际上是计算了重力加速度的方向 重力加速度的值作为常量在函数外处理
    double nor = sqrt(x*x + y*y + z*z);
    x *= -1.0/nor; y *= -1.0/nor; z *= -1.0/nor;

    if (pt->i_type == PressureKernel)
    {
        // Un-comment to compute for spherical coordinates
        double g_rad0 = x*lsf_.tetrahedron(ksi, eta, zta, x1, x2, x3, x4, y1, y2, y3, y4, z1, z2, z3, z4, pt->uv_id[0], gctl::Dx)
            + y*lsf_.tetrahedron(ksi, eta, zta, x1, x2, x3, x4, y1, y2, y3, y4, z1, z2, z3, z4, pt->uv_id[0], gctl::Dy)
            + z*lsf_.tetrahedron(ksi, eta, zta, x1, x2, x3, x4, y1, y2, y3, y4, z1, z2, z3, z4, pt->uv_id[0], gctl::Dz);

        double g_rad1 = x*lsf_.tetrahedron(ksi, eta, zta, x1, x2, x3, x4, y1, y2, y3, y4, z1, z2, z3, z4,pt-> uv_id[1], gctl::Dx)
            + y*lsf_.tetrahedron(ksi, eta, zta, x1, x2, x3, x4, y1, y2, y3, y4, z1, z2, z3, z4, pt->uv_id[1], gctl::Dy)
            + z*lsf_.tetrahedron(ksi, eta, zta, x1, x2, x3, x4, y1, y2, y3, y4, z1, z2, z3, z4, pt->uv_id[1], gctl::Dz);

        return g_rad0*g_rad1;

        // Un-comment to compute for Cartesian coordinates
        //return lsf_.tetrahedron(ksi, eta, zta, x1, x2, x3, x4, y1, y2, y3, y4, z1, z2, z3, z4, pt->uv_id[0], gctl::Dz)*lsf_.tetrahedron(ksi, eta, zta, x1, x2, x3, x4, y1, y2, y3, y4, z1, z2, z3, z4, pt->uv_id[1], gctl::Dz);
    }
    else if (pt->i_type == PressureForce)
    {
        // Un-comment to compute for spherical coordinates
        return x*lsf_.tetrahedron(ksi, eta, zta, x1, x2, x3, x4, y1, y2, y3, y4, z1, z2, z3, z4, pt->uv_id[0], gctl::Dx)
            + y*lsf_.tetrahedron(ksi, eta, zta, x1, x2, x3, x4, y1, y2, y3, y4, z1, z2, z3, z4, pt->uv_id[0], gctl::Dy)
            + z*lsf_.tetrahedron(ksi, eta, zta, x1, x2, x3, x4, y1, y2, y3, y4, z1, z2, z3, z4, pt->uv_id[0], gctl::Dz);

        // Un-comment to compute for Cartesian coordinates
        //return lsf_.tetrahedron(ksi, eta, zta, x1, x2, x3, x4, y1, y2, y3, y4, z1, z2, z3, z4, pt->uv_id[0], gctl::Dz);
    }
    else throw std::runtime_error("Invalid integral type. From GIST::Pressure::glni_tetrahedron_func(...)");
    return 0.0;
}

void GIST::Pressure::set_gauss_order(size_t order)
{
    if (order <= 0)
    {
        throw gctl::runtime_error("Invalid gauss order. From GIST::Pressure::set_gauss_order(...)");
    }

    tetrahedron::clear();
    tetrahedron::initiate(order);
    return;
}

void GIST::Pressure::set_pressure_error(double err)
{
    press_err_ = err;
    err_ready_ = true;
    return;
}

void GIST::Pressure::set_elevation_error(double err)
{
    elev_err_ = err;
    elev_err_ready_ = true;
    return;
}

double GIST::Pressure::get_pressure_error()
{
    if (!err_ready_)
    {
        throw std::runtime_error("Data uncertainty of the pressure system is not set.");
    }

    return press_err_;
}