/********************************************************
 *     _____________________
 *    / ____/  _/ ___/_  __/
 *   / / __ / / \__ \ / /   
 *  / /_/ // / ___/ // /    
 *  \____/___//____//_/ 
 * Geophysical Inversions using Spherical Tetrahedral meshes (GIST)
 *
 * Copyright (c) 2022  Yi Zhang (yizhang-geo@zju.edu.cn)
 *
 * GIST is distributed under a dual licensing scheme. You can redistribute 
 * it and/or modify it under the terms of the GNU Affero General Public 
 * License (AGPL) as published by the Free Software Foundation, either version 
 * 3 of the License, or (at your option) any later version. You should have 
 * received a copy of the GNU Affero General Public License along with this 
 * program. If not, see <http://www.gnu.org/licenses/>.
 * 
 * If the terms and conditions of the AGPL v.3. would prevent you from using 
 * the GIST, please consider the option to obtain a commercial license for a 
 * fee. These licenses are offered by the original author, Yi Zhang. As a rule, 
 * licenses are provided "as-is", unlimited in time for a one time fee. Please 
 * send corresponding requests to: yizhang-geo@zju.edu.cn. Please do not forget 
 * to include some description of your company and the realm of its activities. 
 * Also add information on how to contact you by electronic and paper mail.
 ******************************************************/

#include "pressure.h"
#include "omp.h"

void GIST::Pressure::ImportModelSetup(ModelSpace *m_space, Earth1D *earth_1d)
{
    // Initiate variables
    earth_1d_ = earth_1d;
    m_space_ = m_space;
    node_num_ = m_space->get_node_number();
    elem_num_ = m_space->get_element_number();

    k_list_.resize(elem_num_ * 16);
    ik_list_.resize(elem_num_ * 4);
    pressure_kernel_.malloc(node_num_, node_num_, 0.0);

    tar_.resize(node_num_);
    bnd_correct_.resize(node_num_);

    // create an alias
    array<gctl::tetrahedron> &elem = m_space->get_element();

    vertex3dc *vt[4];
    mat_node<double> tmp_entry;
    pressure_FEM_setup pt;

    size_t i, j, f;
#pragma omp parallel for private (i, j, f, vt, tmp_entry, pt) schedule(guided)
    for (i = 0; i < elem_num_; i++)
	{
        for (j = 0; j < 4; j++)
        {
            vt[j] = elem[i].vert[j];
        }

        for (j = 0; j < 4; j++)
        {
            tmp_entry.r_id = vt[j]->id; // 行坐标为顶点索引
            tmp_entry.c_id = i; // 列坐标为单元体索引

            pt.uv_id[0] = j;
            pt.i_type = PressureForce;

            // Convert the pressure's unit to GPa with the density's unit is set to g/cm^3
            tmp_entry.val = 1e-6 * GIST_NORMAL_GRAVITY * tetrahedron::integral(
                vt[0]->x, vt[1]->x, vt[2]->x, vt[3]->x, vt[0]->y, vt[1]->y, vt[2]->y, vt[3]->y, 
                vt[0]->z, vt[1]->z, vt[2]->z, vt[3]->z, &pt);

            ik_list_[4*i + j] = tmp_entry;
        }

		for (j = 0; j < 4; j++)
		{
            tmp_entry.r_id = vt[j]->id;
            pt.uv_id[0] = j;
            pt.i_type = PressureKernel;

			for (f = 0; f < 4; f++)
			{                
                tmp_entry.c_id = vt[f]->id;
                pt.uv_id[1] = f;

                tmp_entry.val = tetrahedron::integral(
                    vt[0]->x, vt[1]->x, vt[2]->x, vt[3]->x, vt[0]->y, vt[1]->y, vt[2]->y, vt[3]->y, 
                    vt[0]->z, vt[1]->z, vt[2]->z, vt[3]->z, &pt);

                k_list_[16*i + 4*j + f] = tmp_entry;
			}
		}
	}

    // Sort triplts to increase in both row and column indexes
    std::sort(k_list_.begin(), k_list_.end(), [](mat_node<double> a, mat_node<double> b)->bool{
        if (a.r_id < b.r_id) return true;
        if (a.r_id == b.r_id && a.c_id < b.c_id) return true;
        return false;
    });

    kernel_ready_ = true;
    return;
}