/********************************************************
 *     _____________________
 *    / ____/  _/ ___/_  __/
 *   / / __ / / \__ \ / /   
 *  / /_/ // / ___/ // /    
 *  \____/___//____//_/ 
 * Geophysical Inversions using Spherical Tetrahedral meshes (GIST)
 *
 * Copyright (c) 2022  Yi Zhang (yizhang-geo@zju.edu.cn)
 *
 * GIST is distributed under a dual licensing scheme. You can redistribute 
 * it and/or modify it under the terms of the GNU Affero General Public 
 * License (AGPL) as published by the Free Software Foundation, either version 
 * 3 of the License, or (at your option) any later version. You should have 
 * received a copy of the GNU Affero General Public License along with this 
 * program. If not, see <http://www.gnu.org/licenses/>.
 * 
 * If the terms and conditions of the AGPL v.3. would prevent you from using 
 * the GIST, please consider the option to obtain a commercial license for a 
 * fee. These licenses are offered by the original author, Yi Zhang. As a rule, 
 * licenses are provided "as-is", unlimited in time for a one time fee. Please 
 * send corresponding requests to: yizhang-geo@zju.edu.cn. Please do not forget 
 * to include some description of your company and the realm of its activities. 
 * Also add information on how to contact you by electronic and paper mail.
 ******************************************************/

#include "pressure.h"

void GIST::Pressure::ExportElevation(double topo_den, double topo_press, const array<double> &press, 
    const array<size_t> &node_idx, array<double> &out_elev)
{
    press.extract_array(out_elev, node_idx);

    for (size_t i = 0; i < out_elev.size(); i++)
    {
        out_elev[i] -= topo_press;
        out_elev[i] = out_elev[i]*1e+6/(topo_den*GIST_NORMAL_GRAVITY);
    }

    if (elev_err_ready_)
    {
        array<double> noise(out_elev.size());
        random(noise, 0.0, elev_err_);
        out_elev += noise;
    }
    return;
}

void GIST::Pressure::ExportElevation(double topo_den, const array<double> &press, const array<double> &bkg_press, 
    const array<size_t> &node_idx, array<double> &out_elev)
{
    press.extract_array(out_elev, node_idx);

    array<double> bkg_elev;
    bkg_press.extract_array(bkg_elev, node_idx);

    for (size_t i = 0; i < out_elev.size(); i++)
    {
        out_elev[i] -= bkg_elev[i];
        out_elev[i] = out_elev[i]*1e+6/(topo_den*GIST_NORMAL_GRAVITY);
    }

    if (elev_err_ready_)
    {
        array<double> noise(out_elev.size());
        random(noise, 0.0, elev_err_);
        out_elev += noise;
    }
    return;
}

void GIST::Pressure::ExportElevationLevelOff(double topo_den, const array<double> &press, 
    const array<size_t> &node_idx, array<double> &out_elev)
{
    press.extract_array(out_elev, node_idx);

    double mean_press = 0.0;
    for (size_t i = 0; i < out_elev.size(); i++)
    {
        mean_press += out_elev[i];
    }
    mean_press /= out_elev.size();

    for (size_t i = 0; i < out_elev.size(); i++)
    {
        out_elev[i] -= mean_press;
        out_elev[i] = out_elev[i]*1e+6/(topo_den*GIST_NORMAL_GRAVITY);
    }

    if (elev_err_ready_)
    {
        array<double> noise(out_elev.size());
        random(noise, 0.0, elev_err_);
        out_elev += noise;
    }
    return;
}