/********************************************************
 *     _____________________
 *    / ____/  _/ ___/_  __/
 *   / / __ / / \__ \ / /   
 *  / /_/ // / ___/ // /    
 *  \____/___//____//_/ 
 * Geophysical Inversions using Spherical Tetrahedral meshes (GIST)
 *
 * Copyright (c) 2022  Yi Zhang (yizhang-geo@zju.edu.cn)
 *
 * GIST is distributed under a dual licensing scheme. You can redistribute 
 * it and/or modify it under the terms of the GNU Affero General Public 
 * License (AGPL) as published by the Free Software Foundation, either version 
 * 3 of the License, or (at your option) any later version. You should have 
 * received a copy of the GNU Affero General Public License along with this 
 * program. If not, see <http://www.gnu.org/licenses/>.
 * 
 * If the terms and conditions of the AGPL v.3. would prevent you from using 
 * the GIST, please consider the option to obtain a commercial license for a 
 * fee. These licenses are offered by the original author, Yi Zhang. As a rule, 
 * licenses are provided "as-is", unlimited in time for a one time fee. Please 
 * send corresponding requests to: yizhang-geo@zju.edu.cn. Please do not forget 
 * to include some description of your company and the realm of its activities. 
 * Also add information on how to contact you by electronic and paper mail.
 ******************************************************/

#include "pressure.h"

double GIST::Pressure::ElevationObjectiveFunc(const array<double> &press, array<double> &out_grad)
{
    if (!err_ready_ || !tar_ready_)
    {
        throw std::runtime_error("Data uncertainty or targeting pressure is not set. From GIST::Pressure::ElevationObjectiveFunc(...)");
    }

    // Sub-extract pressure value at top surface
    for (size_t i = 0; i < elev_size_; i++)
    {
        pre_elev_press_[i] = press[elev_index_[i]] - bkg_elev_press_[i];
    }

    // Elevation pressure only has gradients at the evaluated nodes
    obs_grad_.resize(node_num_, 0.0);
    for (size_t i = 0; i < elev_size_; i++)
    {
        obs_grad_[elev_index_[i]] = 2.0*(pre_elev_press_[i] - obs_elev_press_[i])/(press_err_*press_err_*elev_size_);
    }

    // Since boundary values are fixed in the FEM calculation. There are no gradients at these nodes.
    for (size_t i = 0; i < node_num_; i++)
    {
        if (has_val_cndt_[i])
        {
            obs_grad_[i] = 0.0;
        }
    }
    
    node_grad_.resize(node_num_, 0.0);
    LCG_Minimize(node_grad_, obs_grad_, gctl::LCG_PCG);

    out_grad.resize(elem_num_, 0.0);
    for (size_t i = 0; i < 4*elem_num_; i++)
    {
        out_grad[ik_list_[i].c_id] += ik_list_[i].val * node_grad_[ik_list_[i].r_id];
    }

    // Apply normalized volume weights
    for (size_t i = 0; i < elem_num_; i++)
    {
        out_grad[i] /= norm_vol_wgts_[i];
    }

    double mini_grad = 1e+30, maxi_grad = -1e+30;
    for (size_t i = 0; i < elem_num_; i++)
    {
        mini_grad = std::min(mini_grad, out_grad[i]);
        maxi_grad = std::max(maxi_grad, out_grad[i]);
    }

    for (size_t i = 0; i < elem_num_; i++)
    {
        out_grad[i] = out_grad[i]/(maxi_grad - mini_grad);
    }

    double msf = 0.0;
    for (size_t i = 0; i < elev_size_; i++)
    {
        msf += pow((pre_elev_press_[i] - obs_elev_press_[i])/press_err_, 2);
    }
    return msf/elev_size_;
}

double GIST::Pressure::ElevationObjectiveFunc(const array<double> &press, array<double> &out_grad, 
    bool cal_pphys, array<double> *temp_grad, array<double> *press_grad, array<double> *comp_grad)
{
    if (!err_ready_ || !tar_ready_)
    {
        throw std::runtime_error("Data uncertainty or targeting pressure is not set.");
    }

    // Sub-extract pressure value at top surface
    for (size_t i = 0; i < elev_size_; i++)
    {
        pre_elev_press_[i] = press[elev_index_[i]] - bkg_elev_press_[i];
    }

    // Elevation pressure only has gradients at the evaluated nodes
    obs_grad_.resize(node_num_, 0.0);
    for (size_t i = 0; i < elev_size_; i++)
    {
        obs_grad_[elev_index_[i]] = 2.0*(pre_elev_press_[i] - obs_elev_press_[i])/(press_err_*press_err_*elev_size_);
    }

    // Since boundary values are fixed in the FEM calculation. There are no gradients at these nodes.
    for (size_t i = 0; i < node_num_; i++)
    {
        if (has_val_cndt_[i])
        {
            obs_grad_[i] = 0.0;
        }
    }
    
    node_grad_.resize(node_num_, 0.0);
    LCG_Minimize(node_grad_, obs_grad_, gctl::LCG_PCG);

    if (cal_pphys)
    {
        out_grad.resize(3*elem_num_, 0.0);

        for (size_t i = 0; i < 4*elem_num_; i++)
        {
            out_grad[ik_list_[i].c_id] += ik_list_[i].val * node_grad_[ik_list_[i].r_id];
        }

        // Apply normalized volume weights
        for (size_t i = 0; i < elem_num_; i++)
        {
            out_grad[i] *= 1.0/norm_vol_wgts_[i];
        }

        double mini_grad = 1e+30, maxi_grad = -1e+30;
        for (size_t i = 0; i < elem_num_; i++)
        {
            mini_grad = std::min(mini_grad, out_grad[i]);
            maxi_grad = std::max(maxi_grad, out_grad[i]);
        }

        for (size_t i = 0; i < elem_num_; i++)
        {
            out_grad[i] = out_grad[i]/(maxi_grad - mini_grad);
        }

        for (size_t i = 0; i < elem_num_; i++)
        {
            out_grad[i + elem_num_] = out_grad[i];
            out_grad[i + 2*elem_num_] = out_grad[i];
        }

        if (temp_grad != nullptr)
        {
            for (size_t i = 0; i < elem_num_; i++)
            {
                out_grad[i] = out_grad[i] * temp_grad->at(i);
            }
        }
        else out_grad.assign(0.0, 0, elem_num_);

        if (press_grad != nullptr)
        {
            for (size_t i = 0; i < elem_num_; i++)
            {
                out_grad[i + elem_num_] = out_grad[i + elem_num_] * press_grad->at(i);
            }
        }
        else out_grad.assign(0.0, elem_num_, elem_num_);

        if (comp_grad != nullptr)
        {
            for (size_t i = 0; i < elem_num_; i++)
            {
                out_grad[i + 2*elem_num_] = out_grad[i + 2*elem_num_] * comp_grad->at(i);
            }
        }
        else out_grad.assign(0.0, 2*elem_num_, elem_num_);
    }
    else
    {
        out_grad.resize(elem_num_, 0.0);

        for (size_t i = 0; i < 4*elem_num_; i++)
        {
            out_grad[ik_list_[i].c_id] += ik_list_[i].val * node_grad_[ik_list_[i].r_id];
        }

        // Apply normalized volume weights
        for (size_t i = 0; i < elem_num_; i++)
        {
            out_grad[i] *= 1.0/norm_vol_wgts_[i];
        }
    }

    double msf = 0.0;
    for (size_t i = 0; i < elev_size_; i++)
    {
        msf += pow((pre_elev_press_[i] - obs_elev_press_[i])/press_err_, 2);
    }
    return msf/elev_size_;
}