/********************************************************
 *     _____________________
 *    / ____/  _/ ___/_  __/
 *   / / __ / / \__ \ / /   
 *  / /_/ // / ___/ // /    
 *  \____/___//____//_/ 
 * Geophysical Inversions using Spherical Tetrahedral meshes (GIST)
 *
 * Copyright (c) 2022  Yi Zhang (yizhang-geo@zju.edu.cn)
 *
 * GIST is distributed under a dual licensing scheme. You can redistribute 
 * it and/or modify it under the terms of the GNU Affero General Public 
 * License (AGPL) as published by the Free Software Foundation, either version 
 * 3 of the License, or (at your option) any later version. You should have 
 * received a copy of the GNU Affero General Public License along with this 
 * program. If not, see <http://www.gnu.org/licenses/>.
 * 
 * If the terms and conditions of the AGPL v.3. would prevent you from using 
 * the GIST, please consider the option to obtain a commercial license for a 
 * fee. These licenses are offered by the original author, Yi Zhang. As a rule, 
 * licenses are provided "as-is", unlimited in time for a one time fee. Please 
 * send corresponding requests to: yizhang-geo@zju.edu.cn. Please do not forget 
 * to include some description of your company and the realm of its activities. 
 * Also add information on how to contact you by electronic and paper mail.
 ******************************************************/

#include "pressure.h"

void GIST::Pressure::BodyPressureGradient(const array<double> &den, const array<double> &press, array<double> &gradient)
{
    // create aliases
    array<gctl::tetrahedron> &elem = m_space_->get_element();
    array<gctl::triangle> &face = m_space_->get_face();
    array<gctl::vertex3dc> &node = m_space_->get_node();

    // 初始化梯度矩阵
    spmat<double> gx_kernal(node_num_, node_num_, 0.0);
    spmat<double> gy_kernal(node_num_, node_num_, 0.0);
    spmat<double> gz_kernal(node_num_, node_num_, 0.0);
    std::vector<std::vector<gctl::tetrahedron*> > node_hosts;

    node_hosts.resize(node_num_);
    for (size_t i = 0; i < elem_num_; i++)
    {
        for (size_t j = 0; j < 4; j++)
        {
            node_hosts[elem[i].vert[j]->id].push_back(elem.get(i));   
        }
    }

    double ksi, eta, zta, wgtx, wgty, wgtz;
    vertex3dc *vt[4];

    double total_vol, vol_den;
    for (size_t i = 0; i < node_num_; i++)
    {
        total_vol = 0.0;
        for (size_t j = 0; j < node_hosts[i].size(); j++)
        {
            total_vol += 1.0/cbrt(3.0*node_hosts[i][j]->volume()); // 以等效块体的边长的倒数作为插值的权重
        }

        for (size_t j = 0; j < node_hosts[i].size(); j++)
        {
            for (size_t k = 0; k < 4; k++)
            {
                vt[k] = node_hosts[i][j]->vert[k];
            }

            vol_den = den[node_hosts[i][j]->id];

            lsf_.global2local_tetrahedron(ksi, eta, zta, node[i].x, node[i].y, node[i].z, 
                vt[0]->x, vt[1]->x, vt[2]->x, vt[3]->x, vt[0]->y, vt[1]->y, vt[2]->y, vt[3]->y, 
                vt[0]->z, vt[1]->z, vt[2]->z, vt[3]->z);

            for (size_t k = 0; k < 4; k++)
            {
                wgtx = -1.0/(cbrt(3.0*node_hosts[i][j]->volume())*total_vol)*lsf_.tetrahedron(ksi, eta, zta, vt[0]->x, vt[1]->x, vt[2]->x, vt[3]->x, 
                    vt[0]->y, vt[1]->y, vt[2]->y, vt[3]->y, vt[0]->z, vt[1]->z, vt[2]->z, vt[3]->z, k, gctl::Dx);
                wgty = -1.0/(cbrt(3.0*node_hosts[i][j]->volume())*total_vol)*lsf_.tetrahedron(ksi, eta, zta, vt[0]->x, vt[1]->x, vt[2]->x, vt[3]->x, 
                    vt[0]->y, vt[1]->y, vt[2]->y, vt[3]->y, vt[0]->z, vt[1]->z, vt[2]->z, vt[3]->z, k, gctl::Dy);
                wgtz = -1.0/(cbrt(3.0*node_hosts[i][j]->volume())*total_vol)*lsf_.tetrahedron(ksi, eta, zta, vt[0]->x, vt[1]->x, vt[2]->x, vt[3]->x, 
                    vt[0]->y, vt[1]->y, vt[2]->y, vt[3]->y, vt[0]->z, vt[1]->z, vt[2]->z, vt[3]->z, k, gctl::Dz);

                gx_kernal.add_scalar(i, node_hosts[i][j]->vert[k]->id, wgtx/vol_den);
                gy_kernal.add_scalar(i, node_hosts[i][j]->vert[k]->id, wgty/vol_den);
                gz_kernal.add_scalar(i, node_hosts[i][j]->vert[k]->id, wgtz/vol_den);
            }
        }
    }

    array<double> gx, gy, gz;
    gx_kernal.multiply_vector(press, gx);
    gy_kernal.multiply_vector(press, gy);
    gz_kernal.multiply_vector(press, gz);

    gradient.resize(node_num_);

    double x, y, z, nor;
    for (size_t i = 0; i < node_num_; i++)
    {
        nor = sqrt(node[i].x * node[i].x + node[i].y * node[i].y + node[i].z * node[i].z);
        x = -1.0*node[i].x/nor; y = -1.0*node[i].y/nor; z = -1.0*node[i].z/nor;

        gradient[i] = x*gx[i] + y*gy[i] + z*gz[i];
    }

    return;
}