/********************************************************
 *     _____________________
 *    / ____/  _/ ___/_  __/
 *   / / __ / / \__ \ / /   
 *  / /_/ // / ___/ // /    
 *  \____/___//____//_/ 
 * Geophysical Inversions using Spherical Tetrahedral meshes (GIST)
 *
 * Copyright (c) 2022  Yi Zhang (yizhang-geo@zju.edu.cn)
 *
 * GIST is distributed under a dual licensing scheme. You can redistribute 
 * it and/or modify it under the terms of the GNU Affero General Public 
 * License (AGPL) as published by the Free Software Foundation, either version 
 * 3 of the License, or (at your option) any later version. You should have 
 * received a copy of the GNU Affero General Public License along with this 
 * program. If not, see <http://www.gnu.org/licenses/>.
 * 
 * If the terms and conditions of the AGPL v.3. would prevent you from using 
 * the GIST, please consider the option to obtain a commercial license for a 
 * fee. These licenses are offered by the original author, Yi Zhang. As a rule, 
 * licenses are provided "as-is", unlimited in time for a one time fee. Please 
 * send corresponding requests to: yizhang-geo@zju.edu.cn. Please do not forget 
 * to include some description of your company and the realm of its activities. 
 * Also add information on how to contact you by electronic and paper mail.
 ******************************************************/

#include "model_space.h"

void GIST::ModelSpace::SaveData2Text(std::string filename, std::string dataname, 
    const Earth1D &ref_earth, gctl::mesh_data_type_e d_type, const array<double> &data, array<size_t> *index)
{
    std::ofstream ofile;
    open_outfile(ofile, filename, ".txt");

    ofile << "# reference system: " << std::to_string(ref_earth.get_minor_radius()) << "/" << std::to_string(ref_earth.get_major_radius()) << "\n";
    ofile << "# observation number\n" << data.size() << "\n";
    ofile << "# lon(deg) lat(deg) elev(m) " << dataname << "\n";

    double ele;
    if (index != nullptr)
    {
        if (data.size() != index->size())
        {
            throw std::runtime_error("Unequal arrays' size. From GIST::ModelSpace::SaveData2Text(...)");
        }

        size_t id;
        gctl::point3ds ps;
        if (d_type == gctl::ElemData)
        {
            for (size_t i = 0; i < data.size(); i++)
            {
                id = index->at(i);
                ps = elems_[id].center().c2s();

                ele = ps.rad - ref_earth.get_radius(ps.lat);
                ofile << ps.lon << " " << ps.lat << " " << ele << " " << data[i] << "\n";
            }
        }
        else if (d_type == gctl::NodeData)
        {
            for (size_t i = 0; i < data.size(); i++)
            {
                id = index->at(i);
                ps = nodes_[id].c2s();

                ele = ps.rad - ref_earth.get_radius(ps.lat);
                ofile << ps.lon << " " << ps.lat << " " << ele << " " << data[i] << "\n";
            }
        }
        else throw std::runtime_error("Invalid model data type. From GIST::ModelSpace::SaveData2Text(...)");
    }
    else
    {
        gctl::point3ds ps;
        if (d_type == gctl::ElemData)
        {
            for (size_t i = 0; i < elem_num_; i++)
            {
                ps = elems_[i].center().c2s();

                ele = ps.rad - ref_earth.get_radius(ps.lat);
                ofile << ps.lon << " " << ps.lat << " " << ele << " " << data[i] << "\n";
            }
        }
        else if (d_type == gctl::NodeData)
        {
            for (size_t i = 0; i < node_num_; i++)
            {
                ps = nodes_[i].c2s();

                ele = ps.rad - ref_earth.get_radius(ps.lat);
                ofile << ps.lon << " " << ps.lat << " " << ele << " " << data[i] << "\n";
            }
        }
        else throw std::runtime_error("Invalid model data type. From GIST::ModelSpace::SaveData2Text(...)");
    }

    ofile.close();
	return;
}