/********************************************************
 *     _____________________
 *    / ____/  _/ ___/_  __/
 *   / / __ / / \__ \ / /   
 *  / /_/ // / ___/ // /    
 *  \____/___//____//_/ 
 * Geophysical Inversions using Spherical Tetrahedral meshes (GIST)
 *
 * Copyright (c) 2022  Yi Zhang (yizhang-geo@zju.edu.cn)
 *
 * GIST is distributed under a dual licensing scheme. You can redistribute 
 * it and/or modify it under the terms of the GNU Affero General Public 
 * License (AGPL) as published by the Free Software Foundation, either version 
 * 3 of the License, or (at your option) any later version. You should have 
 * received a copy of the GNU Affero General Public License along with this 
 * program. If not, see <http://www.gnu.org/licenses/>.
 * 
 * If the terms and conditions of the AGPL v.3. would prevent you from using 
 * the GIST, please consider the option to obtain a commercial license for a 
 * fee. These licenses are offered by the original author, Yi Zhang. As a rule, 
 * licenses are provided "as-is", unlimited in time for a one time fee. Please 
 * send corresponding requests to: yizhang-geo@zju.edu.cn. Please do not forget 
 * to include some description of your company and the realm of its activities. 
 * Also add information on how to contact you by electronic and paper mail.
 ******************************************************/

#include "model_space.h"

void GIST::ModelSpace::ReadMesh(std::string filename, index_packed_e filemode, msh_tag_e with_tags)
{
    fio_.init_file(filename, Input);
    fio_.set_packed(filemode, Input);

    if (with_tags)
    {
        _2i_vector etags;
        fio_.read_mesh(elems_, nodes_, &etags);

        node_num_ = nodes_.size();
        elem_num_  = elems_.size();

        etags_.resize(elem_num_, etags[0].size()); // all tags, including both physical tags, geometric tags and others.
        elem_tags_.resize(elem_num_, 0); // only physical tags.
        for (size_t i = 0; i < elem_num_; i++)
        {
            elem_tags_[i] = etags[i][0];

            for (size_t j = 0; j < etags[i].size(); j++)
            {
                etags_[i][j] = etags[i][j];   
            }
        }
        destroy_vector(etags);

        node_etags_.resize(node_num_, 0); // element associated physical tag for nodes.
        for (size_t i = 0; i < elem_num_; i++)
        {
            for (size_t j = 0; j < 4; j++)
            {
                node_etags_[elems_[i].vert[j]->id] = elem_tags_[i];
            }
        }

        elem_tag_ready_ = true;
    }
    else
    {
        fio_.read_mesh(elems_, nodes_);

        node_num_ = nodes_.size();
        elem_num_  = elems_.size();
    }

    // sort neighbors
    geometry3d::sort_node_neighbor(elems_, node_neigh_, &node_num_);
    geometry3d::sort_tet_neighbor(elems_, &node_num_);

    // Initiate mesh volumes
    elem_vols_.resize(elem_num_);
    for (size_t i = 0; i < elem_num_; i++)
    {
        elem_vols_[i] = elems_[i].volume();
    }

    // 初始化顶点到单元体属性的插值矩阵
    n2e_kernel_.malloc(elem_num_, node_num_, 0.0);

    double ksi, eta, zta, wgt;
    point3dc cen;
    vertex3dc *vt[4];
    for (size_t i = 0; i < elem_num_; i++)
    {
        // 元素属性的插值位置为单元体的中心
        // 注意这里也可以在单元体内按照高斯了让德积分采点并计算，以后有时间再说
        cen = elems_[i].center();

        for (size_t j = 0; j < 4; j++)
        {
            vt[j] = elems_[i].vert[j];
        }

        lsf_.global2local_tetrahedron(ksi, eta, zta, cen.x, cen.y, cen.z, 
            vt[0]->x, vt[1]->x, vt[2]->x, vt[3]->x, 
            vt[0]->y, vt[1]->y, vt[2]->y, vt[3]->y, 
            vt[0]->z, vt[1]->z, vt[2]->z, vt[3]->z);
        
        for (size_t j = 0; j < 4; j++)
        {
            wgt = lsf_.tetrahedron(ksi, eta, zta, 
                vt[0]->x, vt[1]->x, vt[2]->x, vt[3]->x, 
                vt[0]->y, vt[1]->y, vt[2]->y, vt[3]->y, 
                vt[0]->z, vt[1]->z, vt[2]->z, vt[3]->z, j, gctl::Value);

            n2e_kernel_.insert(i, elems_[i].vert[j]->id, wgt);
        }
    }

    // 初始化单元体到顶点属性的插值矩阵
    e2n_kernel_.malloc(node_num_, elem_num_, 0.0);

    double f;
    for (size_t i = 0; i < node_neigh_.size(); i++)
    {
        f = node_neigh_[i].size();
        for (size_t j = 0; j < node_neigh_[i].size(); j++)
        {
            e2n_kernel_.insert(i, node_neigh_[i][j]->id, 1.0/f);
        }
    }

    // 初始化公共面及其相邻元素的信息
    // Sort common faces. Firstly find pairs of tetrahedron neighbors
    std::vector<std::vector<int> > face_n; // destory after use
    face_n.resize(elem_num_);
    for (size_t i = 0; i < elem_num_; i++)
    {
        face_n[i].reserve(8); // 每个list最多8个
    }

    int small_id, big_id;
    for (size_t i = 0; i < elem_num_; i++)
    {
        for (size_t j = 0; j < 4; j++)
        {
            if (elems_[i].neigh[j] != nullptr) // 邻居必须存在
            {
                // 这里我们总是按小-大排序序号，保证不会输入混乱
                small_id = std::min(elems_[i].id, elems_[i].neigh[j]->id);
                big_id   = std::max(elems_[i].id, elems_[i].neigh[j]->id);
                face_n[small_id].push_back(big_id);
            }
        }
    }

    // 去除face_n中的重复输入
    std::vector<int>::iterator pos;
    for (size_t i = 0; i < elem_num_; i++)
    {
        if (!face_n[i].empty()) // 很有可能某个pair没有输入
        {
            std::sort(face_n[i].begin(), face_n[i].end()); //对顶点序列由小到大排序
            pos = std::unique(face_n[i].begin(), face_n[i].end()); //获取重复序列开始的位置
            face_n[i].erase(pos, face_n[i].end()); //删除重复点
        }
    }

    // 保存三角形的公共边的列表 首先统计公共面的数量
    cmn_face_num_ = 0;
    for (size_t i = 0; i < elem_num_; i++)
    {
        cmn_face_num_ += face_n[i].size();
    }
    cmn_faces_.resize(cmn_face_num_);

    // 找出公共面两侧的邻接四面体
    size_t c = 0;
    for (size_t i = 0; i < elem_num_; i++)
    {
        if (!face_n[i].empty())
        {
            for (size_t j = 0; j < face_n[i].size(); j++)
            {
                cmn_faces_[c].tet_ptr[0] = elems_.get(i);
                cmn_faces_[c].tet_ptr[1] = elems_.get(face_n[i][j]);
                c++;
            }
        }
    }

    // 找出公共面的顶点
    gctl::vertex3dc *tmp_vert;
    for (size_t i = 0; i < cmn_face_num_; i++)
    {
        for (size_t j = 0; j < 4; j++)
        {
            tmp_vert = cmn_faces_[i].tet_ptr[0]->vert[j];
            if (tmp_vert != cmn_faces_[i].tet_ptr[1]->vert[0] &&
                tmp_vert != cmn_faces_[i].tet_ptr[1]->vert[1] &&
                tmp_vert != cmn_faces_[i].tet_ptr[1]->vert[2] &&
                tmp_vert != cmn_faces_[i].tet_ptr[1]->vert[3])
            {
                cmn_faces_[i].vert_ptr[0] = cmn_faces_[i].tet_ptr[0]->vert[(j+1)%4];
                cmn_faces_[i].vert_ptr[1] = cmn_faces_[i].tet_ptr[0]->vert[(j+2)%4];
                cmn_faces_[i].vert_ptr[2] = cmn_faces_[i].tet_ptr[0]->vert[(j+3)%4];
                break;
            }
        }
    }
    
    in_ready_ = true;
    return;
}