/********************************************************
 *     _____________________
 *    / ____/  _/ ___/_  __/
 *   / / __ / / \__ \ / /   
 *  / /_/ // / ___/ // /    
 *  \____/___//____//_/ 
 * Geophysical Inversions using Spherical Tetrahedral meshes (GIST)
 *
 * Copyright (c) 2022  Yi Zhang (yizhang-geo@zju.edu.cn)
 *
 * GIST is distributed under a dual licensing scheme. You can redistribute 
 * it and/or modify it under the terms of the GNU Affero General Public 
 * License (AGPL) as published by the Free Software Foundation, either version 
 * 3 of the License, or (at your option) any later version. You should have 
 * received a copy of the GNU Affero General Public License along with this 
 * program. If not, see <http://www.gnu.org/licenses/>.
 * 
 * If the terms and conditions of the AGPL v.3. would prevent you from using 
 * the GIST, please consider the option to obtain a commercial license for a 
 * fee. These licenses are offered by the original author, Yi Zhang. As a rule, 
 * licenses are provided "as-is", unlimited in time for a one time fee. Please 
 * send corresponding requests to: yizhang-geo@zju.edu.cn. Please do not forget 
 * to include some description of your company and the realm of its activities. 
 * Also add information on how to contact you by electronic and paper mail.
 ******************************************************/

#include "model_space.h"

// 单调模型约束激活函数
double active_func(double diff)
{
    if (diff > 0) return diff;
    else return 0.0;
}

double active_grad(double diff)
{
    if (diff > 0) return 1.0;
    else return 0.0;
}

double GIST::ModelSpace::MonotonicModelConstraint(const array<double> &model, array<double> &out_grad)
{
    if (!monowgt_ready_)
    {
        throw std::runtime_error("[GIST::ModelSpace::MonotonicModelConstraint] Monotonic model weight is not initialized.");
    }

    // 计算模型单调性
    mono_mat_.multiply_vector(model, face_diff_);
    // 乘以激活函数
    for (size_t i = 0; i < face_diff_.size(); i++)
    {
        face_diff_[i] = active_func(face_diff_[i]);
    }

    double sum = 0.0;
    for (size_t i = 0; i < cmn_face_num_; i++)
    {
        sum += face_diff_[i]*face_diff_[i];
    }
    sum *= mono_wgt_;

    int id1, id2;
    double grad;
    out_grad.resize(elem_num_, 0.0);
    for (size_t i = 0; i < cmn_face_num_; i++)
    {
        id1 = cmn_faces_[i].tet_ptr[0]->id;
        id2 = cmn_faces_[i].tet_ptr[1]->id;
        grad = active_grad(face_diff_[i]);

        out_grad[id1] += 2.0*mono_wgt_*face_diff_[i]*grad*mono_mat_.at(i, id1);
        out_grad[id2] += 2.0*mono_wgt_*face_diff_[i]*grad*mono_mat_.at(i, id2);
    }

    return sum;
}
