/********************************************************
 *     _____________________
 *    / ____/  _/ ___/_  __/
 *   / / __ / / \__ \ / /   
 *  / /_/ // / ___/ // /    
 *  \____/___//____//_/ 
 * Geophysical Inversions using Spherical Tetrahedral meshes (GIST)
 *
 * Copyright (c) 2022  Yi Zhang (yizhang-geo@zju.edu.cn)
 *
 * GIST is distributed under a dual licensing scheme. You can redistribute 
 * it and/or modify it under the terms of the GNU Affero General Public 
 * License (AGPL) as published by the Free Software Foundation, either version 
 * 3 of the License, or (at your option) any later version. You should have 
 * received a copy of the GNU Affero General Public License along with this 
 * program. If not, see <http://www.gnu.org/licenses/>.
 * 
 * If the terms and conditions of the AGPL v.3. would prevent you from using 
 * the GIST, please consider the option to obtain a commercial license for a 
 * fee. These licenses are offered by the original author, Yi Zhang. As a rule, 
 * licenses are provided "as-is", unlimited in time for a one time fee. Please 
 * send corresponding requests to: yizhang-geo@zju.edu.cn. Please do not forget 
 * to include some description of your company and the realm of its activities. 
 * Also add information on how to contact you by electronic and paper mail.
 ******************************************************/

#include "model_space.h"

GIST::ModelSpace::ModelSpace()
{
    node_num_ = face_num_ = cmn_face_num_ = elem_num_ = 0;
    in_ready_ = out_ready_ = face_ready_ = phys_ready_ = false;
    face_tag_ready_ = elem_tag_ready_ = node_tag_ready_ = false;
    mwgt_ready_ = swgt_ready_ = rwgt_ready_ = modwgt_ready_ = monowgt_ready_ = false;
    vwgt_ready_ = dwgt_ready_ = false;
}

GIST::ModelSpace::~ModelSpace()
{
    node_num_ = face_num_ = cmn_face_num_ = elem_num_ = 0;
    in_ready_ = out_ready_ = face_ready_ = phys_ready_ = false;
    face_tag_ready_ = elem_tag_ready_ = node_tag_ready_ = false;
    mwgt_ready_ = swgt_ready_ = rwgt_ready_ = modwgt_ready_ = monowgt_ready_ = false;
    vwgt_ready_ = dwgt_ready_ = false;
}

array<vertex3dc> &GIST::ModelSpace::get_node()
{
    if (in_ready_ == false)
    {
        throw gctl::runtime_error("Model space not initialized. From GIST::ModelSpace::get_node(...)");
    }

    return nodes_;
}

array<enode> &GIST::ModelSpace::get_enode()
{
    if (enode_num_ == 0)
    {
        throw gctl::runtime_error("No element-nodes found. From GIST::ModelSpace::get_enode(...)");
    }

    return enodes_;
}

array<triangle> &GIST::ModelSpace::get_face()
{
    if (face_ready_ == false)
    {
        throw gctl::runtime_error("Faces not initialized. From GIST::ModelSpace::get_face(...)");
    }

    return faces_;
}

array<tetrahedron> &GIST::ModelSpace::get_element()
{
    if (in_ready_ == false)
    {
        throw gctl::runtime_error("Model space not initialized. From GIST::ModelSpace::get_element(...)");
    }

    return elems_;
}

array<gmsh_physical_group> &GIST::ModelSpace::get_physics()
{
    if (phys_ready_ == false)
    {
        throw gctl::runtime_error("Physics are not initialized. From GIST::ModelSpace::get_physics(...)");
    }

    return phys_;
}

array<double> &GIST::ModelSpace::get_volume()
{
    if (in_ready_ == false)
    {
        throw gctl::runtime_error("The model space is not initialized. From GIST::ModelSpace::get_volume(...)");
    }

    return elem_vols_;
}

array<int> &GIST::ModelSpace::get_tag(tag_type_e tag_t)
{
    if (tag_t == Node)
    {
        if (!node_tag_ready_)
        {
            throw std::runtime_error("Node tags are not initialized. From GIST::ModelSpace::get_tag(...)");
        }
        
        return enode_tags_;
    }
    
    if (tag_t == NodeF || tag_t == Face)
    {
        if (!face_tag_ready_)
        {
            throw std::runtime_error("Face tags are not initialized. From GIST::ModelSpace::get_tag(...)");
        }
        
        if (tag_t == NodeF) return node_ftags_;
        else return face_tags_; // tag_t = Face
    }

    if (!elem_tag_ready_)
    {
        throw std::runtime_error("Element tags are not initialized. From GIST::ModelSpace::get_tag(...)");
    }
    
    if (tag_t == NodeE) return node_etags_;
    else return elem_tags_; // tag_t == Element
}

size_t GIST::ModelSpace::get_node_number() const
{
    if (in_ready_ == false)
    {
        throw gctl::runtime_error("Model space not initialized. From GIST::ModelSpace::get_node_number(...)");
    }

    return node_num_;
}

size_t GIST::ModelSpace::get_face_number() const
{
    if (face_ready_ == false)
    {
        throw gctl::runtime_error("Model space not initialized. From GIST::ModelSpace::get_face_number(...)");
    }

    return face_num_;
}

size_t GIST::ModelSpace::get_element_number() const
{
    if (in_ready_ == false)
    {
        throw gctl::runtime_error("Model space not initialized. From GIST::ModelSpace::get_element_number(...)");
    }

    return elem_num_;
}

point3dc GIST::ModelSpace::get_element_center(size_t id)
{
    if (in_ready_ == false)
    {
        throw gctl::runtime_error("Model space not initialized. From GIST::ModelSpace::get_element_center(...)");
    }

    if (id >= elem_num_)
    {
        throw gctl::runtime_error("Invalid element index. From GIST::ModelSpace::get_element_center(...)");
    }

    return elems_[id].center();
}

array<double> &GIST::ModelSpace::get_depth_weight()
{
    if (!dwgt_ready_)
    {
        throw std::runtime_error("Depth weights are not initialized. From GIST::ModelSpace::get_depth_weight().");
    }
    
    return dep_wgts_;
}

array<double> &GIST::ModelSpace::get_volume_weight()
{
    if (!vwgt_ready_)
    {
        throw std::runtime_error("Volume weights are not initialized. From GIST::ModelSpace::get_volume_weight().");
    }

    return vol_wgts_;
}

array<double> &GIST::ModelSpace::get_model_weight()
{
    if (!modwgt_ready_)
    {
        throw std::runtime_error("Model weights are not initialized. From GIST::ModelSpace::get_model_weight().");
    }

    return model_wgts_;
}