/********************************************************
 *     _____________________
 *    / ____/  _/ ___/_  __/
 *   / / __ / / \__ \ / /   
 *  / /_/ // / ___/ // /    
 *  \____/___//____//_/ 
 * Geophysical Inversions using Spherical Tetrahedral meshes (GIST)
 *
 * Copyright (c) 2022  Yi Zhang (yizhang-geo@zju.edu.cn)
 *
 * GIST is distributed under a dual licensing scheme. You can redistribute 
 * it and/or modify it under the terms of the GNU Affero General Public 
 * License (AGPL) as published by the Free Software Foundation, either version 
 * 3 of the License, or (at your option) any later version. You should have 
 * received a copy of the GNU Affero General Public License along with this 
 * program. If not, see <http://www.gnu.org/licenses/>.
 * 
 * If the terms and conditions of the AGPL v.3. would prevent you from using 
 * the GIST, please consider the option to obtain a commercial license for a 
 * fee. These licenses are offered by the original author, Yi Zhang. As a rule, 
 * licenses are provided "as-is", unlimited in time for a one time fee. Please 
 * send corresponding requests to: yizhang-geo@zju.edu.cn. Please do not forget 
 * to include some description of your company and the realm of its activities. 
 * Also add information on how to contact you by electronic and paper mail.
 ******************************************************/

#include "model_space.h"

void GIST::ModelSpace::InitMesh(array<vertex3dc> &nodes, array<triangle> &faces, array<tetrahedron> &elems, 
    array<gmsh_physical_group> &phys, matrix<int> &etags, matrix<int> &ftags)
{
    // Initiate basic parameters
    node_num_ = nodes.size();
    face_num_ = faces.size();
    elem_num_ = elems.size();

    // Initiate faces and elements
    nodes_.resize(node_num_);
    for (size_t i = 0; i < node_num_; i++)
    {
        nodes_[i] = nodes[i];
        nodes_[i].id = i;
    }

    faces_.resize(face_num_);
    for (size_t i = 0; i < face_num_; i++)
    {
        faces_[i].id = i;
        for (size_t j = 0; j < 3; j++)
        {
            faces_[i].vert[j] = nodes_.get(faces[i].vert[j]->id);   
        }
    }

    elems_.resize(elem_num_);
    for (size_t i = 0; i < elem_num_; i++)
    {
        elems_[i].id = i;
        for (size_t j = 0; j < 4; j++)
        {
            elems_[i].vert[j] = nodes_.get(elems[i].vert[j]->id);   
        }
    }

    in_ready_ = true;
    
    // Copy physics groups
    phys_ = phys;
    phys_ready_ = true;

    // Copy element tags
    etags_ = etags;
    elem_tags_.resize(elem_num_, 0); // only physical tags.
    for (size_t i = 0; i < elem_num_; i++)
    {
        elem_tags_[i] = etags[i][0];
    }

    node_etags_.resize(node_num_, 0); // element associated physical tag for nodes.
    for (size_t i = 0; i < elem_num_; i++)
    {
        for (size_t j = 0; j < 4; j++)
        {
            node_etags_[elems_[i].vert[j]->id] = elem_tags_[i];
        }
    }

    elem_tag_ready_ = true;

    // Copy face tags
    ftags_ = ftags;
    face_tags_.resize(face_num_, 0); // only physical tags.
    for (size_t i = 0; i < face_num_; i++)
    {
        face_tags_[i] = ftags[i][0];
    }

    node_ftags_.resize(node_num_, 0); // face associated physical tag for nodes.
    for (size_t i = 0; i < face_num_; i++)
    {
        for (size_t j = 0; j < 3; j++)
        {
            node_ftags_[faces_[i].vert[j]->id] = face_tags_[i];
        }
    }

    face_ready_ = true;
    face_tag_ready_ = true;
    return;
}