/********************************************************
 *     _____________________
 *    / ____/  _/ ___/_  __/
 *   / / __ / / \__ \ / /   
 *  / /_/ // / ___/ // /    
 *  \____/___//____//_/ 
 * Geophysical Inversions using Spherical Tetrahedral meshes (GIST)
 *
 * Copyright (c) 2022  Yi Zhang (yizhang-geo@zju.edu.cn)
 *
 * GIST is distributed under a dual licensing scheme. You can redistribute 
 * it and/or modify it under the terms of the GNU Affero General Public 
 * License (AGPL) as published by the Free Software Foundation, either version 
 * 3 of the License, or (at your option) any later version. You should have 
 * received a copy of the GNU Affero General Public License along with this 
 * program. If not, see <http://www.gnu.org/licenses/>.
 * 
 * If the terms and conditions of the AGPL v.3. would prevent you from using 
 * the GIST, please consider the option to obtain a commercial license for a 
 * fee. These licenses are offered by the original author, Yi Zhang. As a rule, 
 * licenses are provided "as-is", unlimited in time for a one time fee. Please 
 * send corresponding requests to: yizhang-geo@zju.edu.cn. Please do not forget 
 * to include some description of your company and the realm of its activities. 
 * Also add information on how to contact you by electronic and paper mail.
 ******************************************************/

#include "model_space.h"

void GIST::ModelSpace::InitMatStructuralSimilarity(const array<double> &ref_model)
{
    // Differential matrix at three directions
    gctl::spmat<double> d1_mat(elem_num_, elem_num_, 0.0);
    gctl::spmat<double> d2_mat(elem_num_, elem_num_, 0.0);
    gctl::spmat<double> d3_mat(elem_num_, elem_num_, 0.0);
    gctl::spmat<double> d4_mat(elem_num_, elem_num_, 0.0);

    for (int i = 0; i < elem_num_; i++)
    {
        if (elems_[i].neigh[0] != nullptr)
        {
            d1_mat.insert(i, i, -1.0);
            d1_mat.insert(i, elems_[i].neigh[0]->id, 1.0);
        }

        if (elems_[i].neigh[1] != nullptr)
        {
            d2_mat.insert(i, i, -1.0);
            d2_mat.insert(i, elems_[i].neigh[1]->id, 1.0);
        }

        if (elems_[i].neigh[2] != nullptr)
        {
            d3_mat.insert(i, i, -1.0);
            d3_mat.insert(i, elems_[i].neigh[2]->id, 1.0);
        }

        if (elems_[i].neigh[3] != nullptr)
        {
            d4_mat.insert(i, i, -1.0);
            d4_mat.insert(i, elems_[i].neigh[3]->id, 1.0);
        }
    }

    gctl::array<double> f1_d(elem_num_);
    gctl::array<double> f2_d(elem_num_);
    gctl::array<double> f3_d(elem_num_);
    gctl::array<double> f4_d(elem_num_);

    d1_mat.multiply_vector(ref_model, f1_d);
    d2_mat.multiply_vector(ref_model, f2_d);
    d3_mat.multiply_vector(ref_model, f3_d);
    d4_mat.multiply_vector(ref_model, f4_d);

    ref1_mat_.malloc(elem_num_, elem_num_, 0.0);
    ref2_mat_.malloc(elem_num_, elem_num_, 0.0);
    ref3_mat_.malloc(elem_num_, elem_num_, 0.0);
    ref4_mat_.malloc(elem_num_, elem_num_, 0.0);
    ref5_mat_.malloc(elem_num_, elem_num_, 0.0);
    ref6_mat_.malloc(elem_num_, elem_num_, 0.0);
    ref_diff_.resize(elem_num_);
    grad_dm_.resize(elem_num_);

    gctl::spmat<double> tmp_mat(elem_num_, elem_num_, 0.0);

    // 1->2
    d1_mat.multiply_diagonal_matrix(f2_d, tmp_mat, true);
    ref1_mat_.add_sparse_matrix(tmp_mat);
    d2_mat.multiply_diagonal_matrix(f1_d, tmp_mat, true);
    ref1_mat_.minus_sparse_matrix(tmp_mat);

    // 1->3
    d1_mat.multiply_diagonal_matrix(f3_d, tmp_mat, true);
    ref2_mat_.add_sparse_matrix(tmp_mat);
    d3_mat.multiply_diagonal_matrix(f1_d, tmp_mat, true);
    ref2_mat_.minus_sparse_matrix(tmp_mat);

    // 1->4
    d1_mat.multiply_diagonal_matrix(f4_d, tmp_mat, true);
    ref3_mat_.add_sparse_matrix(tmp_mat);
    d4_mat.multiply_diagonal_matrix(f1_d, tmp_mat, true);
    ref3_mat_.minus_sparse_matrix(tmp_mat);

    // 2->3
    d2_mat.multiply_diagonal_matrix(f3_d, tmp_mat, true);
    ref4_mat_.add_sparse_matrix(tmp_mat);
    d3_mat.multiply_diagonal_matrix(f2_d, tmp_mat, true);
    ref4_mat_.minus_sparse_matrix(tmp_mat);

    // 2->4
    d2_mat.multiply_diagonal_matrix(f4_d, tmp_mat, true);
    ref5_mat_.add_sparse_matrix(tmp_mat);
    d4_mat.multiply_diagonal_matrix(f2_d, tmp_mat, true);
    ref5_mat_.minus_sparse_matrix(tmp_mat);

    // 3->4
    d3_mat.multiply_diagonal_matrix(f4_d, tmp_mat, true);
    ref6_mat_.add_sparse_matrix(tmp_mat);
    d4_mat.multiply_diagonal_matrix(f3_d, tmp_mat, true);
    ref6_mat_.minus_sparse_matrix(tmp_mat);

    tmp_mat.clear();
    d1_mat.clear();
    d2_mat.clear();
    d3_mat.clear();
    d4_mat.clear();
    f1_d.clear();
    f2_d.clear();
    f3_d.clear();
    f4_d.clear();
}