/********************************************************
 *     _____________________
 *    / ____/  _/ ___/_  __/
 *   / / __ / / \__ \ / /   
 *  / /_/ // / ___/ // /    
 *  \____/___//____//_/ 
 * Geophysical Inversions using Spherical Tetrahedral meshes (GIST)
 *
 * Copyright (c) 2022  Yi Zhang (yizhang-geo@zju.edu.cn)
 *
 * GIST is distributed under a dual licensing scheme. You can redistribute 
 * it and/or modify it under the terms of the GNU Affero General Public 
 * License (AGPL) as published by the Free Software Foundation, either version 
 * 3 of the License, or (at your option) any later version. You should have 
 * received a copy of the GNU Affero General Public License along with this 
 * program. If not, see <http://www.gnu.org/licenses/>.
 * 
 * If the terms and conditions of the AGPL v.3. would prevent you from using 
 * the GIST, please consider the option to obtain a commercial license for a 
 * fee. These licenses are offered by the original author, Yi Zhang. As a rule, 
 * licenses are provided "as-is", unlimited in time for a one time fee. Please 
 * send corresponding requests to: yizhang-geo@zju.edu.cn. Please do not forget 
 * to include some description of your company and the realm of its activities. 
 * Also add information on how to contact you by electronic and paper mail.
 ******************************************************/

#include "model_space.h"

void GIST::ModelSpace::InitMatRoughness(double smth_x, double smth_y, double smth_z, double smth_w, array<point3dc> *smth_wgts)
{
    face_diff_.resize(cmn_face_num_);
    grad_dm_.resize(elem_num_);
    smooth_mat_.malloc(cmn_face_num_, elem_num_, 0.0);
    //smth_mdl_.resize(elem_num_);
    //smth_face_diff_.resize(cmn_face_num_);

    // Initate smooth weight
    gctl::point3dc smooth_wgt;
    if (smth_wgts == nullptr)
    {
        gctl::point3dc smooth_wgt(smth_x, smth_y, smth_z);
        smooth_wgt.set2module(smth_w);
    }

    double maxi_ratio = 0.0;
    double edge_len[3], p, cmn_area, dist, mean_wgt;
    gctl::point3dc cen_p1, cen_p2;
    gctl::point3dc f_l1, f_l2, nf;
    
    for (size_t i = 0; i < cmn_face_num_; i++)
    {
        edge_len[0] = gctl::distance(*cmn_faces_[i].vert_ptr[0], *cmn_faces_[i].vert_ptr[1]);
        edge_len[1] = gctl::distance(*cmn_faces_[i].vert_ptr[1], *cmn_faces_[i].vert_ptr[2]);
        edge_len[2] = gctl::distance(*cmn_faces_[i].vert_ptr[2], *cmn_faces_[i].vert_ptr[0]);
        double p = 0.5*(edge_len[0] + edge_len[1] + edge_len[2]);
        cmn_area = sqrt(p*(p - edge_len[0])*(p - edge_len[1])*(p - edge_len[2]));

        // 计算相邻四面体的中心位置
        cen_p1 = cmn_faces_[i].tet_ptr[0]->center();
        cen_p2 = cmn_faces_[i].tet_ptr[1]->center();
        dist = gctl::distance(cen_p1, cen_p2);

        maxi_ratio = std::max(maxi_ratio, sqrt(cmn_area/dist));
    }
    
    for (size_t i = 0; i < cmn_face_num_; i++)
    {
        edge_len[0] = gctl::distance(*cmn_faces_[i].vert_ptr[0], *cmn_faces_[i].vert_ptr[1]);
        edge_len[1] = gctl::distance(*cmn_faces_[i].vert_ptr[1], *cmn_faces_[i].vert_ptr[2]);
        edge_len[2] = gctl::distance(*cmn_faces_[i].vert_ptr[2], *cmn_faces_[i].vert_ptr[0]);
        double p = 0.5*(edge_len[0] + edge_len[1] + edge_len[2]);
        cmn_area = sqrt(p*(p - edge_len[0])*(p - edge_len[1])*(p - edge_len[2]));

        // 计算相邻四面体的中心位置
        cen_p1 = cmn_faces_[i].tet_ptr[0]->center();
        cen_p2 = cmn_faces_[i].tet_ptr[1]->center();
        dist = gctl::distance(cen_p1, cen_p2);

        // 计算公共面的光滑度权重 权重矢量与边的单位法向量的点积
        // 法向量与权重矢量方向一致或相反时加权系数最大，两者正交时加权系数最小（为0,可考虑是否需要设置最小加权阈值）
        f_l1 = *cmn_faces_[i].vert_ptr[1] - *cmn_faces_[i].vert_ptr[0];
        f_l2 = *cmn_faces_[i].vert_ptr[2] - *cmn_faces_[i].vert_ptr[0];
        nf = gctl::cross(f_l1, f_l2).normal();

        if (smth_wgts != nullptr) smooth_wgt = smth_wgts->at(i);

        if (smooth_wgt.module() > 1e-10) mean_wgt = fabs(gctl::dot(nf, smooth_wgt));
        else mean_wgt = 1.0;

        // 将 sqrt(cmn_area/dist)/maxi_ratio 的比例控制在0到1内
        smooth_mat_.insert(i, cmn_faces_[i].tet_ptr[0]->id, -1.0*mean_wgt*sqrt(cmn_area/dist)/maxi_ratio);
        smooth_mat_.insert(i, cmn_faces_[i].tet_ptr[1]->id, mean_wgt*sqrt(cmn_area/dist)/maxi_ratio);
    }
    return;
}